/**
 * The Odds API Client
 *
 * Fetches real sportsbook odds and rotation numbers from api.the-odds-api.com/v4.
 * Used to enrich Kalshi game data with external market context.
 *
 * Features:
 * - In-memory cache with 15-minute TTL
 * - Graceful degradation on missing API key or network errors
 * - Fuzzy team name matching between abbreviations and full names
 * - Rotation number extraction
 */

import {
  NBA_TEAM_NAMES,
  NHL_TEAM_NAMES,
  NFL_TEAM_NAMES,
  MLB_TEAM_NAMES,
} from '@/lib/nbaConsolidated/teamNames';

// ---------------------------------------------------------------------------
// Types
// ---------------------------------------------------------------------------

export interface OddsApiOutcome {
  name: string;
  price: number; // American odds
  point?: number;
}

export interface OddsApiMarket {
  key: string; // "h2h", "spreads", "totals"
  outcomes: OddsApiOutcome[];
}

export interface OddsApiBookmaker {
  key: string; // "draftkings", "fanduel", etc.
  title: string;
  markets: OddsApiMarket[];
}

export interface OddsApiEvent {
  id: string;
  sport_key: string;
  sport_title: string;
  commence_time: string; // ISO 8601
  home_team: string;
  away_team: string;
  home_rotation: number | null;
  away_rotation: number | null;
  bookmakers: OddsApiBookmaker[];
}

// ---------------------------------------------------------------------------
// Sport Key Mapping
// ---------------------------------------------------------------------------

/** Maps dashboard sport codes to The Odds API sport_key values. */
export const SPORT_KEY_MAP: Record<string, string> = {
  nba: 'basketball_nba',
  cbb: 'basketball_ncaab',
  nfl: 'americanfootball_nfl',
  nhl: 'icehockey_nhl',
  mlb: 'baseball_mlb',
};

// ---------------------------------------------------------------------------
// Team Name Lookup (abbreviation <-> full name)
// ---------------------------------------------------------------------------

/**
 * Build a bidirectional lookup: given any of abbreviation, full name, city,
 * or mascot — return a normalized key (the full name in lowercase).
 * This powers fuzzy matching between The Odds API team names and our codes.
 */
function buildTeamLookup(mapping: Record<string, string>): Map<string, string> {
  const lookup = new Map<string, string>();

  for (const [abbr, fullName] of Object.entries(mapping)) {
    const normalized = fullName.toLowerCase();

    // Map abbreviation -> normalized full name
    lookup.set(abbr.toLowerCase(), normalized);

    // Map full name -> normalized full name
    lookup.set(normalized, normalized);

    // Map each word in the full name individually (city, mascot)
    // e.g., "Boston Celtics" -> "boston" and "celtics" both map to "boston celtics"
    const words = fullName.toLowerCase().split(/\s+/);
    for (const word of words) {
      // Only map single words if they're reasonably unique (4+ chars)
      // Avoids collisions on "New", "Los", etc.
      if (word.length >= 4) {
        // Don't overwrite if already mapped (first mapping wins)
        if (!lookup.has(word)) {
          lookup.set(word, normalized);
        }
      }
    }
  }

  return lookup;
}

const TEAM_LOOKUPS: Record<string, Map<string, string>> = {
  basketball_nba: buildTeamLookup(NBA_TEAM_NAMES),
  icehockey_nhl: buildTeamLookup(NHL_TEAM_NAMES),
  americanfootball_nfl: buildTeamLookup(NFL_TEAM_NAMES),
  baseball_mlb: buildTeamLookup(MLB_TEAM_NAMES),
  // CBB has no static mapping — uses direct string matching
};

/**
 * Normalize a team identifier to a canonical string for comparison.
 * Tries abbreviation lookup, full name lookup, and individual word lookup.
 * Falls back to the lowercased input if no mapping found.
 */
function normalizeTeamName(name: string, sportKey: string): string {
  const lower = name.toLowerCase().trim();
  const lookup = TEAM_LOOKUPS[sportKey];

  if (lookup) {
    // Direct lookup (abbreviation or full name)
    const direct = lookup.get(lower);
    if (direct) return direct;

    // Try matching the last word (mascot name, e.g., "Celtics" from "Boston Celtics")
    const words = lower.split(/\s+/);
    const lastWord = words[words.length - 1];
    if (lastWord && lastWord.length >= 4) {
      const byLast = lookup.get(lastWord);
      if (byLast) return byLast;
    }

    // Try matching the first word (city name)
    const firstWord = words[0];
    if (firstWord && firstWord.length >= 4) {
      const byFirst = lookup.get(firstWord);
      if (byFirst) return byFirst;
    }
  }

  // For CBB or unrecognized: just return lowercased
  return lower;
}

/**
 * Check if two team identifiers refer to the same team.
 */
function teamsMatch(a: string, b: string, sportKey: string): boolean {
  const normA = normalizeTeamName(a, sportKey);
  const normB = normalizeTeamName(b, sportKey);

  if (normA === normB) return true;

  // Substring containment as fallback (handles "NC State" vs "North Carolina State" etc.)
  if (normA.length > 3 && normB.length > 3) {
    if (normA.includes(normB) || normB.includes(normA)) return true;
  }

  return false;
}

// ---------------------------------------------------------------------------
// Cache
// ---------------------------------------------------------------------------

interface CacheEntry {
  data: OddsApiEvent[];
  fetchedAtMs: number;
}

const CACHE_TTL_MS = 15 * 60 * 1000; // 15 minutes
const cache = new Map<string, CacheEntry>();

function getCached(sportKey: string): OddsApiEvent[] | null {
  const entry = cache.get(sportKey);
  if (!entry) return null;

  const age = Date.now() - entry.fetchedAtMs;
  if (age > CACHE_TTL_MS) {
    cache.delete(sportKey);
    return null;
  }

  return entry.data;
}

function setCache(sportKey: string, data: OddsApiEvent[]): void {
  cache.set(sportKey, { data, fetchedAtMs: Date.now() });
}

// ---------------------------------------------------------------------------
// API Client
// ---------------------------------------------------------------------------

const BASE_URL = 'https://api.the-odds-api.com/v4';

const ODDS_API_KEY_STORAGE_KEY = 'galactus_odds_api_key';

function getApiKey(): string | null {
  // localStorage takes priority (user-configured in Settings)
  const stored = localStorage.getItem(ODDS_API_KEY_STORAGE_KEY)?.trim();
  if (stored && stored.length > 0) return stored;

  // Fall back to build-time env var
  const envKey = (import.meta.env.VITE_ODDS_API_KEY as string | undefined)?.trim();
  if (!envKey || envKey.length === 0) return null;
  return envKey;
}

/** Read the current Odds API key (for prefilling settings UI). */
export function getOddsApiKey(): string {
  return (
    localStorage.getItem(ODDS_API_KEY_STORAGE_KEY)?.trim() ||
    (import.meta.env.VITE_ODDS_API_KEY as string | undefined)?.trim() ||
    ''
  );
}

/** Save Odds API key to localStorage. */
export function setOddsApiKey(key: string): void {
  const trimmed = key.trim();
  if (trimmed) {
    localStorage.setItem(ODDS_API_KEY_STORAGE_KEY, trimmed);
  } else {
    localStorage.removeItem(ODDS_API_KEY_STORAGE_KEY);
  }
  // Clear cache so next fetch uses the new key
  cache.clear();
}

/**
 * Fetch odds for a sport from The Odds API, with caching.
 *
 * Returns cached data if available and within TTL.
 * Returns empty array if API key is missing or on network errors.
 */
export async function fetchOddsForSport(sportKey: string): Promise<OddsApiEvent[]> {
  // Check cache first
  const cached = getCached(sportKey);
  if (cached) return cached;

  const apiKey = getApiKey();
  if (!apiKey) {
    console.warn('[oddsApi] No VITE_ODDS_API_KEY configured — skipping odds fetch');
    return [];
  }

  // Use commenceTimeFrom/To to fetch all events in a wide window.
  // Start from beginning of today so in-progress games aren't excluded.
  const todayStart = new Date();
  todayStart.setHours(0, 0, 0, 0);
  const weekOut = new Date(todayStart.getTime() + 7 * 24 * 60 * 60 * 1000);

  const params = new URLSearchParams({
    apiKey,
    regions: 'us',
    markets: 'h2h',
    oddsFormat: 'american',
    includeRotationNumbers: 'true',
    commenceTimeFrom: todayStart.toISOString(),
    commenceTimeTo: weekOut.toISOString(),
  });

  const url = `${BASE_URL}/sports/${encodeURIComponent(sportKey)}/odds?${params.toString()}`;

  try {
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), 15000); // 15s timeout

    const response = await fetch(url, { signal: controller.signal });
    clearTimeout(timeoutId);

    if (!response.ok) {
      const body = await response.text().catch(() => '');
      console.warn(`[oddsApi] API error ${response.status} for ${sportKey}: ${body.slice(0, 200)}`);
      // Return stale cache if available
      const stale = cache.get(sportKey);
      if (stale) return stale.data;
      return [];
    }

    const data = (await response.json()) as OddsApiEvent[];
    setCache(sportKey, data);
    return data;
  } catch (err) {
    if (err instanceof Error && err.name === 'AbortError') {
      console.warn(`[oddsApi] Request timeout for ${sportKey}`);
    } else {
      console.warn(`[oddsApi] Fetch error for ${sportKey}:`, err);
    }
    // Return stale cache if available
    const stale = cache.get(sportKey);
    if (stale) return stale.data;
    return [];
  }
}

/**
 * Fetch events for a sport (free endpoint, 0 credits).
 * Useful for getting event IDs and commence times without consuming quota.
 */
export async function fetchEventsForSport(sportKey: string): Promise<OddsApiEvent[]> {
  const apiKey = getApiKey();
  if (!apiKey) {
    console.warn('[oddsApi] No VITE_ODDS_API_KEY configured — skipping events fetch');
    return [];
  }

  // Use commenceTimeFrom/To to fetch all events in a wide window.
  // The API has no documented `limit` param — without time filtering it returns ~10 events.
  // Start from beginning of today so in-progress games aren't excluded.
  const todayStart = new Date();
  todayStart.setHours(0, 0, 0, 0);
  const weekOut = new Date(todayStart.getTime() + 7 * 24 * 60 * 60 * 1000);

  const params = new URLSearchParams({
    apiKey,
    includeRotationNumbers: 'true',
    commenceTimeFrom: todayStart.toISOString(),
    commenceTimeTo: weekOut.toISOString(),
  });
  const url = `${BASE_URL}/sports/${encodeURIComponent(sportKey)}/events?${params.toString()}`;

  try {
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), 15000);

    const response = await fetch(url, { signal: controller.signal });
    clearTimeout(timeoutId);

    if (!response.ok) {
      console.warn(`[oddsApi] Events API error ${response.status} for ${sportKey}`);
      return [];
    }

    return (await response.json()) as OddsApiEvent[];
  } catch (err) {
    console.warn(`[oddsApi] Events fetch error for ${sportKey}:`, err);
    return [];
  }
}

// ---------------------------------------------------------------------------
// Game Matching
// ---------------------------------------------------------------------------

/** Time window for matching commence_time (ms) */
const MATCH_WINDOW_MS = 30 * 60 * 1000; // 30 minutes

/**
 * Match an Odds API event to a Kalshi game by date + team names.
 *
 * @param oddsEvents - Array of events from The Odds API
 * @param gameDate - Game date in YYYY-MM-DD format
 * @param homeTeam - Home team code or name from Kalshi
 * @param awayTeam - Away team code or name from Kalshi
 * @returns The best matching event, or null if no match found
 */
export function matchOddsToGame(
  oddsEvents: OddsApiEvent[],
  gameDate: string,
  homeTeam: string,
  awayTeam: string
): OddsApiEvent | null {
  if (!oddsEvents.length || !homeTeam || !awayTeam) return null;

  // Parse the game date into a range for the entire day
  const dayStart = new Date(`${gameDate}T00:00:00`).getTime();
  const dayEnd = dayStart + 24 * 60 * 60 * 1000;

  // Widen the window by MATCH_WINDOW_MS on each side to handle timezone edge cases
  const windowStart = dayStart - MATCH_WINDOW_MS;
  const windowEnd = dayEnd + MATCH_WINDOW_MS;

  let bestMatch: OddsApiEvent | null = null;
  let bestScore = 0;

  for (const event of oddsEvents) {
    const commenceMs = new Date(event.commence_time).getTime();
    if (isNaN(commenceMs)) continue;

    // Time filter
    if (commenceMs < windowStart || commenceMs > windowEnd) continue;

    // Determine sport key for team matching
    const sportKey = event.sport_key;

    // Check team matching (both directions since Odds API may swap home/away)
    const homeMatches =
      teamsMatch(homeTeam, event.home_team, sportKey) &&
      teamsMatch(awayTeam, event.away_team, sportKey);

    const swappedMatches =
      teamsMatch(homeTeam, event.away_team, sportKey) &&
      teamsMatch(awayTeam, event.home_team, sportKey);

    if (!homeMatches && !swappedMatches) continue;

    // Score: prefer exact day match over edge-of-day match
    let score = 1;
    if (commenceMs >= dayStart && commenceMs < dayEnd) {
      score = 2; // Within the exact day
    }
    // Prefer non-swapped match
    if (homeMatches) {
      score += 1;
    }

    if (score > bestScore) {
      bestScore = score;
      bestMatch = event;
    }
  }

  return bestMatch;
}

// ---------------------------------------------------------------------------
// Rotation Numbers
// ---------------------------------------------------------------------------

/**
 * Extract rotation numbers from an Odds API event.
 */
export function getRotationNumbers(event: OddsApiEvent): {
  home: number | null;
  away: number | null;
} {
  return {
    home: event.home_rotation ?? null,
    away: event.away_rotation ?? null,
  };
}

// ---------------------------------------------------------------------------
// Bookmaker Odds Extraction
// ---------------------------------------------------------------------------

/** Default bookmaker preference order (highest-quality lines first) */
const BOOKMAKER_PREFERENCE = [
  'draftkings',
  'fanduel',
  'betmgm',
  'caesars',
  'pointsbet',
  'betrivers',
  'unibet',
  'bovada',
  'betonlineag',
];

/**
 * Extract h2h (moneyline) odds from an Odds API event.
 *
 * @param event - The Odds API event
 * @param bookmakerKey - Specific bookmaker to use, or undefined for best available
 * @returns Home and away American odds, or null if not available
 */
export function getBookmakerOdds(
  event: OddsApiEvent,
  bookmakerKey?: string
): { home: number; away: number } | null {
  if (!event.bookmakers.length) return null;

  let bookmaker: OddsApiBookmaker | undefined;

  if (bookmakerKey) {
    bookmaker = event.bookmakers.find((b) => b.key === bookmakerKey);
  } else {
    // Find best available from preference list
    for (const preferred of BOOKMAKER_PREFERENCE) {
      bookmaker = event.bookmakers.find((b) => b.key === preferred);
      if (bookmaker) break;
    }
    // Fallback to first available bookmaker
    if (!bookmaker) {
      bookmaker = event.bookmakers[0];
    }
  }

  if (!bookmaker) return null;

  // Find h2h market
  const h2hMarket = bookmaker.markets.find((m) => m.key === 'h2h');
  if (!h2hMarket || h2hMarket.outcomes.length < 2) return null;

  // Match outcomes to home/away by team name
  const homeOutcome = h2hMarket.outcomes.find(
    (o) => o.name.toLowerCase() === event.home_team.toLowerCase()
  );
  const awayOutcome = h2hMarket.outcomes.find(
    (o) => o.name.toLowerCase() === event.away_team.toLowerCase()
  );

  if (!homeOutcome || !awayOutcome) {
    // Fallback: assume first outcome is home, second is away
    // (Odds API typically returns home first for h2h)
    const first = h2hMarket.outcomes[0];
    const second = h2hMarket.outcomes[1];
    if (first && second) {
      return { home: first.price, away: second.price };
    }
    return null;
  }

  return { home: homeOutcome.price, away: awayOutcome.price };
}
