/**
 * Odds Format Conversion
 *
 * Bidirectional conversion between probability and various odds formats
 */

export type OddsFormat = 'probability' | 'american' | 'decimal' | 'fractional';

const STORAGE_KEY = 'odds_format';
let currentFormat: OddsFormat = 'probability';

/**
 * Initialize format from localStorage
 */
export function initOddsFormat(): OddsFormat {
  const saved = localStorage.getItem(STORAGE_KEY);
  if (
    saved === 'probability' ||
    saved === 'american' ||
    saved === 'decimal' ||
    saved === 'fractional'
  ) {
    currentFormat = saved;
  }
  return currentFormat;
}

/**
 * Get current format
 */
export function getOddsFormat(): OddsFormat {
  return currentFormat;
}

/**
 * Set format and save to localStorage
 */
export function setOddsFormat(format: OddsFormat): void {
  currentFormat = format;
  localStorage.setItem(STORAGE_KEY, format);
}

/**
 * Cycle through formats: probability → american → decimal → fractional → probability
 */
export function cycleOddsFormat(): OddsFormat {
  const formats: OddsFormat[] = ['probability', 'american', 'decimal', 'fractional'];
  const currentIndex = formats.indexOf(currentFormat);
  const nextIndex = (currentIndex + 1) % formats.length;
  const newFormat = formats[nextIndex];
  setOddsFormat(newFormat);
  return newFormat;
}

/**
 * Convert probability to American odds
 * @param prob Probability (0-1)
 * @returns American odds (e.g., -150, +200)
 */
export function probabilityToAmerican(prob: number): number {
  if (prob >= 0.5) {
    // Favorite: negative odds
    return Math.round((-100 * prob) / (1 - prob));
  } else {
    // Underdog: positive odds
    return Math.round((100 * (1 - prob)) / prob);
  }
}

/**
 * Convert American odds to probability
 * @param odds American odds (e.g., -150, +200)
 * @returns Probability (0-1)
 */
export function americanToProbability(odds: number): number {
  if (odds > 0) {
    // Positive odds (underdog)
    return 100 / (odds + 100);
  } else {
    // Negative odds (favorite)
    return Math.abs(odds) / (Math.abs(odds) + 100);
  }
}

/**
 * Convert probability to decimal odds
 * @param prob Probability (0-1)
 * @returns Decimal odds (e.g., 2.5)
 */
export function probabilityToDecimal(prob: number): number {
  return 1 / prob;
}

/**
 * Convert decimal odds to probability
 * @param odds Decimal odds (e.g., 2.5)
 * @returns Probability (0-1)
 */
export function decimalToProbability(odds: number): number {
  return 1 / odds;
}

/**
 * Convert probability to fractional odds
 * @param prob Probability (0-1)
 * @returns Fractional odds as string (e.g., "3/2")
 */
export function probabilityToFractional(prob: number): string {
  const decimal = probabilityToDecimal(prob);
  const fractional = decimal - 1; // e.g., 2.5 -> 1.5 -> 3/2

  // Find simplest fraction
  const tolerance = 1.0e-6;
  let h1 = 1,
    h2 = 0,
    k1 = 0,
    k2 = 1;
  let b = fractional;

  do {
    const a = Math.floor(b);
    const aux = h1;
    h1 = a * h1 + h2;
    h2 = aux;
    const aux2 = k1;
    k1 = a * k1 + k2;
    k2 = aux2;
    b = 1 / (b - a);
  } while (Math.abs(fractional - h1 / k1) > fractional * tolerance);

  return `${h1}/${k1}`;
}

/**
 * Convert fractional odds to probability
 * @param fractional Fractional odds as string (e.g., "3/2")
 * @returns Probability (0-1)
 */
export function fractionalToProbability(fractional: string): number {
  const [num, den] = fractional.split('/').map(Number);
  if (!num || !den || den === 0) {
    throw new Error('Invalid fractional odds');
  }
  const decimal = num / den + 1;
  return decimalToProbability(decimal);
}

/**
 * Format probability/odds for display
 * @param prob Probability (0-1) or null
 * @returns Formatted string
 */
export function formatOdds(prob: number | null): string {
  if (prob === null || prob === undefined) {
    return '—';
  }

  switch (currentFormat) {
    case 'probability':
      return prob.toFixed(4);

    case 'american': {
      const odds = probabilityToAmerican(prob);
      return odds > 0 ? `+${odds}` : `${odds}`;
    }

    case 'decimal': {
      const odds = probabilityToDecimal(prob);
      return odds.toFixed(2);
    }

    case 'fractional': {
      try {
        return probabilityToFractional(prob);
      } catch {
        return prob.toFixed(4);
      }
    }

    default:
      return prob.toFixed(4);
  }
}

/**
 * Get format label for display
 */
export function getFormatLabel(): string {
  switch (currentFormat) {
    case 'probability':
      return 'PROB';
    case 'american':
      return 'AMERICAN';
    case 'decimal':
      return 'DECIMAL';
    case 'fractional':
      return 'FRACTIONAL';
    default:
      return 'PROB';
  }
}
