/**
 * Order Monitor
 *
 * Monitors order status changes and detects fills
 */

import type { KalshiApiClient, KalshiOrder, FillNotification, KalshiFill } from '../types';
import { normalizeKalshiPriceToProbability } from './price';

export interface OrderMonitorCallbacks {
  onFill: (fill: FillNotification) => void;
  onStatusChange: (orderId: string, oldStatus: string, newStatus: string) => void;
}

export class OrderMonitor {
  private apiClient: KalshiApiClient | null = null;
  private callbacks: OrderMonitorCallbacks | null = null;
  private previousOrders: Map<string, KalshiOrder> = new Map();
  private monitoringInterval: number | null = null;
  private isMonitoring = false;
  private stoppedDueToAuthError = false;
  private pollIntervalMs = 2000;

  /**
   * Start monitoring orders
   */
  start(
    apiClient: KalshiApiClient,
    callbacks: OrderMonitorCallbacks,
    options?: { pollIntervalMs?: number }
  ): void {
    this.apiClient = apiClient;
    this.callbacks = callbacks;
    this.isMonitoring = true;
    this.stoppedDueToAuthError = false;
    this.pollIntervalMs = options?.pollIntervalMs ?? 2000;

    // Initial fetch
    this.checkOrders();

    // Set up polling (will be replaced with WebSocket later)
    this.monitoringInterval = window.setInterval(() => {
      if (this.isMonitoring) {
        this.checkOrders();
      }
    }, this.pollIntervalMs);
  }

  /**
   * Stop monitoring
   */
  stop(): void {
    this.isMonitoring = false;
    if (this.monitoringInterval !== null) {
      clearInterval(this.monitoringInterval);
      this.monitoringInterval = null;
    }
    this.previousOrders.clear();
  }

  /**
   * Check for order status changes
   */
  private async checkOrders(): Promise<void> {
    if (!this.apiClient || !this.callbacks) return;
    if (this.stoppedDueToAuthError) return;

    try {
      const currentOrders = await this.apiClient.getOpenOrders();
      const currentOrdersMap = new Map<string, KalshiOrder>();

      // Build current orders map
      for (const order of currentOrders) {
        currentOrdersMap.set(order.order_id, order);
      }

      // Check for status changes and fills
      for (const [orderId, previousOrder] of this.previousOrders.entries()) {
        const currentOrder = currentOrdersMap.get(orderId);

        if (!currentOrder) {
          // Order disappeared - might be filled or cancelled
          if (previousOrder.status === 'resting' || previousOrder.status === 'pending') {
            // Check if it was filled by checking filled orders
            const fills = await this.apiClient.getFilledOrders({ limit: 10 });
            const fill = fills.find((f) => f.order_id === orderId);

            if (fill) {
              // Order was filled!
              this.handleKalshiFill(fill);
            } else {
              // Order was cancelled
              this.callbacks.onStatusChange(orderId, previousOrder.status, 'cancelled');
            }
          }
        } else {
          // Order still exists - check status change
          if (currentOrder.status !== previousOrder.status) {
            this.callbacks.onStatusChange(orderId, previousOrder.status, currentOrder.status);

            // If status changed to filled, handle fill
            if (currentOrder.status === 'filled') {
              const price =
                currentOrder.side === 'yes' ? currentOrder.yes_price : currentOrder.no_price;
              const fill: FillNotification = {
                ticker: currentOrder.ticker,
                side: currentOrder.side,
                action: currentOrder.action,
                quantity: currentOrder.remaining_count,
                price: price ?? 0,
                timestamp: Date.now(),
                orderId: currentOrder.order_id,
              };
              this.callbacks.onFill(fill);
            }
          }
        }
      }

      // Update previous orders
      this.previousOrders = currentOrdersMap;
    } catch (error) {
      const message = error instanceof Error ? error.message : String(error);

      // If portfolio endpoints aren't available (wrong environment, key not provisioned, etc),
      // stop hammering the API every 2s. Keep market data running.
      if (
        message.includes('API error: 401') ||
        message.includes('"code":"authentication_error"') ||
        message.includes('"code": "authentication_error"')
      ) {
        this.stoppedDueToAuthError = true;
        this.stop();
        console.warn(
          'OrderMonitor disabled: portfolio endpoints returned 401 authentication_error. Market data can still run.'
        );
        return;
      }

      console.error('Error checking orders:', error);
    }
  }

  /**
   * Handle a fill
   */
  private handleKalshiFill(fill: KalshiFill): void {
    if (!this.callbacks) return;

    const fillNotification: FillNotification = {
      ticker: fill.ticker,
      side: fill.side,
      action: fill.action,
      quantity: fill.count,
      price: normalizeKalshiPriceToProbability(fill.price) ?? fill.price,
      timestamp: Date.now(),
      orderId: fill.order_id,
    };

    this.callbacks.onFill(fillNotification);
  }

  /**
   * Manually trigger a fill check (for immediate detection after order placement)
   */
  async checkImmediate(): Promise<void> {
    await this.checkOrders();
  }
}
