import { ClobClient, OrderType, Side } from '@polymarket/clob-client';
import { Wallet } from 'ethers';

export interface PolyApiCreds {
  key: string;
  secret: string;
  passphrase: string;
}

export interface PolyUserConfig {
  privateKey: string; // 0x...
  signatureType: 0 | 1 | 2;
  funder?: string;
  apiCreds?: PolyApiCreds;
}

const HOST = 'https://clob.polymarket.com';
const CHAIN_ID = 137;

export function normalizePrivateKey(pk: string): string {
  const s = String(pk || '').trim();
  if (!s) return '';
  return s.startsWith('0x') ? s : `0x${s}`;
}

export async function createOrDerivePolyApiCreds(privateKey: string): Promise<PolyApiCreds> {
  const signer = new Wallet(normalizePrivateKey(privateKey));
  const temp = new ClobClient(HOST, CHAIN_ID, signer);
  const creds = (await temp.createOrDeriveApiKey()) as unknown as Record<string, unknown>;
  return {
    key: String(creds.key ?? creds.apiKey ?? ''),
    secret: String(creds.secret ?? ''),
    passphrase: String(creds.passphrase ?? ''),
  };
}

export function createPolyTradingClient(config: PolyUserConfig): ClobClient {
  const signer = new Wallet(normalizePrivateKey(config.privateKey));
  if (!config.apiCreds) {
    throw new Error('Missing Polymarket API credentials (derive first).');
  }
  // ClobClient SDK has loose constructor types - use unknown intermediary
  return new ClobClient(
    HOST,
    CHAIN_ID,
    signer,
    config.apiCreds as unknown as undefined,
    config.signatureType as unknown as undefined,
    config.funder as unknown as undefined
  );
}

/**
 * Place a Polymarket order.
 *
 * @param args.tokenId - The outcome token ID (specifies WHICH outcome you're trading)
 * @param args.action - 'buy' to purchase tokens, 'sell' to sell tokens you own
 * @param args.priceCents - Limit price in cents (1-99)
 * @param args.contracts - Number of contracts/tokens
 */
export async function placePolyOrder(args: {
  client: ClobClient;
  tokenId: string;
  action: 'buy' | 'sell';
  priceCents: number;
  contracts: number;
  postOnly?: boolean;
  tickSize?: string;
  negRisk?: boolean;
}): Promise<unknown> {
  const price = Math.max(0.0001, Math.min(0.9999, Number(args.priceCents) / 100));
  const size = Number(args.contracts);
  if (!Number.isFinite(size) || size <= 0) throw new Error('Invalid size');

  // action maps directly to Polymarket's Side enum
  const orderSide = args.action === 'buy' ? Side.BUY : Side.SELL;
  const marketParams = {
    tickSize: args.tickSize ?? '0.001',
    negRisk: Boolean(args.negRisk ?? false),
  };

  const userOrder = { tokenID: String(args.tokenId), price, side: orderSide, size };

  // ClobClient SDK methods are not fully typed
  const client = args.client as {
    createOrder: (o: unknown, p: unknown) => Promise<unknown>;
    postOrder: (s: unknown, t: OrderType, b: boolean) => Promise<unknown>;
    createAndPostOrder: (o: unknown, p: unknown, t: OrderType) => Promise<unknown>;
  };

  if (args.postOnly) {
    const signed = await client.createOrder(userOrder, marketParams);
    return await client.postOrder(signed, OrderType.GTC, true);
  }

  return await client.createAndPostOrder(userOrder, marketParams, OrderType.GTC);
}

/**
 * Cancel a single Polymarket order by ID.
 */
export async function cancelPolyOrder(client: ClobClient, orderId: string): Promise<unknown> {
  const c = client as unknown as {
    cancelOrder: (args: { orderID: string }) => Promise<unknown>;
  };
  return await c.cancelOrder({ orderID: orderId });
}

/**
 * Cancel all open Polymarket orders.
 */
export async function cancelAllPolyOrders(client: ClobClient): Promise<unknown> {
  const c = client as unknown as {
    cancelAll: () => Promise<unknown>;
  };
  return await c.cancelAll();
}

const STORAGE_KEY = 'polymarket_config_v1';

export function loadPolyUserConfig(): PolyUserConfig | null {
  try {
    const raw = localStorage.getItem(STORAGE_KEY);
    if (!raw) return null;
    const obj = JSON.parse(raw);
    if (!obj || typeof obj !== 'object') return null;

    // Migrate old apiKey -> key format
    if (obj.apiCreds && typeof obj.apiCreds === 'object') {
      const creds = obj.apiCreds as Record<string, unknown>;
      if (!creds.key && creds.apiKey) {
        creds.key = creds.apiKey;
        delete creds.apiKey;
        localStorage.setItem(STORAGE_KEY, JSON.stringify(obj));
      }
    }

    return obj as PolyUserConfig;
  } catch {
    return null;
  }
}

export function savePolyUserConfig(config: PolyUserConfig): void {
  localStorage.setItem(STORAGE_KEY, JSON.stringify(config));
}

export function clearPolyUserConfig(): void {
  localStorage.removeItem(STORAGE_KEY);
}
