import { safeJsonParse } from '../typeCoercers';

export interface PolyBookLevel {
  price: number; // 0..1
  size: number;
}

export interface PolyBookSnapshot {
  assetId: string;
  bids: PolyBookLevel[];
  asks: PolyBookLevel[];
  tsMs: number;
}

type BookCallback = (snap: PolyBookSnapshot) => void;
type ErrorCallback = (err: Error) => void;

function normalizeLevelArray(raw: unknown): PolyBookLevel[] {
  if (!Array.isArray(raw)) return [];
  const out: PolyBookLevel[] = [];
  for (const lvl of raw) {
    if (!lvl || typeof lvl !== 'object') continue;
    const lvlObj = lvl as Record<string, unknown>;
    const p = Number(lvlObj.price);
    const sz = Number(lvlObj.size);
    if (!Number.isFinite(p) || !Number.isFinite(sz)) continue;
    if (p <= 0 || p >= 1) continue;
    if (sz <= 0) continue;
    out.push({ price: p, size: sz });
  }
  return out;
}

export function createPolyMarketStream(): {
  connect: () => Promise<void>;
  subscribe: (assetIds: string[]) => void;
  disconnect: () => void;
  isConnected: () => boolean;
  onBook: (cb: BookCallback) => void;
  onError: (cb: ErrorCallback) => void;
} {
  const WS_URL = 'wss://ws-subscriptions-clob.polymarket.com/ws/market';
  let ws: WebSocket | null = null;
  let connected = false;
  let subscribed: string[] = [];
  const bookCallbacks = new Set<BookCallback>();
  const errorCallbacks = new Set<ErrorCallback>();
  let reconnectTimer: number | null = null;
  let backoffMs = 500;

  const emitError = (err: Error) => {
    errorCallbacks.forEach((cb) => {
      try {
        cb(err);
      } catch {
        // ignore
      }
    });
  };

  const clearReconnect = () => {
    if (reconnectTimer !== null) {
      window.clearTimeout(reconnectTimer);
      reconnectTimer = null;
    }
  };

  const scheduleReconnect = () => {
    if (reconnectTimer !== null) return;
    const wait = Math.min(backoffMs, 30_000);
    backoffMs = Math.min(backoffMs * 2, 30_000);
    reconnectTimer = window.setTimeout(() => {
      reconnectTimer = null;
      void connect().catch(() => {
        // error already emitted
      });
    }, wait);
  };

  const handleMessage = (ev: MessageEvent) => {
    const tsMs = Date.now();
    const data = typeof ev.data === 'string' ? safeJsonParse(ev.data) : null;
    if (!data) return;

    // Polymarket WS can send:
    // - list: [ {asset_id, bids, asks}, ... ]
    // - dict: { data: [ ... ] } or single {asset_id,...}
    const updates: Record<string, unknown>[] = [];
    if (Array.isArray(data)) {
      updates.push(...data.filter((x): x is Record<string, unknown> => x && typeof x === 'object'));
    } else if (typeof data === 'object') {
      const dataObj = data as Record<string, unknown>;
      if (Array.isArray(dataObj.data))
        updates.push(
          ...(dataObj.data as unknown[]).filter(
            (x): x is Record<string, unknown> => x != null && typeof x === 'object'
          )
        );
      else if (dataObj.asset_id && (dataObj.bids || dataObj.asks)) updates.push(dataObj);
    }

    for (const u of updates) {
      const assetId = String(u.asset_id ?? '');
      if (!assetId) continue;
      const bids = normalizeLevelArray(u.bids);
      const asks = normalizeLevelArray(u.asks);
      if (bids.length === 0 && asks.length === 0) continue;
      const snap: PolyBookSnapshot = { assetId, bids, asks, tsMs };
      bookCallbacks.forEach((cb) => cb(snap));
    }
  };

  const connect = async () => {
    clearReconnect();
    if (ws && (ws.readyState === WebSocket.OPEN || ws.readyState === WebSocket.CONNECTING)) return;

    ws = new WebSocket(WS_URL);
    connected = false;

    await new Promise<void>((resolve, reject) => {
      if (!ws) return reject(new Error('WS not created'));
      const timeoutId = window.setTimeout(() => {
        reject(new Error('Polymarket WS connection timed out (15s)'));
      }, 15_000);
      const onOpen = () => {
        window.clearTimeout(timeoutId);
        connected = true;
        backoffMs = 500;
        resolve();
      };
      const onErr = () => {
        window.clearTimeout(timeoutId);
        reject(new Error('Polymarket WS connection failed'));
      };
      ws.addEventListener('open', onOpen, { once: true });
      ws.addEventListener('error', onErr, { once: true });
    }).catch((err) => {
      emitError(err instanceof Error ? err : new Error(String(err)));
      scheduleReconnect();
      throw err;
    });

    if (!ws) return;
    ws.addEventListener('message', handleMessage);
    ws.addEventListener('close', () => {
      connected = false;
      scheduleReconnect();
    });
    ws.addEventListener('error', () => {
      connected = false;
      emitError(new Error('Polymarket WS error'));
      scheduleReconnect();
    });

    // Re-subscribe if we had ids already.
    if (subscribed.length > 0) {
      subscribe(subscribed);
    }
  };

  const subscribe = (assetIds: string[]) => {
    subscribed = Array.from(new Set(assetIds.filter(Boolean).map(String)));
    if (!ws || ws.readyState !== WebSocket.OPEN) return;
    // Mirror `nba/dash_server.py` subscribe payload.
    const msg = { assets_ids: subscribed, type: 'MARKET', custom_feature_enabled: false };
    try {
      ws.send(JSON.stringify(msg));
    } catch (err) {
      emitError(err instanceof Error ? err : new Error(String(err)));
    }
  };

  const disconnect = () => {
    clearReconnect();
    if (ws) {
      try {
        ws.close();
      } catch {
        // ignore
      }
    }
    ws = null;
    connected = false;
  };

  return {
    connect,
    subscribe,
    disconnect,
    isConnected: () => connected && ws?.readyState === WebSocket.OPEN,
    onBook: (cb) => bookCallbacks.add(cb),
    onError: (cb) => errorCallbacks.add(cb),
  };
}
