import type { PolyApiCreds } from './client';
import { safeJsonParse } from '../typeCoercers';

export interface PolyUserOrderEvent {
  event_type: 'order';
  type: 'PLACEMENT' | 'UPDATE' | 'CANCELLATION' | string;
  id: string;
  asset_id: string;
  market: string; // condition ID
  price: string;
  side: 'BUY' | 'SELL' | string;
  original_size?: string;
  size_matched?: string;
  timestamp?: string;
}

export interface PolyUserTradeEvent {
  event_type: 'trade';
  id: string;
  asset_id: string;
  market: string; // condition ID
  price: string;
  side: 'BUY' | 'SELL' | string;
  size: string;
  status: string;
  timestamp?: string;
}

type OrderCb = (e: PolyUserOrderEvent) => void;
type TradeCb = (e: PolyUserTradeEvent) => void;
type ErrCb = (e: Error) => void;

export function createPolyUserStream(creds: PolyApiCreds): {
  connect: (markets: string[]) => Promise<void>;
  subscribe: (markets: string[]) => void;
  disconnect: () => void;
  isConnected: () => boolean;
  onOrder: (cb: OrderCb) => void;
  onTrade: (cb: TradeCb) => void;
  onError: (cb: ErrCb) => void;
} {
  const WS_URL = 'wss://ws-subscriptions-clob.polymarket.com/ws/user';
  let ws: WebSocket | null = null;
  let subscribedMarkets: string[] = [];
  const orderCbs = new Set<OrderCb>();
  const tradeCbs = new Set<TradeCb>();
  const errCbs = new Set<ErrCb>();

  const emitErr = (e: Error) => errCbs.forEach((cb) => cb(e));

  const send = (msg: Record<string, unknown>) => {
    if (!ws || ws.readyState !== WebSocket.OPEN) return;
    ws.send(JSON.stringify(msg));
  };

  const subscribe = (markets: string[]) => {
    subscribedMarkets = Array.from(new Set(markets.filter(Boolean).map(String)));
    // After initial connect, we can subscribe/unsubscribe with operation messages (per docs).
    send({ markets: subscribedMarkets, operation: 'subscribe', custom_feature_enabled: false });
  };

  const connect = async (markets: string[]) => {
    subscribedMarkets = Array.from(new Set(markets.filter(Boolean).map(String)));
    if (ws && (ws.readyState === WebSocket.OPEN || ws.readyState === WebSocket.CONNECTING)) return;

    ws = new WebSocket(WS_URL);
    await new Promise<void>((resolve, reject) => {
      if (!ws) return reject(new Error('WS not created'));
      ws.addEventListener('open', () => resolve(), { once: true });
      ws.addEventListener(
        'error',
        () => reject(new Error('Polymarket user WS connection failed')),
        { once: true }
      );
    });

    ws.addEventListener('message', (ev) => {
      const parsed = typeof ev.data === 'string' ? safeJsonParse(ev.data) : null;
      if (!parsed || typeof parsed !== 'object') return;
      const msgObj = parsed as Record<string, unknown>;
      const eventType = String(msgObj.event_type || '');
      if (eventType === 'order')
        orderCbs.forEach((cb) => cb(msgObj as unknown as PolyUserOrderEvent));
      if (eventType === 'trade')
        tradeCbs.forEach((cb) => cb(msgObj as unknown as PolyUserTradeEvent));
    });
    ws.addEventListener('close', () => emitErr(new Error('Polymarket user WS closed')));
    ws.addEventListener('error', () => emitErr(new Error('Polymarket user WS error')));

    // Initial subscription message includes auth + markets + type
    send({
      auth: {
        apikey: creds.key,
        secret: creds.secret,
        passphrase: creds.passphrase,
      },
      markets: subscribedMarkets,
      type: 'USER',
      custom_feature_enabled: false,
    });
  };

  const disconnect = () => {
    if (ws) {
      try {
        ws.close();
      } catch {
        // ignore
      }
    }
    ws = null;
  };

  return {
    connect,
    subscribe,
    disconnect,
    isConnected: () => Boolean(ws && ws.readyState === WebSocket.OPEN),
    onOrder: (cb) => orderCbs.add(cb),
    onTrade: (cb) => tradeCbs.add(cb),
    onError: (cb) => errCbs.add(cb),
  };
}
