/**
 * Pricing POV (Point of View) Logic
 *
 * Maker POV: Show maker prices (bid for buy, ask for sell)
 * Taker POV: Show taker prices (ask for buy, bid for sell)
 *
 * Mirrors venue-specific dashboard logic
 */

export type PricingPOV = 'maker' | 'taker';

const STORAGE_KEY = 'pricing_pov';
let currentPOV: PricingPOV = 'maker';

/**
 * Initialize POV from localStorage
 */
export function initPOV(): PricingPOV {
  const saved = localStorage.getItem(STORAGE_KEY);
  if (saved === 'maker' || saved === 'taker') {
    currentPOV = saved;
  }
  return currentPOV;
}

/**
 * Get current POV
 */
export function getPOV(): PricingPOV {
  return currentPOV;
}

/**
 * Set POV and save to localStorage
 */
export function setPOV(pov: PricingPOV): void {
  currentPOV = pov;
  localStorage.setItem(STORAGE_KEY, pov);
}

/**
 * Toggle between Maker and Taker
 */
export function togglePOV(): PricingPOV {
  const newPOV = currentPOV === 'maker' ? 'taker' : 'maker';
  setPOV(newPOV);
  return newPOV;
}

/**
 * Get price for buying (yes side)
 * - Maker: bid price (you're making liquidity)
 * - Taker: ask price (you're taking liquidity)
 */
export function getBuyPrice(yesBid: number | null, yesAsk: number | null): number | null {
  if (currentPOV === 'maker') {
    return yesBid; // Maker: bid for buy
  } else {
    return yesAsk; // Taker: ask for buy
  }
}

/**
 * Get price for selling (yes side)
 * - Maker: ask price (you're making liquidity)
 * - Taker: bid price (you're taking liquidity)
 */
export function getSellPrice(yesBid: number | null, yesAsk: number | null): number | null {
  if (currentPOV === 'maker') {
    return yesAsk; // Maker: ask for sell
  } else {
    return yesBid; // Taker: bid for sell
  }
}

/**
 * Get price for buying NO side
 */
export function getBuyNoPrice(noBid: number | null, noAsk: number | null): number | null {
  if (currentPOV === 'maker') {
    return noBid; // Maker: bid for buy
  } else {
    return noAsk; // Taker: ask for buy
  }
}

/**
 * Get price for selling NO side
 */
export function getSellNoPrice(noBid: number | null, noAsk: number | null): number | null {
  if (currentPOV === 'maker') {
    return noAsk; // Maker: ask for sell
  } else {
    return noBid; // Taker: bid for sell
  }
}

/**
 * Get display price for a market (for display in table)
 * For YES side, shows buy price
 */
export function getDisplayPrice(
  yesBid: number | null,
  yesAsk: number | null,
  side: 'yes' | 'no' = 'yes'
): number | null {
  if (side === 'yes') {
    return getBuyPrice(yesBid, yesAsk);
  } else {
    return getBuyNoPrice(yesBid, yesAsk);
  }
}
