/**
 * SmartRelayAdapter — Adapts the smart relay stream to produce ConsolidatedNbaUpdate
 * compatible with the existing dashboard.
 *
 * When dataMode === 'smart-relay', this replaces the REST discovery + per-ticker WS subscriptions
 * with a single stream from the relay that already has all cached data.
 */

import type { CachedMarket, CachedOrderbook } from '@galactus/shared';
import { createSmartRelayStream, type SmartRelayStream } from './smartRelayStream';
import type { ConsolidatedNbaUpdate, ConsolidatedNbaStream } from './stream';
import type { ConsolidatedGameBooks } from './nbaConsolidated/types';
import { makeGameKey } from './nbaConsolidated/types';
import {
  parseKalshiNbaEventTicker,
  parseKalshiNbaMarketTicker,
} from './nbaConsolidated/kalshiNbaTicker';

/**
 * Create a ConsolidatedNbaStream backed by the smart relay.
 *
 * The smart relay already has all market data cached. We subscribe once,
 * receive a snapshot, then get incremental updates — no per-ticker REST calls.
 */
export function createSmartRelayConsolidatedStream(): ConsolidatedNbaStream {
  const callbacks: Set<(u: ConsolidatedNbaUpdate) => void> = new Set();
  let stream: SmartRelayStream | null = null;
  let lastUpdateMs = 0;

  // Reconstruct ConsolidatedGameBooks from cached data
  const marketsByTicker: Map<string, CachedMarket> = new Map();
  const orderbooksByTicker: Map<string, CachedOrderbook> = new Map();
  const gamesByEvent: Map<string, ConsolidatedGameBooks> = new Map();

  const emit = () => {
    const games = Array.from(gamesByEvent.values());
    games.sort(
      (a, b) =>
        (a.date || '').localeCompare(b.date || '') ||
        a.awayCode.localeCompare(b.awayCode) ||
        a.homeCode.localeCompare(b.homeCode)
    );
    const payload: ConsolidatedNbaUpdate = { games, lastUpdateMs };
    callbacks.forEach((cb) => {
      try {
        cb(payload);
      } catch (err) {
        console.error('Error in smart relay consolidated callback:', err);
      }
    });
  };

  const rebuildGames = () => {
    gamesByEvent.clear();

    // Group Kalshi markets by event ticker
    const grouped: Map<
      string,
      {
        eventInfo: ReturnType<typeof parseKalshiNbaEventTicker>;
        away?: { ticker: string; market: CachedMarket; ob?: CachedOrderbook };
        home?: { ticker: string; market: CachedMarket; ob?: CachedOrderbook };
      }
    > = new Map();

    for (const [ticker, market] of marketsByTicker) {
      if (market.venue !== 'kalshi') continue;

      const mi = parseKalshiNbaMarketTicker(ticker);
      if (!mi) continue;
      const ei = parseKalshiNbaEventTicker(mi.eventTicker);
      if (!ei) continue;

      const entry = grouped.get(mi.eventTicker) ?? { eventInfo: ei };
      const ob = orderbooksByTicker.get(ticker);

      if (mi.teamCode === ei.awayCode) {
        entry.away = { ticker, market, ob };
      }
      if (mi.teamCode === ei.homeCode) {
        entry.home = { ticker, market, ob };
      }
      grouped.set(mi.eventTicker, entry);
    }

    for (const [eventTicker, entry] of grouped) {
      if (!entry.eventInfo || !entry.away || !entry.home) continue;
      const { dateYyyyMmDd, awayCode, homeCode } = entry.eventInfo;

      const game: ConsolidatedGameBooks = {
        date: dateYyyyMmDd,
        awayCode,
        homeCode,
        key: makeGameKey({ date: dateYyyyMmDd, awayCode, homeCode }),
        startTimePt: null,
        kalshi: {
          venue: 'kalshi',
          eventId: eventTicker,
          awayCode,
          homeCode,
          tsMs: Date.now(),
          markets: {
            away: {
              marketTicker: entry.away.ticker,
              yes: entry.away.ob?.yes ?? [],
              no: entry.away.ob?.no ?? [],
            },
            home: {
              marketTicker: entry.home.ticker,
              yes: entry.home.ob?.yes ?? [],
              no: entry.home.ob?.no ?? [],
            },
          },
        },
        polymarket: null,
      };

      // Attach polymarket data if available
      // Polymarket tokens are keyed as poly:<tokenId>
      // We don't have game-level matching here, so skip for now
      // The relay could send richer metadata in the future

      gamesByEvent.set(eventTicker, game);
    }

    lastUpdateMs = Date.now();
  };

  const handleMarketUpdate = (market: CachedMarket) => {
    marketsByTicker.set(market.ticker, market);
    // Incremental: update existing game if it exists
    for (const [_eventTicker, game] of gamesByEvent) {
      const kb = game.kalshi;
      if (!kb) continue;
      if (
        kb.markets.away.marketTicker === market.ticker ||
        kb.markets.home.marketTicker === market.ticker
      ) {
        lastUpdateMs = Date.now();
        emit();
        return;
      }
    }
    // If not found in existing games, rebuild (new market discovered)
    rebuildGames();
    emit();
  };

  const handleOrderbookUpdate = (orderbook: CachedOrderbook) => {
    orderbooksByTicker.set(orderbook.ticker, orderbook);
    // Update the relevant game's orderbook
    for (const [_eventTicker, game] of gamesByEvent) {
      const kb = game.kalshi;
      if (!kb) continue;
      if (kb.markets.away.marketTicker === orderbook.ticker) {
        kb.markets.away.yes = orderbook.yes;
        kb.markets.away.no = orderbook.no;
        kb.tsMs = Date.now();
        lastUpdateMs = Date.now();
        emit();
        return;
      }
      if (kb.markets.home.marketTicker === orderbook.ticker) {
        kb.markets.home.yes = orderbook.yes;
        kb.markets.home.no = orderbook.no;
        kb.tsMs = Date.now();
        lastUpdateMs = Date.now();
        emit();
        return;
      }
    }
  };

  return {
    start: async () => {
      stream = createSmartRelayStream({
        onMarketSnapshot: (markets) => {
          marketsByTicker.clear();
          for (const m of markets) {
            marketsByTicker.set(m.ticker, m);
          }
          rebuildGames();
          emit();
        },
        onOrderbookSnapshot: (orderbooks) => {
          orderbooksByTicker.clear();
          for (const ob of orderbooks) {
            orderbooksByTicker.set(ob.ticker, ob);
          }
          rebuildGames();
          emit();
        },
        onMarketUpdate: handleMarketUpdate,
        onOrderbookUpdate: handleOrderbookUpdate,
        onError: (err) => console.error('Smart relay stream error:', err),
      });

      stream.connect();

      // Subscribe to both channels
      // Small delay to ensure WS is open before subscribing
      await new Promise<void>((resolve) => {
        // Timeout after 10s
        const timeout = setTimeout(() => resolve(), 10000);
        const wrappedCheck = () => {
          if (stream!.isConnected()) {
            clearTimeout(timeout);
            stream!.subscribe('markets');
            stream!.subscribe('orderbooks');
            resolve();
          } else {
            setTimeout(wrappedCheck, 100);
          }
        };
        wrappedCheck();
      });
    },

    stop: () => {
      if (stream) {
        stream.disconnect();
        stream = null;
      }
      marketsByTicker.clear();
      orderbooksByTicker.clear();
      gamesByEvent.clear();
      lastUpdateMs = 0;
    },

    onUpdate: (cb) => callbacks.add(cb),
    offUpdate: (cb) => callbacks.delete(cb),
  };
}
