import { describe, expect, it } from 'vitest';
import { groupMarketsByEvent, type DiscoveredMarket } from './discover';

function makeDiscoveredMarket(overrides: Partial<DiscoveredMarket>): DiscoveredMarket {
  return {
    market: {} as DiscoveredMarket['market'],
    eventTicker: 'KXNBAGAME-26FEB04BOSMIA',
    marketTicker: 'KXNBAGAME-26FEB04BOSMIA-BOS',
    sport: 'nba',
    marketType: 'moneyline',
    dateYyyyMmDd: '2026-02-04',
    awayCode: 'BOS',
    homeCode: 'MIA',
    ...overrides,
  };
}

describe('groupMarketsByEvent', () => {
  it('returns empty map for empty input', () => {
    const result = groupMarketsByEvent([]);
    expect(result.size).toBe(0);
  });

  it('groups markets by event ticker', () => {
    const markets: DiscoveredMarket[] = [
      makeDiscoveredMarket({
        eventTicker: 'KXNBAGAME-26FEB04BOSMIA',
        marketTicker: 'KXNBAGAME-26FEB04BOSMIA-BOS',
        outcomeCode: 'BOS',
        outcomeName: 'Boston Celtics',
      }),
      makeDiscoveredMarket({
        eventTicker: 'KXNBAGAME-26FEB04BOSMIA',
        marketTicker: 'KXNBAGAME-26FEB04BOSMIA-MIA',
        outcomeCode: 'MIA',
        outcomeName: 'Miami Heat',
      }),
    ];

    const result = groupMarketsByEvent(markets);
    expect(result.size).toBe(1);

    const event = result.get('KXNBAGAME-26FEB04BOSMIA')!;
    expect(event.markets.size).toBe(2);
    expect(event.awayCode).toBe('BOS');
    expect(event.homeCode).toBe('MIA');
  });

  it('assigns away/home names from outcomeName', () => {
    const markets: DiscoveredMarket[] = [
      makeDiscoveredMarket({
        outcomeCode: 'BOS',
        outcomeName: 'Boston Celtics',
      }),
      makeDiscoveredMarket({
        marketTicker: 'KXNBAGAME-26FEB04BOSMIA-MIA',
        outcomeCode: 'MIA',
        outcomeName: 'Miami Heat',
      }),
    ];

    const result = groupMarketsByEvent(markets);
    const event = result.get('KXNBAGAME-26FEB04BOSMIA')!;
    expect(event.awayName).toBe('Boston Celtics');
    expect(event.homeName).toBe('Miami Heat');
  });

  it('groups multiple events separately', () => {
    const markets: DiscoveredMarket[] = [
      makeDiscoveredMarket({
        eventTicker: 'KXNBAGAME-26FEB04BOSMIA',
        marketTicker: 'KXNBAGAME-26FEB04BOSMIA-BOS',
        outcomeCode: 'BOS',
      }),
      makeDiscoveredMarket({
        eventTicker: 'KXNBAGAME-26FEB04LALGSW',
        marketTicker: 'KXNBAGAME-26FEB04LALGSW-LAL',
        awayCode: 'LAL',
        homeCode: 'GSW',
        outcomeCode: 'LAL',
      }),
    ];

    const result = groupMarketsByEvent(markets);
    expect(result.size).toBe(2);
    expect(result.has('KXNBAGAME-26FEB04BOSMIA')).toBe(true);
    expect(result.has('KXNBAGAME-26FEB04LALGSW')).toBe(true);
  });

  it('handles spread markets with bucket keys', () => {
    const markets: DiscoveredMarket[] = [
      makeDiscoveredMarket({
        eventTicker: 'KXNBASPREAD-26FEB03NYKWAS',
        marketTicker: 'KXNBASPREAD-26FEB03NYKWAS-NYK7',
        marketType: 'spread',
        outcomeCode: 'NYK',
        spreadBucket: 7,
      }),
      makeDiscoveredMarket({
        eventTicker: 'KXNBASPREAD-26FEB03NYKWAS',
        marketTicker: 'KXNBASPREAD-26FEB03NYKWAS-NYK4',
        marketType: 'spread',
        outcomeCode: 'NYK',
        spreadBucket: 4,
      }),
    ];

    const result = groupMarketsByEvent(markets);
    const event = result.get('KXNBASPREAD-26FEB03NYKWAS')!;
    // Spread markets keyed by "{teamCode}-{bucket}" so they don't collide
    expect(event.markets.size).toBe(2);
    expect(event.markets.has('NYK-7')).toBe(true);
    expect(event.markets.has('NYK-4')).toBe(true);
  });
});
