/**
 * Sports Market Discovery
 *
 * Fetches and discovers markets for any supported sport and market type.
 */

import type { KalshiApiClient, KalshiMarket } from '../../types';
import { type Sport, type MarketType, getSeriesTicker, getSportConfig } from './types';
import { parseEventTicker, parseMarketTicker } from './tickerParser';

/** Default number of days ahead to fetch markets */
const DEFAULT_DAYS_AHEAD = 30;

export interface DiscoveryOptions {
  sport: Sport;
  marketType: MarketType;
  /** Number of days ahead to include (default: 30) */
  daysAhead?: number;
  /** Maximum markets to fetch (default: 200) */
  limit?: number;
}

export interface DiscoveredMarket {
  market: KalshiMarket;
  eventTicker: string;
  marketTicker: string;
  sport: Sport;
  marketType: MarketType;
  dateYyyyMmDd: string;
  awayCode: string;
  homeCode: string;
  /** For moneylines: which team/player this market is for */
  outcomeCode?: string;
  /** Display name for this outcome (from yes_sub_title) */
  outcomeName?: string;
  /** For spreads: bucket number from ticker */
  spreadBucket?: number;
  /** Numeric value from floor_strike (precise for spreads/totals) */
  floorStrike?: number;
  /** For totals: strike value */
  totalStrike?: number;
}

export interface GroupedEvent {
  eventTicker: string;
  sport: Sport;
  marketType: MarketType;
  dateYyyyMmDd: string;
  awayCode: string;
  homeCode: string;
  /** Display name for away team/player (from yes_sub_title) */
  awayName?: string;
  /** Display name for home team/player (from yes_sub_title) */
  homeName?: string;
  /** Markets within this event, keyed by outcome */
  markets: Map<string, DiscoveredMarket>;
}

/**
 * Discover markets for a specific sport and market type
 */
export async function discoverSportsMarkets(
  api: KalshiApiClient,
  options: DiscoveryOptions
): Promise<DiscoveredMarket[]> {
  const { sport, marketType, daysAhead = DEFAULT_DAYS_AHEAD, limit = 200 } = options;

  const seriesTicker = getSeriesTicker(sport, marketType);
  if (!seriesTicker) {
    console.warn(`No series ticker for ${sport} ${marketType}`);
    return [];
  }

  const sportConfig = getSportConfig(sport);
  if (!sportConfig) {
    console.warn(`No config for sport: ${sport}`);
    return [];
  }

  try {
    // Fetch markets for the series
    const markets = await api.getMarkets({
      series_ticker: seriesTicker,
      status: 'open',
      limit,
    });

    // Filter by date range using YYYY-MM-DD string comparison.
    // Use local date (not UTC) so evening US users still see today's games.
    const now = new Date();
    const pad2 = (n: number) => String(n).padStart(2, '0');
    const todayStr = `${now.getFullYear()}-${pad2(now.getMonth() + 1)}-${pad2(now.getDate())}`;
    const cutoffDate = new Date(now.getTime() + daysAhead * 24 * 60 * 60 * 1000);
    const cutoffStr = `${cutoffDate.getFullYear()}-${pad2(cutoffDate.getMonth() + 1)}-${pad2(cutoffDate.getDate())}`;

    const discovered: DiscoveredMarket[] = [];

    for (const market of markets) {
      // Parse the market ticker
      const parsed = parseMarketTicker(market.market_ticker);
      if (!parsed) continue;

      // Parse the event ticker to get date and team codes
      const eventInfo = parseEventTicker(parsed.eventTicker);
      if (!eventInfo) continue;

      // Filter by date using string comparison (YYYY-MM-DD sorts correctly)
      const marketDateStr = eventInfo.dateYyyyMmDd;
      if (marketDateStr < todayStr || marketDateStr > cutoffStr) continue;

      discovered.push({
        market,
        eventTicker: parsed.eventTicker,
        marketTicker: market.market_ticker,
        sport: parsed.sport,
        marketType: parsed.marketType,
        dateYyyyMmDd: eventInfo.dateYyyyMmDd,
        awayCode: eventInfo.awayCode,
        homeCode: eventInfo.homeCode,
        outcomeCode: parsed.outcomeCode,
        outcomeName: market.yes_sub_title,
        spreadBucket: parsed.spreadBucket,
        floorStrike: market.floor_strike ?? undefined,
        totalStrike: parsed.totalStrike,
      });
    }

    return discovered;
  } catch (error) {
    console.error(`Error discovering ${sport} ${marketType} markets:`, error);
    return [];
  }
}

/**
 * Group discovered markets by event
 */
export function groupMarketsByEvent(markets: DiscoveredMarket[]): Map<string, GroupedEvent> {
  const grouped = new Map<string, GroupedEvent>();

  for (const dm of markets) {
    let event = grouped.get(dm.eventTicker);
    if (!event) {
      event = {
        eventTicker: dm.eventTicker,
        sport: dm.sport,
        marketType: dm.marketType,
        dateYyyyMmDd: dm.dateYyyyMmDd,
        awayCode: dm.awayCode,
        homeCode: dm.homeCode,
        awayName: undefined,
        homeName: undefined,
        markets: new Map(),
      };
      grouped.set(dm.eventTicker, event);
    }

    // Extract away/home names from outcome markets
    // For moneylines: outcomeCode matches awayCode or homeCode
    if (dm.outcomeName && dm.outcomeCode) {
      const outcomeUpper = dm.outcomeCode.toUpperCase();
      if (outcomeUpper === dm.awayCode.toUpperCase()) {
        event.awayName = dm.outcomeName;
      }
      if (outcomeUpper === dm.homeCode.toUpperCase()) {
        event.homeName = dm.outcomeName;
      }
    }

    // Key by unique identifier for this market within the event
    // For spreads: include bucket to avoid collisions (PHI1, PHI4, PHI7 are different markets)
    // For totals: outcomeCode already includes strike (O232, O237)
    // For moneylines: outcomeCode is the team/player code
    let marketKey: string;
    if (dm.spreadBucket !== undefined) {
      marketKey = `${dm.outcomeCode}-${dm.spreadBucket}`;
    } else {
      marketKey = dm.outcomeCode ?? dm.marketTicker;
    }
    event.markets.set(marketKey, dm);
  }

  // Fix-up: for moneyline events, correct awayCode/homeCode using outcomeCode.
  // Market tickers are unambiguous (e.g., "...-UTAH") so outcomeCode is always correct,
  // even when the event ticker's matchup string was mis-split (e.g., "UTAHBYU" → "UTA"+"HBYU").
  // We reconstruct the correct split by checking which outcomeCode ordering matches the
  // original concatenated matchup string.
  for (const event of grouped.values()) {
    if (event.marketType !== 'moneyline') continue;
    // Already correctly matched — names are set
    if (event.awayName && event.homeName) continue;

    const outcomeCodes: string[] = [];
    for (const dm of event.markets.values()) {
      if (dm.outcomeCode) outcomeCodes.push(dm.outcomeCode.toUpperCase());
    }
    if (outcomeCodes.length !== 2) continue;

    const matchupStr = (event.awayCode + event.homeCode).toUpperCase();
    const [c1, c2] = outcomeCodes;
    if (!c1 || !c2) continue;

    if (c1 + c2 === matchupStr) {
      event.awayCode = c1;
      event.homeCode = c2;
    } else if (c2 + c1 === matchupStr) {
      event.awayCode = c2;
      event.homeCode = c1;
    } else {
      continue; // Couldn't match — leave as-is
    }

    // Re-run name matching with corrected codes
    for (const dm of event.markets.values()) {
      if (dm.outcomeName && dm.outcomeCode) {
        const outcomeUpper = dm.outcomeCode.toUpperCase();
        if (outcomeUpper === event.awayCode) {
          event.awayName = dm.outcomeName;
        }
        if (outcomeUpper === event.homeCode) {
          event.homeName = dm.outcomeName;
        }
      }
    }
  }

  return grouped;
}

/**
 * Discover all market types for a sport
 */
export async function discoverAllMarketsForSport(
  api: KalshiApiClient,
  sport: Sport,
  daysAhead: number = DEFAULT_DAYS_AHEAD
): Promise<Map<MarketType, DiscoveredMarket[]>> {
  const config = getSportConfig(sport);
  if (!config) return new Map();

  const results = new Map<MarketType, DiscoveredMarket[]>();

  for (const marketType of config.marketTypes) {
    const markets = await discoverSportsMarkets(api, {
      sport,
      marketType,
      daysAhead,
    });
    results.set(marketType, markets);
  }

  return results;
}
