import { describe, expect, it, vi } from 'vitest';
import { createSportsStream, type SportsStream } from './stream';
import type { SportsStreamUpdate } from './types';

/**
 * Integration tests for the sports stream loading state machine.
 *
 * These use mock mode (useMock: true) to bypass external API/WS calls,
 * validating the state machine lifecycle without network dependencies.
 */

/** Mock CryptoKey (unused in mock mode) */
const MOCK_KEY = {} as CryptoKey;

function createMockStream(
  sport: 'nba' | 'nhl' = 'nba',
  marketType: 'moneyline' | 'spread' | 'total' = 'moneyline'
): SportsStream {
  return createSportsStream(
    {
      api: null,
      accessKeyId: '',
      privateKey: MOCK_KEY,
      environment: 'demo',
      useRelay: false,
      useMock: true,
    },
    { sport, marketType }
  );
}

describe('createSportsStream — mock mode lifecycle', () => {
  it('emits update when started in mock mode', async () => {
    const stream = createMockStream();
    const updates: SportsStreamUpdate[] = [];
    stream.onUpdate((u) => updates.push(u));

    await stream.start();

    expect(updates.length).toBeGreaterThanOrEqual(1);
    const lastUpdate = updates[updates.length - 1]!;
    expect(lastUpdate.sport).toBe('nba');
    expect(lastUpdate.marketType).toBe('moneyline');
    expect(lastUpdate.loadingState.phase).toBe('live');
    expect(lastUpdate.loadingState.isLoading).toBe(false);
  });

  it('goes live with empty games in mock mode', async () => {
    const stream = createMockStream();
    const updates: SportsStreamUpdate[] = [];
    stream.onUpdate((u) => updates.push(u));

    await stream.start();

    const lastUpdate = updates[updates.length - 1]!;
    expect(lastUpdate.games).toEqual([]);
  });

  it('progresses through discovering-markets before going live', async () => {
    const stream = createMockStream();
    const phases: string[] = [];
    stream.onUpdate((u) => phases.push(u.loadingState.phase));

    await stream.start();

    expect(phases).toContain('discovering-markets');
    expect(phases[phases.length - 1]).toBe('live');
  });

  it('stop resets state cleanly', async () => {
    const stream = createMockStream();
    const updates: SportsStreamUpdate[] = [];
    stream.onUpdate((u) => updates.push(u));

    await stream.start();
    expect(updates.length).toBeGreaterThan(0);

    stream.stop();
    // After stop, stream should be re-startable
    updates.length = 0;
    await stream.start();
    expect(updates.length).toBeGreaterThan(0);
  });

  it('offUpdate removes callback', async () => {
    const stream = createMockStream();
    const cb = vi.fn();
    stream.onUpdate(cb);
    stream.offUpdate(cb);

    await stream.start();
    expect(cb).not.toHaveBeenCalled();
  });

  it('works for spread market type', async () => {
    const stream = createMockStream('nba', 'spread');
    const updates: SportsStreamUpdate[] = [];
    stream.onUpdate((u) => updates.push(u));

    await stream.start();

    const lastUpdate = updates[updates.length - 1]!;
    expect(lastUpdate.sport).toBe('nba');
    expect(lastUpdate.marketType).toBe('spread');
    expect(lastUpdate.loadingState.phase).toBe('live');
  });

  it('works for total market type', async () => {
    const stream = createMockStream('nba', 'total');
    const updates: SportsStreamUpdate[] = [];
    stream.onUpdate((u) => updates.push(u));

    await stream.start();

    const lastUpdate = updates[updates.length - 1]!;
    expect(lastUpdate.sport).toBe('nba');
    expect(lastUpdate.marketType).toBe('total');
  });

  it('works for NHL sport', async () => {
    const stream = createMockStream('nhl', 'moneyline');
    const updates: SportsStreamUpdate[] = [];
    stream.onUpdate((u) => updates.push(u));

    await stream.start();

    const lastUpdate = updates[updates.length - 1]!;
    expect(lastUpdate.sport).toBe('nhl');
  });
});
