/**
 * Sports Stream
 *
 * Generalized WebSocket stream for any supported sport and market type.
 * Uses sportsDiscovery module for market discovery and ticker parsing.
 */

import type { KalshiApiClient } from '../../types';
import type { Environment } from '@galactus/shared';
import { createMarketStream, type MarketStream } from '../marketStream';
import { createPolyMarketStream } from '../polymarket/marketStream';
import {
  makePolyGameSlug,
  resolvePolyGameSlugByParticipants,
  resolvePolyMarketInfoBySlug,
  resolvePolyTennisGameSlug,
  resolvePolySpreadMarketsByGameSlug,
  type PolySpreadMarketInfo,
  resolvePolyTotalMarketsByGameSlug,
  type PolyTotalMarketInfo,
  clearGammaEventsCache,
} from '../polymarket/gamma';
import { polyAsksAsNoLevels, polyBidsAsYesLevels } from '../polymarket/normalize';
import {
  discoverSportsMarkets,
  groupMarketsByEvent,
  type DiscoveredMarket,
  type GroupedEvent,
} from '../sportsDiscovery';
import { applyStartTimeOffset } from '../nbaConsolidated/startTimeSettings';
import {
  type SportsStreamOptions,
  type SportsStreamUpdate,
  type SportsLoadingState,
  type SportsLoadPhase,
  type SportsLoadEvent,
  type GameData,
  type MoneylineGameData,
  type SpreadGameData,
  type TotalGameData,
  type MarketBook,
  type SpreadMarketInfo,
  type BookLevel,
  getBestBid,
} from './types';

/** Stream connection options */
interface StreamConnectionOptions {
  api: KalshiApiClient | null;
  accessKeyId: string;
  privateKey: CryptoKey;
  environment: Environment;
  useRelay: boolean;
  useMock: boolean;
}

/** Callback for stream updates */
type StreamCallback = (update: SportsStreamUpdate) => void;

export interface SportsStream {
  start: () => Promise<void>;
  stop: () => void;
  onUpdate: (callback: StreamCallback) => void;
  offUpdate: (callback: StreamCallback) => void;
}

/**
 * Parse team names from event title
 * Format: "{Away} at {Home}: Spread" or "{Away} at {Home}"
 */
function parseTeamNamesFromTitle(title: string): { awayName: string; homeName: string } | null {
  // Try with suffix (": Spread", ": Total", etc.)
  let match = title.match(/^(.+?) at (.+?)(?::|$)/);
  if (match && match[1] && match[2]) {
    return { awayName: match[1].trim(), homeName: match[2].trim() };
  }
  // Try without suffix
  match = title.match(/^(.+?) at (.+)$/);
  if (match && match[1] && match[2]) {
    return { awayName: match[1].trim(), homeName: match[2].trim() };
  }
  return null;
}

/** Convert raw orderbook to BookLevel array */
function orderbookToLevels(raw: Array<[number, number]> | null | undefined): BookLevel[] {
  if (!Array.isArray(raw)) return [];
  const out: BookLevel[] = [];
  for (const rec of raw) {
    if (!Array.isArray(rec) || rec.length < 2) continue;
    const [p, q] = rec;
    const priceCents = Number(p);
    const size = Number(q);
    if (!Number.isFinite(priceCents) || !Number.isFinite(size)) continue;
    if (size <= 0) continue;
    out.push({ priceCents, size });
  }
  // Ensure descending by price
  out.sort((a, b) => b.priceCents - a.priceCents);
  return out;
}

function normalizeParticipantName(name: string | null | undefined): string {
  return String(name ?? '')
    .normalize('NFKD')
    .replace(/[^\w\s]/g, '')
    .replace(/\s+/g, ' ')
    .trim()
    .toLowerCase();
}

/**
 * Build MoneylineGameData from grouped event
 */
function buildMoneylineGame(
  event: GroupedEvent,
  marketBooks: Map<string, MarketBook>
): MoneylineGameData {
  let awayBook: MarketBook | null = null;
  let homeBook: MarketBook | null = null;

  // PRIMARY: name-based matching (reliable for all sports, especially CBB where
  // ticker code splits are ambiguous). Uses display names from yes_sub_title.
  if (event.awayName && event.homeName) {
    const awayNameKey = normalizeParticipantName(event.awayName);
    const homeNameKey = normalizeParticipantName(event.homeName);
    for (const dm of event.markets.values()) {
      const outcomeNameKey = normalizeParticipantName(dm.outcomeName);
      if (!outcomeNameKey) continue;
      if (!awayBook && outcomeNameKey === awayNameKey) {
        awayBook = marketBooks.get(dm.marketTicker) ?? null;
      }
      if (!homeBook && outcomeNameKey === homeNameKey) {
        homeBook = marketBooks.get(dm.marketTicker) ?? null;
      }
      if (awayBook && homeBook) break;
    }
  }

  // FALLBACK: code-based matching (works for unambiguous sports like NBA/NHL)
  if (!awayBook) awayBook = marketBooks.get(event.awayCode) ?? null;
  if (!homeBook) homeBook = marketBooks.get(event.homeCode) ?? null;

  return {
    marketType: 'moneyline',
    eventTicker: event.eventTicker,
    date: event.dateYyyyMmDd,
    awayCode: event.awayCode,
    homeCode: event.homeCode,
    awayName: event.awayName,
    homeName: event.homeName,
    startTimePt: null, // Will be fetched from event API
    away: awayBook,
    home: homeBook,
    polymarket: null,
  };
}

/**
 * Build SpreadGameData from grouped event
 */
function buildSpreadGame(
  event: GroupedEvent,
  marketBooks: Map<string, MarketBook>
): SpreadGameData {
  const markets = new Map<string, SpreadMarketInfo>();

  for (const [outcomeKey, dm] of event.markets.entries()) {
    const book = marketBooks.get(dm.marketTicker);
    if (book) {
      // Key format: "{teamCode}-{bucket}" e.g., "NYK-7"
      const key =
        dm.spreadBucket !== undefined ? `${dm.outcomeCode}-${dm.spreadBucket}` : outcomeKey;

      // Get floor_strike from discovered market (more accurate than bucket-based calculation)
      const floorStrike =
        dm.floorStrike ?? (dm.spreadBucket !== undefined ? dm.spreadBucket + 0.5 : 0);

      markets.set(key, {
        book,
        floorStrike,
        teamCode: dm.outcomeCode ?? '',
      });
    }
  }

  return {
    marketType: 'spread',
    eventTicker: event.eventTicker,
    date: event.dateYyyyMmDd,
    awayCode: event.awayCode,
    homeCode: event.homeCode,
    awayName: event.awayName,
    homeName: event.homeName,
    startTimePt: null,
    markets,
    polymarketMain: null,
  };
}

/**
 * Build TotalGameData from grouped event
 */
function buildTotalGame(event: GroupedEvent, marketBooks: Map<string, MarketBook>): TotalGameData {
  const markets = new Map<number, MarketBook>();

  for (const [_outcomeKey, dm] of event.markets.entries()) {
    const book = marketBooks.get(dm.marketTicker);
    // Prefer floor_strike for precise line values (e.g., 210.5) and fallback to ticker strike.
    const preciseLine = dm.floorStrike ?? dm.totalStrike;
    if (book && preciseLine !== undefined && Number.isFinite(preciseLine)) {
      markets.set(preciseLine, book);
    }
  }

  return {
    marketType: 'total',
    eventTicker: event.eventTicker,
    date: event.dateYyyyMmDd,
    awayCode: event.awayCode,
    homeCode: event.homeCode,
    awayName: event.awayName,
    homeName: event.homeName,
    startTimePt: null,
    markets,
    polymarketMain: null,
  };
}

/**
 * Create a sports stream for a specific sport and market type
 */
export function createSportsStream(
  connectionOptions: StreamConnectionOptions,
  streamOptions: SportsStreamOptions
): SportsStream {
  const { api, accessKeyId, privateKey, environment, useRelay, useMock } = connectionOptions;
  const { sport, marketType, daysAhead = 7 } = streamOptions;

  const callbacks: Set<StreamCallback> = new Set();
  let marketStream: MarketStream | null = null;
  let polyStream: ReturnType<typeof createPolyMarketStream> | null = null;

  // Internal state
  const gamesByEvent = new Map<string, GameData>();
  const marketBooks = new Map<string, MarketBook>();
  const eventStartTimes = new Map<string, string | null>(); // eventTicker -> startTimePt
  const eventTeamNames = new Map<string, { awayName: string; homeName: string }>(); // eventTicker -> names from title
  let discoveredMarkets: DiscoveredMarket[] = [];
  let fallbackEventScaffoldOnly = false;
  let tickers: string[] = [];
  let lastUpdateMs = 0;
  const polyTokenToEvent = new Map<
    string,
    { eventTicker: string; side: 'away' | 'home' | 'favorite' | 'underdog' | 'over' | 'under' }
  >();
  const polyMoneylineByEvent = new Map<string, NonNullable<MoneylineGameData['polymarket']>>();
  const polySpreadByEvent = new Map<string, NonNullable<SpreadGameData['polymarketMain']>>();
  const polyTotalByEvent = new Map<string, NonNullable<TotalGameData['polymarketMain']>>();
  let loadingState: SportsLoadingState = {
    isLoading: false,
    phase: 'idle',
    startedAtMs: 0,
    updatedAtMs: 0,
    events: [],
  };

  const updateLoadingState = (
    phase: SportsLoadPhase,
    args?: { message?: string; done?: number; total?: number; level?: SportsLoadEvent['level'] }
  ) => {
    const now = Date.now();
    const shouldAppendEvent = typeof args?.message === 'string' && args.message.length > 0;
    loadingState = {
      ...loadingState,
      phase,
      isLoading: phase !== 'live' && phase !== 'idle' && phase !== 'error',
      updatedAtMs: now,
      done: args?.done,
      total: args?.total,
      events: shouldAppendEvent
        ? [
            ...loadingState.events,
            {
              tsMs: now,
              phase,
              level: args?.level ?? 'info',
              message: args.message ?? '',
              done: args?.done,
              total: args?.total,
            },
          ]
        : loadingState.events,
    };
  };

  const pickMainKalshiSpread = (
    game: SpreadGameData
  ): { floorStrike: number; favoriteCode: string } | null => {
    let best: { floorStrike: number; favoriteCode: string } | null = null;
    let minDiff = Infinity;
    for (const [, info] of game.markets) {
      const yesBid = getBestBid(info.book, 'yes');
      if (yesBid.priceCents === null) continue;
      const diff = Math.abs(yesBid.priceCents - 50);
      if (diff < minDiff) {
        minDiff = diff;
        best = {
          floorStrike: info.floorStrike,
          favoriteCode: info.teamCode,
        };
      }
    }
    return best;
  };

  const pickPolySpreadCandidate = (
    candidates: PolySpreadMarketInfo[],
    desiredFavoriteSide: 'away' | 'home',
    desiredLine: number
  ): PolySpreadMarketInfo | null => {
    let pool = candidates.filter((c) => c.favoriteSide === desiredFavoriteSide);
    if (pool.length === 0) pool = candidates;
    if (pool.length === 0) return null;
    let best = pool[0] ?? null;
    let minDiff = best ? Math.abs(best.spreadValue - desiredLine) : Infinity;
    for (const c of pool) {
      const diff = Math.abs(c.spreadValue - desiredLine);
      if (diff < minDiff) {
        minDiff = diff;
        best = c;
      }
    }
    return best;
  };

  const pickMainKalshiTotal = (game: TotalGameData): number | null => {
    let line: number | null = null;
    let minDiff = Infinity;
    for (const [strike, book] of game.markets) {
      const yesBid = getBestBid(book, 'yes');
      if (yesBid.priceCents === null) continue;
      const diff = Math.abs(yesBid.priceCents - 50);
      if (diff < minDiff) {
        minDiff = diff;
        line = strike;
      }
    }
    return line;
  };

  const pickPolyTotalCandidate = (
    candidates: PolyTotalMarketInfo[],
    desiredLine: number
  ): PolyTotalMarketInfo | null => {
    if (candidates.length === 0) return null;
    let best = candidates[0] ?? null;
    let minDiff = best ? Math.abs(best.totalLine - desiredLine) : Infinity;
    for (const c of candidates) {
      const diff = Math.abs(c.totalLine - desiredLine);
      if (diff < minDiff) {
        minDiff = diff;
        best = c;
      }
    }
    return best;
  };

  const hydratePolyMoneylines = async (): Promise<void> => {
    if (marketType !== 'moneyline') return;
    polyTokenToEvent.clear();
    polyMoneylineByEvent.clear();
    clearGammaEventsCache();

    const grouped = groupMarketsByEvent(discoveredMarkets);
    const tokenIds: string[] = [];
    const totalEvents = grouped.size;
    let resolvedEvents = 0;
    updateLoadingState('hydrating-polymarket', {
      message: 'Resolving Polymarket moneylines',
      done: 0,
      total: totalEvents,
    });
    emit();

    const resolves = Array.from(grouped.values()).map(async (event) => {
      try {
        let slug = makePolyGameSlug({
          sport,
          dateYyyyMmDd: event.dateYyyyMmDd,
          awayCode: event.awayCode,
          homeCode: event.homeCode,
        });

        let info = slug
          ? await resolvePolyMarketInfoBySlug({
              slug,
              useRelay,
              awayName: event.awayName,
              homeName: event.homeName,
            })
          : null;
        if (
          !info &&
          (sport === 'cbb' || sport === 'wcbb' || sport === 'tennis-atp' || sport === 'tennis-wta')
        ) {
          const resolved = await resolvePolyGameSlugByParticipants({
            sport,
            dateYyyyMmDd: event.dateYyyyMmDd,
            awayName: event.awayName,
            homeName: event.homeName,
            useRelay,
          });
          if (resolved) {
            slug = resolved;
            info = await resolvePolyMarketInfoBySlug({
              slug,
              useRelay,
              awayName: event.awayName,
              homeName: event.homeName,
            });
          }
        }
        return { eventTicker: event.eventTicker, info };
      } catch (err) {
        console.warn(`Polymarket moneyline resolve failed for ${event.eventTicker}:`, err);
        return { eventTicker: event.eventTicker, info: null };
      } finally {
        resolvedEvents += 1;
        updateLoadingState('hydrating-polymarket', {
          done: resolvedEvents,
          total: totalEvents,
        });
        emit();
      }
    });

    const selected = await Promise.all(resolves);
    for (const rec of selected) {
      if (!rec.info) continue;
      const game = gamesByEvent.get(rec.eventTicker);
      if (!game || game.marketType !== 'moneyline') continue;

      const awayBook: MarketBook = {
        marketTicker: `asset:${rec.info.awayTokenId}`,
        tokenId: rec.info.awayTokenId,
        conditionId: rec.info.conditionId,
        tickSize: rec.info.tickSize,
        negRisk: rec.info.negRisk,
        yes:
          typeof rec.info.awayPriceCents === 'number'
            ? [{ priceCents: rec.info.awayPriceCents, size: 1 }]
            : [],
        no:
          typeof rec.info.awayPriceCents === 'number'
            ? [{ priceCents: 100 - rec.info.awayPriceCents, size: 1 }]
            : [],
      };
      const homeBook: MarketBook = {
        marketTicker: `asset:${rec.info.homeTokenId}`,
        tokenId: rec.info.homeTokenId,
        conditionId: rec.info.conditionId,
        tickSize: rec.info.tickSize,
        negRisk: rec.info.negRisk,
        yes:
          typeof rec.info.homePriceCents === 'number'
            ? [{ priceCents: rec.info.homePriceCents, size: 1 }]
            : [],
        no:
          typeof rec.info.homePriceCents === 'number'
            ? [{ priceCents: 100 - rec.info.homePriceCents, size: 1 }]
            : [],
      };

      const polyMoneyline: NonNullable<MoneylineGameData['polymarket']> = {
        eventId: rec.info.slug,
        liquidityUsd: rec.info.liquidityUsd,
        markets: {
          away: awayBook,
          home: homeBook,
        },
      };
      polyMoneylineByEvent.set(rec.eventTicker, polyMoneyline);
      game.polymarket = polyMoneyline;

      polyTokenToEvent.set(rec.info.awayTokenId, {
        eventTicker: rec.eventTicker,
        side: 'away',
      });
      polyTokenToEvent.set(rec.info.homeTokenId, {
        eventTicker: rec.eventTicker,
        side: 'home',
      });
      tokenIds.push(rec.info.awayTokenId, rec.info.homeTokenId);
    }

    if (tokenIds.length === 0) {
      updateLoadingState('hydrating-polymarket', {
        message: 'No Polymarket moneylines matched',
        done: resolvedEvents,
        total: totalEvents,
        level: 'warn',
      });
      emit();
      return;
    }

    polyStream = createPolyMarketStream();
    polyStream.onError((e) => {
      console.warn(`Polymarket moneyline stream error (${sport} ${marketType}):`, e);
      updateLoadingState(loadingState.phase, {
        message: 'Polymarket stream error',
        level: 'error',
      });
      emit();
    });
    polyStream.onBook((snap) => {
      const mapping = polyTokenToEvent.get(snap.assetId);
      if (!mapping) return;
      const poly = polyMoneylineByEvent.get(mapping.eventTicker);
      if (!poly) return;

      const book = mapping.side === 'away' ? poly.markets.away : poly.markets.home;
      book.yes = polyBidsAsYesLevels(snap.bids);
      book.no = polyAsksAsNoLevels(snap.asks);
      book.tsMs = snap.tsMs;
      poly.tsMs = snap.tsMs;
      const game = gamesByEvent.get(mapping.eventTicker);
      if (game && game.marketType === 'moneyline') {
        game.polymarket = poly;
      }
      lastUpdateMs = Date.now();
      emit();
    });

    await polyStream.connect();
    polyStream.subscribe(Array.from(new Set(tokenIds)));
  };

  const hydratePolySpreads = async (): Promise<void> => {
    if (marketType !== 'spread') return;
    if (sport !== 'nfl' && sport !== 'nba' && sport !== 'tennis-atp' && sport !== 'tennis-wta')
      return;
    polyTokenToEvent.clear();
    polySpreadByEvent.clear();
    clearGammaEventsCache();

    const grouped = groupMarketsByEvent(discoveredMarkets);
    const tokenIds: string[] = [];
    const totalEvents = grouped.size;
    let resolvedEvents = 0;
    updateLoadingState('hydrating-polymarket', {
      message: 'Resolving Polymarket spreads',
      done: 0,
      total: totalEvents,
    });
    emit();

    const resolves = Array.from(grouped.values()).map(async (event) => {
      try {
        const isTennis = sport === 'tennis-atp' || sport === 'tennis-wta';
        let slug = isTennis
          ? null
          : makePolyGameSlug({
              sport,
              dateYyyyMmDd: event.dateYyyyMmDd,
              awayCode: event.awayCode,
              homeCode: event.homeCode,
            });
        if (isTennis) {
          slug = await resolvePolyTennisGameSlug({
            sport,
            dateYyyyMmDd: event.dateYyyyMmDd,
            awayName: event.awayName,
            homeName: event.homeName,
            useRelay,
          });
        }
        if (!slug) {
          return { eventTicker: event.eventTicker, candidate: null as PolySpreadMarketInfo | null };
        }

        const game = gamesByEvent.get(event.eventTicker);
        if (!game || game.marketType !== 'spread') {
          return { eventTicker: event.eventTicker, candidate: null as PolySpreadMarketInfo | null };
        }

        const mainKalshi = pickMainKalshiSpread(game);
        const desiredFavoriteSide =
          mainKalshi && mainKalshi.favoriteCode.toUpperCase() === event.awayCode.toUpperCase()
            ? ('away' as const)
            : ('home' as const);
        const desiredLine = mainKalshi?.floorStrike ?? 0;

        const candidates = await resolvePolySpreadMarketsByGameSlug({ slug, useRelay });
        const candidate = pickPolySpreadCandidate(candidates, desiredFavoriteSide, desiredLine);
        return { eventTicker: event.eventTicker, candidate };
      } catch (err) {
        console.warn(`Polymarket spread resolve failed for ${event.eventTicker}:`, err);
        return { eventTicker: event.eventTicker, candidate: null as PolySpreadMarketInfo | null };
      } finally {
        resolvedEvents += 1;
        updateLoadingState('hydrating-polymarket', {
          done: resolvedEvents,
          total: totalEvents,
        });
        emit();
      }
    });

    const selected = await Promise.all(resolves);
    for (const rec of selected) {
      if (!rec.candidate) continue;
      const game = gamesByEvent.get(rec.eventTicker);
      if (!game || game.marketType !== 'spread') continue;

      const favoriteBook: MarketBook = {
        marketTicker: `asset:${rec.candidate.favoriteTokenId}`,
        tokenId: rec.candidate.favoriteTokenId,
        conditionId: rec.candidate.conditionId,
        tickSize: rec.candidate.tickSize,
        negRisk: rec.candidate.negRisk,
        yes:
          typeof rec.candidate.favoritePriceCents === 'number'
            ? [{ priceCents: rec.candidate.favoritePriceCents, size: 1 }]
            : [],
        no:
          typeof rec.candidate.favoritePriceCents === 'number'
            ? [{ priceCents: 100 - rec.candidate.favoritePriceCents, size: 1 }]
            : [],
      };
      const underdogBook: MarketBook = {
        marketTicker: `asset:${rec.candidate.underdogTokenId}`,
        tokenId: rec.candidate.underdogTokenId,
        conditionId: rec.candidate.conditionId,
        tickSize: rec.candidate.tickSize,
        negRisk: rec.candidate.negRisk,
        yes:
          typeof rec.candidate.underdogPriceCents === 'number'
            ? [{ priceCents: rec.candidate.underdogPriceCents, size: 1 }]
            : [],
        no:
          typeof rec.candidate.underdogPriceCents === 'number'
            ? [{ priceCents: 100 - rec.candidate.underdogPriceCents, size: 1 }]
            : [],
      };

      const polyMain: NonNullable<SpreadGameData['polymarketMain']> = {
        line: rec.candidate.spreadValue,
        favoriteSide: rec.candidate.favoriteSide,
        liquidityUsd: rec.candidate.liquidityUsd,
        markets: {
          favorite: favoriteBook,
          underdog: underdogBook,
        },
      };
      polySpreadByEvent.set(rec.eventTicker, polyMain);
      game.polymarketMain = polyMain;

      polyTokenToEvent.set(rec.candidate.favoriteTokenId, {
        eventTicker: rec.eventTicker,
        side: 'favorite',
      });
      polyTokenToEvent.set(rec.candidate.underdogTokenId, {
        eventTicker: rec.eventTicker,
        side: 'underdog',
      });
      tokenIds.push(rec.candidate.favoriteTokenId, rec.candidate.underdogTokenId);
    }

    if (tokenIds.length === 0) return;

    polyStream = createPolyMarketStream();
    polyStream.onError((e) => {
      console.warn(`Polymarket spread stream error (${sport} ${marketType}):`, e);
      updateLoadingState(loadingState.phase, {
        message: 'Polymarket stream error',
        level: 'error',
      });
      emit();
    });
    polyStream.onBook((snap) => {
      const mapping = polyTokenToEvent.get(snap.assetId);
      if (!mapping) return;
      const polyMain = polySpreadByEvent.get(mapping.eventTicker);
      if (!polyMain) return;

      const book =
        mapping.side === 'favorite' ? polyMain.markets.favorite : polyMain.markets.underdog;
      book.yes = polyBidsAsYesLevels(snap.bids);
      book.no = polyAsksAsNoLevels(snap.asks);
      book.tsMs = snap.tsMs;
      const game = gamesByEvent.get(mapping.eventTicker);
      if (game && game.marketType === 'spread') {
        game.polymarketMain = polyMain;
      }
      lastUpdateMs = Date.now();
      emit();
    });

    await polyStream.connect();
    polyStream.subscribe(Array.from(new Set(tokenIds)));
  };

  const hydratePolyTotals = async (): Promise<void> => {
    if (marketType !== 'total') return;
    if (sport !== 'nba' && sport !== 'nfl' && sport !== 'tennis-atp' && sport !== 'tennis-wta')
      return;
    polyTokenToEvent.clear();
    polyTotalByEvent.clear();
    clearGammaEventsCache();

    const grouped = groupMarketsByEvent(discoveredMarkets);
    const tokenIds: string[] = [];
    const totalEvents = grouped.size;
    let resolvedEvents = 0;
    updateLoadingState('hydrating-polymarket', {
      message: 'Resolving Polymarket totals',
      done: 0,
      total: totalEvents,
    });
    emit();

    const resolves = Array.from(grouped.values()).map(async (event) => {
      try {
        const isTennis = sport === 'tennis-atp' || sport === 'tennis-wta';
        let slug = isTennis
          ? null
          : makePolyGameSlug({
              sport,
              dateYyyyMmDd: event.dateYyyyMmDd,
              awayCode: event.awayCode,
              homeCode: event.homeCode,
            });
        if (isTennis) {
          slug = await resolvePolyTennisGameSlug({
            sport,
            dateYyyyMmDd: event.dateYyyyMmDd,
            awayName: event.awayName,
            homeName: event.homeName,
            useRelay,
          });
        }
        if (!slug) {
          return { eventTicker: event.eventTicker, candidate: null as PolyTotalMarketInfo | null };
        }

        const game = gamesByEvent.get(event.eventTicker);
        if (!game || game.marketType !== 'total') {
          return { eventTicker: event.eventTicker, candidate: null as PolyTotalMarketInfo | null };
        }

        const desiredLine = pickMainKalshiTotal(game) ?? 0;
        const candidates = await resolvePolyTotalMarketsByGameSlug({ slug, useRelay });
        const candidate = pickPolyTotalCandidate(candidates, desiredLine);
        return { eventTicker: event.eventTicker, candidate };
      } catch (err) {
        console.warn(`Polymarket total resolve failed for ${event.eventTicker}:`, err);
        return { eventTicker: event.eventTicker, candidate: null as PolyTotalMarketInfo | null };
      } finally {
        resolvedEvents += 1;
        updateLoadingState('hydrating-polymarket', {
          done: resolvedEvents,
          total: totalEvents,
        });
        emit();
      }
    });

    const selected = await Promise.all(resolves);
    for (const rec of selected) {
      if (!rec.candidate) continue;
      const game = gamesByEvent.get(rec.eventTicker);
      if (!game || game.marketType !== 'total') continue;

      const overBook: MarketBook = {
        marketTicker: `asset:${rec.candidate.overTokenId}`,
        tokenId: rec.candidate.overTokenId,
        conditionId: rec.candidate.conditionId,
        tickSize: rec.candidate.tickSize,
        negRisk: rec.candidate.negRisk,
        yes:
          typeof rec.candidate.overPriceCents === 'number'
            ? [{ priceCents: rec.candidate.overPriceCents, size: 1 }]
            : [],
        no:
          typeof rec.candidate.overPriceCents === 'number'
            ? [{ priceCents: 100 - rec.candidate.overPriceCents, size: 1 }]
            : [],
      };
      const underBook: MarketBook = {
        marketTicker: `asset:${rec.candidate.underTokenId}`,
        tokenId: rec.candidate.underTokenId,
        conditionId: rec.candidate.conditionId,
        tickSize: rec.candidate.tickSize,
        negRisk: rec.candidate.negRisk,
        yes:
          typeof rec.candidate.underPriceCents === 'number'
            ? [{ priceCents: rec.candidate.underPriceCents, size: 1 }]
            : [],
        no:
          typeof rec.candidate.underPriceCents === 'number'
            ? [{ priceCents: 100 - rec.candidate.underPriceCents, size: 1 }]
            : [],
      };

      const polyMain: NonNullable<TotalGameData['polymarketMain']> = {
        line: rec.candidate.totalLine,
        liquidityUsd: rec.candidate.liquidityUsd,
        markets: {
          over: overBook,
          under: underBook,
        },
      };
      polyTotalByEvent.set(rec.eventTicker, polyMain);
      game.polymarketMain = polyMain;

      polyTokenToEvent.set(rec.candidate.overTokenId, {
        eventTicker: rec.eventTicker,
        side: 'over',
      });
      polyTokenToEvent.set(rec.candidate.underTokenId, {
        eventTicker: rec.eventTicker,
        side: 'under',
      });
      tokenIds.push(rec.candidate.overTokenId, rec.candidate.underTokenId);
    }

    if (tokenIds.length === 0) return;

    polyStream = createPolyMarketStream();
    polyStream.onError((e) => {
      console.warn(`Polymarket total stream error (${sport} ${marketType}):`, e);
      updateLoadingState(loadingState.phase, {
        message: 'Polymarket stream error',
        level: 'error',
      });
      emit();
    });
    polyStream.onBook((snap) => {
      const mapping = polyTokenToEvent.get(snap.assetId);
      if (!mapping) return;
      const polyMain = polyTotalByEvent.get(mapping.eventTicker);
      if (!polyMain) return;

      const book = mapping.side === 'over' ? polyMain.markets.over : polyMain.markets.under;
      book.yes = polyBidsAsYesLevels(snap.bids);
      book.no = polyAsksAsNoLevels(snap.asks);
      book.tsMs = snap.tsMs;
      const game = gamesByEvent.get(mapping.eventTicker);
      if (game && game.marketType === 'total') {
        game.polymarketMain = polyMain;
      }
      lastUpdateMs = Date.now();
      emit();
    });

    await polyStream.connect();
    polyStream.subscribe(Array.from(new Set(tokenIds)));
  };

  const emit = () => {
    const games = Array.from(gamesByEvent.values());
    // Sort by date, then by team codes
    games.sort((a, b) => {
      const dateComp = (a.date || '').localeCompare(b.date || '');
      if (dateComp !== 0) return dateComp;
      const awayComp = a.awayCode.localeCompare(b.awayCode);
      if (awayComp !== 0) return awayComp;
      return a.homeCode.localeCompare(b.homeCode);
    });

    const payload: SportsStreamUpdate = {
      sport,
      marketType,
      games,
      lastUpdateMs,
      loadingState: {
        ...loadingState,
        events: [...loadingState.events],
      },
    };

    callbacks.forEach((cb) => {
      try {
        cb(payload);
      } catch (err) {
        console.error('Error in sports stream callback:', err);
      }
    });
  };

  const rebuildGames = () => {
    const grouped = groupMarketsByEvent(discoveredMarkets);

    gamesByEvent.clear();
    for (const [eventTicker, event] of grouped.entries()) {
      const titleNames = eventTeamNames.get(eventTicker);
      const eventForBuild: GroupedEvent = titleNames
        ? {
            ...event,
            awayName: titleNames.awayName,
            homeName: titleNames.homeName,
          }
        : event;

      // Build market book map for this event
      const eventBooks = new Map<string, MarketBook>();
      for (const [outcomeKey, dm] of eventForBuild.markets.entries()) {
        const book = marketBooks.get(dm.marketTicker);
        if (book) {
          if (marketType === 'moneyline') {
            // Moneyline lookup can happen by parsed outcome code (away/home) or raw market ticker.
            eventBooks.set(outcomeKey, book);
            eventBooks.set(dm.marketTicker, book);
          } else {
            eventBooks.set(dm.marketTicker, book);
          }
        }
      }

      // Build appropriate game data based on market type
      let gameData: GameData;
      if (marketType === 'moneyline') {
        gameData = buildMoneylineGame(eventForBuild, eventBooks);
        const poly = polyMoneylineByEvent.get(eventTicker);
        if (poly) {
          gameData.polymarket = poly;
        }
      } else if (marketType === 'spread') {
        gameData = fallbackEventScaffoldOnly
          ? {
              marketType: 'spread',
              eventTicker: eventForBuild.eventTicker,
              date: eventForBuild.dateYyyyMmDd,
              awayCode: eventForBuild.awayCode,
              homeCode: eventForBuild.homeCode,
              awayName: eventForBuild.awayName,
              homeName: eventForBuild.homeName,
              startTimePt: null,
              markets: new Map(),
              polymarketMain: null,
            }
          : buildSpreadGame(eventForBuild, eventBooks);
        const polyMain = polySpreadByEvent.get(eventTicker);
        if (polyMain) {
          gameData.polymarketMain = polyMain;
        }
      } else {
        gameData = fallbackEventScaffoldOnly
          ? {
              marketType: 'total',
              eventTicker: eventForBuild.eventTicker,
              date: eventForBuild.dateYyyyMmDd,
              awayCode: eventForBuild.awayCode,
              homeCode: eventForBuild.homeCode,
              awayName: eventForBuild.awayName,
              homeName: eventForBuild.homeName,
              startTimePt: null,
              markets: new Map(),
              polymarketMain: null,
            }
          : buildTotalGame(eventForBuild, eventBooks);
        const polyMain = polyTotalByEvent.get(eventTicker);
        if (polyMain) {
          gameData.polymarketMain = polyMain;
        }
      }

      // Apply stored start time if available
      const storedStartTime = eventStartTimes.get(eventTicker);
      if (storedStartTime !== undefined) {
        gameData.startTimePt = storedStartTime;
      }

      // Apply team names from event title (overrides discovery-based names)
      if (titleNames) {
        gameData.awayName = titleNames.awayName;
        gameData.homeName = titleNames.homeName;
      }

      gamesByEvent.set(eventTicker, gameData);
    }
  };

  const initialize = async () => {
    gamesByEvent.clear();
    marketBooks.clear();
    polyMoneylineByEvent.clear();
    polySpreadByEvent.clear();
    polyTotalByEvent.clear();
    eventStartTimes.clear();
    eventTeamNames.clear();
    tickers = [];
    lastUpdateMs = Date.now();
    loadingState = {
      isLoading: true,
      phase: 'discovering-markets',
      startedAtMs: lastUpdateMs,
      updatedAtMs: lastUpdateMs,
      events: [],
    };
    updateLoadingState('discovering-markets', {
      message: `Discovering ${sport.toUpperCase()} ${marketType} markets`,
    });
    emit();

    if (useMock || !api) {
      updateLoadingState('live', { message: 'Using mock/no API mode' });
      emit();
      return;
    }

    // Discover markets for this sport/market type
    discoveredMarkets = await discoverSportsMarkets(api, {
      sport,
      marketType,
      daysAhead,
      limit: 200,
    });

    fallbackEventScaffoldOnly = false;
    if (
      discoveredMarkets.length === 0 &&
      (sport === 'tennis-atp' || sport === 'tennis-wta') &&
      (marketType === 'spread' || marketType === 'total')
    ) {
      discoveredMarkets = await discoverSportsMarkets(api, {
        sport,
        marketType: 'moneyline',
        daysAhead,
        limit: 200,
      });
      fallbackEventScaffoldOnly = discoveredMarkets.length > 0;
    }

    if (discoveredMarkets.length === 0) {
      console.warn(`No ${sport} ${marketType} markets found`);
      updateLoadingState('live', {
        message: `No ${sport.toUpperCase()} ${marketType} markets found`,
        level: 'warn',
      });
      emit();
      return;
    }

    // Collect unique tickers
    tickers = [...new Set(discoveredMarkets.map((m) => m.marketTicker))];

    // Fetch initial orderbooks via REST
    let fetchedOrderbooks = 0;
    updateLoadingState('fetching-orderbooks', {
      message: 'Fetching initial orderbooks',
      done: 0,
      total: tickers.length,
    });
    emit();
    const orderbookPromises = tickers.map(async (ticker) => {
      try {
        const ob = await api.getOrderbook(ticker);
        fetchedOrderbooks += 1;
        updateLoadingState('fetching-orderbooks', {
          done: fetchedOrderbooks,
          total: tickers.length,
        });
        emit();
        return { ticker, ob };
      } catch (err) {
        console.warn(`Failed to fetch orderbook for ${ticker}:`, err);
        fetchedOrderbooks += 1;
        updateLoadingState('fetching-orderbooks', {
          message: `Orderbook fetch failed: ${ticker}`,
          done: fetchedOrderbooks,
          total: tickers.length,
          level: 'warn',
        });
        emit();
        return { ticker, ob: null };
      }
    });

    const orderbooks = await Promise.all(orderbookPromises);

    for (const { ticker, ob } of orderbooks) {
      if (!ob) continue;
      marketBooks.set(ticker, {
        marketTicker: ticker,
        yes: orderbookToLevels(ob.yes),
        no: orderbookToLevels(ob.no),
        tsMs: Date.now(),
      });
    }

    // Fetch event data to get start times
    const grouped = groupMarketsByEvent(discoveredMarkets);
    const eventTickers = Array.from(grouped.keys());

    let fetchedEvents = 0;
    updateLoadingState('fetching-events', {
      message: 'Fetching event metadata',
      done: 0,
      total: eventTickers.length,
    });
    emit();
    const eventPromises = eventTickers.map(async (eventTicker) => {
      try {
        const event = await api.getEvent(eventTicker);
        fetchedEvents += 1;
        updateLoadingState('fetching-events', {
          done: fetchedEvents,
          total: eventTickers.length,
        });
        emit();
        return { eventTicker, event };
      } catch (err) {
        console.warn(`Failed to fetch event ${eventTicker}:`, err);
        fetchedEvents += 1;
        updateLoadingState('fetching-events', {
          message: `Event fetch failed: ${eventTicker}`,
          done: fetchedEvents,
          total: eventTickers.length,
          level: 'warn',
        });
        emit();
        return { eventTicker, event: null };
      }
    });

    const events = await Promise.all(eventPromises);

    for (const { eventTicker, event } of events) {
      const rawStartTime = event?.start_time ?? null;
      let startTimePt: string | null = null;

      if (rawStartTime) {
        try {
          const adjustedStartTime = applyStartTimeOffset(rawStartTime);
          const parts = new Intl.DateTimeFormat('en-US', {
            hour: '2-digit',
            minute: '2-digit',
            hour12: false,
            timeZone: 'America/Los_Angeles',
          }).formatToParts(new Date(adjustedStartTime));
          const hh = parts.find((p) => p.type === 'hour')?.value;
          const mm = parts.find((p) => p.type === 'minute')?.value;
          if (hh && mm) startTimePt = `${hh}:${mm}`;
        } catch {
          startTimePt = null;
        }
      }

      eventStartTimes.set(eventTicker, startTimePt);

      // Parse team names from event title (more reliable than ticker parsing for CBB)
      if (event?.title) {
        const parsedNames = parseTeamNamesFromTitle(event.title);
        if (parsedNames) {
          eventTeamNames.set(eventTicker, parsedNames);
        }
      }
    }

    // Build game structures
    updateLoadingState('building-rows', { message: 'Building game rows' });
    emit();
    rebuildGames();

    lastUpdateMs = Date.now();
    emit();

    // Attach optional Polymarket spreads for supported sports (main line only).
    await hydratePolyMoneylines();
    await hydratePolySpreads();
    await hydratePolyTotals();
    lastUpdateMs = Date.now();
    emit();

    // Connect WebSocket and subscribe to orderbook updates
    updateLoadingState('connecting-stream', { message: 'Connecting live stream' });
    emit();
    marketStream = createMarketStream(useRelay);

    marketStream.onError((error) => {
      console.error(`Sports stream error (${sport} ${marketType}):`, error);
      updateLoadingState(loadingState.phase === 'live' ? 'live' : 'error', {
        message: 'Kalshi stream error',
        level: 'error',
      });
      emit();
    });

    marketStream.onOrderbookUpdate((u) => {
      lastUpdateMs = Date.now();

      // Update the market book
      const existing = marketBooks.get(u.ticker);
      if (existing) {
        existing.yes = u.yes.map((lvl) => ({ priceCents: lvl.price, size: lvl.quantity }));
        existing.no = u.no.map((lvl) => ({ priceCents: lvl.price, size: lvl.quantity }));
        existing.tsMs = lastUpdateMs;
      } else {
        marketBooks.set(u.ticker, {
          marketTicker: u.ticker,
          yes: u.yes.map((lvl) => ({ priceCents: lvl.price, size: lvl.quantity })),
          no: u.no.map((lvl) => ({ priceCents: lvl.price, size: lvl.quantity })),
          tsMs: lastUpdateMs,
        });
      }

      // Rebuild and emit
      rebuildGames();
      emit();
    });

    await marketStream.connect(accessKeyId, privateKey, environment);
    marketStream.subscribeOrderbook(tickers);
    updateLoadingState('live', { message: 'Live stream connected' });
    emit();
  };

  return {
    start: async () => {
      if (marketStream && marketStream.isConnected()) return;
      try {
        await initialize();
      } catch (error) {
        const message = error instanceof Error ? error.message : String(error);
        updateLoadingState('error', { message, level: 'error' });
        emit();
        throw error;
      }
    },

    stop: () => {
      if (marketStream) {
        marketStream.disconnect();
        marketStream = null;
      }
      if (polyStream) {
        polyStream.disconnect();
        polyStream = null;
      }
      gamesByEvent.clear();
      marketBooks.clear();
      eventStartTimes.clear();
      eventTeamNames.clear();
      discoveredMarkets = [];
      fallbackEventScaffoldOnly = false;
      tickers = [];
      lastUpdateMs = 0;
      polyTokenToEvent.clear();
      polyMoneylineByEvent.clear();
      polySpreadByEvent.clear();
      polyTotalByEvent.clear();
      loadingState = {
        isLoading: false,
        phase: 'idle',
        startedAtMs: 0,
        updatedAtMs: 0,
        events: [],
      };
    },

    onUpdate: (cb) => callbacks.add(cb),
    offUpdate: (cb) => callbacks.delete(cb),
  };
}

/**
 * Create multiple sports streams in parallel
 */
export function createMultiSportsStream(
  connectionOptions: StreamConnectionOptions,
  streamOptionsList: SportsStreamOptions[]
): {
  streams: Map<string, SportsStream>;
  startAll: () => Promise<void>;
  stopAll: () => void;
} {
  const streams = new Map<string, SportsStream>();

  for (const opts of streamOptionsList) {
    const key = `${opts.sport}-${opts.marketType}`;
    streams.set(key, createSportsStream(connectionOptions, opts));
  }

  return {
    streams,
    startAll: async () => {
      await Promise.all(Array.from(streams.values()).map((s) => s.start()));
    },
    stopAll: () => {
      streams.forEach((s) => s.stop());
    },
  };
}
