/**
 * Sports Stream Types
 *
 * Generalized types for multi-sport, multi-market-type streaming.
 * Supports moneylines, spreads, and totals across all sports.
 */

import type { Sport, MarketType } from '../sportsDiscovery';

export type Venue = 'kalshi' | 'polymarket';

/** Price level in an orderbook */
export interface BookLevel {
  /** Price in cents, 0..100 */
  priceCents: number;
  /** Size/quantity at that price level */
  size: number;
}

/** Single market orderbook */
export interface MarketBook {
  /** Kalshi market_ticker */
  marketTicker: string;
  /** Polymarket token_id (optional) */
  tokenId?: string;
  /** Polymarket conditionId (optional) */
  conditionId?: string;
  /** Polymarket min tick size (optional) */
  tickSize?: string;
  /** Polymarket negRisk flag (optional) */
  negRisk?: boolean;
  /** YES bids (descending by price) */
  yes: BookLevel[];
  /** NO bids (descending by price) */
  no: BookLevel[];
  /** Last update timestamp (ms) */
  tsMs?: number;
}

/** Spread market info - includes floor strike for proper display */
export interface SpreadMarketInfo {
  book: MarketBook;
  /** Numeric spread value from floor_strike (e.g., 7.5) */
  floorStrike: number;
  /** Team code this spread is for (favored team) */
  teamCode: string;
}

/**
 * Moneyline game data - two outcomes (away/home)
 */
export interface MoneylineGameData {
  marketType: 'moneyline';
  eventTicker: string;
  date: string; // YYYY-MM-DD
  awayCode: string;
  homeCode: string;
  /** Display name for away team/player */
  awayName?: string;
  /** Display name for home team/player */
  homeName?: string;
  startTimePt?: string | null;
  /** Away team market */
  away: MarketBook | null;
  /** Home team market */
  home: MarketBook | null;
  /** Optional Polymarket moneyline market for this game */
  polymarket?: {
    eventId: string;
    liquidityUsd?: number;
    markets: {
      away: MarketBook;
      home: MarketBook;
    };
    tsMs?: number;
  } | null;
}

/**
 * Spread game data - multiple buckets per team
 * Each bucket represents a spread range (e.g., -3.5 to -1.5)
 */
export interface SpreadGameData {
  marketType: 'spread';
  eventTicker: string;
  date: string;
  awayCode: string;
  homeCode: string;
  /** Display name for away team/player */
  awayName?: string;
  /** Display name for home team/player */
  homeName?: string;
  startTimePt?: string | null;
  /**
   * Spread markets keyed by "{teamCode}-{bucket}"
   * e.g., "NYK-7" for NYK bucket 7
   * Value includes floor_strike for accurate spread display
   */
  markets: Map<string, SpreadMarketInfo>;
  /** Optional Polymarket main spread market for this game */
  polymarketMain?: {
    line: number;
    favoriteSide: 'away' | 'home';
    liquidityUsd?: number;
    markets: {
      favorite: MarketBook;
      underdog: MarketBook;
    };
  } | null;
}

/**
 * Total game data - strike values for over/under
 */
export interface TotalGameData {
  marketType: 'total';
  eventTicker: string;
  date: string;
  awayCode: string;
  homeCode: string;
  /** Display name for away team/player */
  awayName?: string;
  /** Display name for home team/player */
  homeName?: string;
  startTimePt?: string | null;
  /**
   * Total markets keyed by display line value (prefers floor_strike precision)
   * YES = Over, NO = Under
   */
  markets: Map<number, MarketBook>;
  /** Optional Polymarket main total market for this game */
  polymarketMain?: {
    line: number;
    liquidityUsd?: number;
    markets: {
      over: MarketBook;
      under: MarketBook;
    };
  } | null;
}

export type GameData = MoneylineGameData | SpreadGameData | TotalGameData;

export type SportsLoadPhase =
  | 'idle'
  | 'discovering-markets'
  | 'fetching-orderbooks'
  | 'fetching-events'
  | 'building-rows'
  | 'hydrating-polymarket'
  | 'connecting-stream'
  | 'live'
  | 'error';

export interface SportsLoadEvent {
  tsMs: number;
  phase: SportsLoadPhase;
  level: 'info' | 'warn' | 'error';
  message: string;
  done?: number;
  total?: number;
}

export interface SportsLoadingState {
  isLoading: boolean;
  phase: SportsLoadPhase;
  startedAtMs: number;
  updatedAtMs: number;
  done?: number;
  total?: number;
  events: SportsLoadEvent[];
}

/** Key parts for game identification */
export interface GameKeyParts {
  sport: Sport;
  marketType: MarketType;
  date: string;
  awayCode: string;
  homeCode: string;
}

export function makeGameKey(parts: GameKeyParts): string {
  return `${parts.sport}|${parts.marketType}|${parts.date}|${parts.awayCode.toUpperCase()}|${parts.homeCode.toUpperCase()}`;
}

/** Parsed game key */
export function parseGameKey(key: string): GameKeyParts | null {
  const parts = key.split('|');
  if (parts.length !== 5) return null;
  return {
    sport: parts[0] as Sport,
    marketType: parts[1] as MarketType,
    date: parts[2] ?? '',
    awayCode: parts[3] ?? '',
    homeCode: parts[4] ?? '',
  };
}

/** Get best bid from book levels */
export function getBestBid(
  book: MarketBook | null,
  side: 'yes' | 'no'
): { priceCents: number | null; size: number } {
  if (!book) return { priceCents: null, size: 0 };
  const levels = side === 'yes' ? book.yes : book.no;
  const bestLevel = levels[0];
  if (!bestLevel) return { priceCents: null, size: 0 };
  return { priceCents: bestLevel.priceCents, size: bestLevel.size };
}

/** Stream update payload */
export interface SportsStreamUpdate {
  sport: Sport;
  marketType: MarketType;
  games: GameData[];
  lastUpdateMs: number;
  loadingState: SportsLoadingState;
}

/** Stream configuration options */
export interface SportsStreamOptions {
  sport: Sport;
  marketType: MarketType;
  /** Number of days ahead to include (default: 7) */
  daysAhead?: number;
}

/**
 * Convert MoneylineGameData to ConsolidatedGameBooks format.
 * This allows using the same table component for all sports.
 */
export function moneylineToConsolidated(game: MoneylineGameData): {
  key: string;
  date: string;
  awayCode: string;
  homeCode: string;
  awayName?: string;
  homeName?: string;
  startTimePt?: string | null;
  kalshi: {
    venue: 'kalshi';
    eventId: string;
    awayCode: string;
    homeCode: string;
    markets: {
      away: {
        marketTicker: string;
        yes: BookLevel[];
        no: BookLevel[];
      };
      home: {
        marketTicker: string;
        yes: BookLevel[];
        no: BookLevel[];
      };
    };
    tsMs?: number;
  } | null;
  polymarket: {
    venue: 'polymarket';
    eventId: string;
    awayCode: string;
    homeCode: string;
    markets: {
      away: {
        marketTicker: string;
        tokenId?: string;
        conditionId?: string;
        tickSize?: string;
        negRisk?: boolean;
        yes: BookLevel[];
        no: BookLevel[];
      };
      home: {
        marketTicker: string;
        tokenId?: string;
        conditionId?: string;
        tickSize?: string;
        negRisk?: boolean;
        yes: BookLevel[];
        no: BookLevel[];
      };
    };
    tsMs?: number;
  } | null;
} {
  const key = `${game.date}|${game.awayCode.toUpperCase()}|${game.homeCode.toUpperCase()}`;

  // Convert to Kalshi venue format if we have data
  const kalshi =
    game.away || game.home
      ? {
          venue: 'kalshi' as const,
          eventId: game.eventTicker,
          awayCode: game.awayCode,
          homeCode: game.homeCode,
          markets: {
            away: {
              marketTicker: game.away?.marketTicker ?? '',
              yes: game.away?.yes ?? [],
              no: game.away?.no ?? [],
            },
            home: {
              marketTicker: game.home?.marketTicker ?? '',
              yes: game.home?.yes ?? [],
              no: game.home?.no ?? [],
            },
          },
          tsMs: game.away?.tsMs ?? game.home?.tsMs,
        }
      : null;

  const polymarket = game.polymarket
    ? {
        venue: 'polymarket' as const,
        eventId: game.polymarket.eventId,
        awayCode: game.awayCode,
        homeCode: game.homeCode,
        markets: {
          away: {
            marketTicker: game.polymarket.markets.away.marketTicker,
            tokenId: game.polymarket.markets.away.tokenId,
            conditionId: game.polymarket.markets.away.conditionId,
            tickSize: game.polymarket.markets.away.tickSize,
            negRisk: game.polymarket.markets.away.negRisk,
            yes: game.polymarket.markets.away.yes,
            no: game.polymarket.markets.away.no,
          },
          home: {
            marketTicker: game.polymarket.markets.home.marketTicker,
            tokenId: game.polymarket.markets.home.tokenId,
            conditionId: game.polymarket.markets.home.conditionId,
            tickSize: game.polymarket.markets.home.tickSize,
            negRisk: game.polymarket.markets.home.negRisk,
            yes: game.polymarket.markets.home.yes,
            no: game.polymarket.markets.home.no,
          },
        },
        tsMs: game.polymarket.tsMs,
      }
    : null;

  return {
    key,
    date: game.date,
    awayCode: game.awayCode,
    homeCode: game.homeCode,
    awayName: game.awayName,
    homeName: game.homeName,
    startTimePt: game.startTimePt,
    kalshi,
    polymarket,
  };
}
