/**
 * User Stream (Orders + Fills)
 *
 * Connects to Kalshi WebSocket (via relay) and subscribes to user-specific channels.
 * Goal: drive near-realtime Orders + Positions without aggressive polling.
 */

import { createRelayWs, type RelayWsClient } from './relayWs';
import { buildAuthHeaders, getBaseUrl } from './kalshiAuth';
import { normalizeKalshiPriceToProbability } from './price';
import {
  asRecord,
  pickFirst,
  toNumber,
  toString,
  toSide,
  toAction,
  normalizeTimestampMsWithFallback as normalizeTimestampMs,
  type UnknownRecord,
} from './typeCoercers';
import type { Environment, FillNotification } from '../types';

export interface UserStream {
  connect(accessKeyId: string, privateKey: CryptoKey, environment: Environment): Promise<void>;
  subscribe(): void;
  onFill(callback: (fill: FillNotification) => void): void;
  onOrderEvent(callback: (event: unknown) => void): void;
  onError(callback: (error: Error) => void): void;
  disconnect(): void;
  isConnected(): boolean;
}

// Kalshi WS path per docs (Jan 2026)
const KALSHI_WS_PATH = '/trade-api/ws/v2';

export function createUserStream(useRelay: boolean = true): UserStream {
  let relayClient: RelayWsClient | null = null;
  let streamId: string | null = null;
  let connected = false;
  let upstreamConnected = false;
  let upstreamConnectPromise: Promise<void> | null = null;
  let resolveUpstreamConnect: (() => void) | null = null;
  let rejectUpstreamConnect: ((err: Error) => void) | null = null;

  const fillCallbacks = new Set<(fill: FillNotification) => void>();
  const orderCallbacks = new Set<(event: unknown) => void>();
  const errorCallbacks = new Set<(error: Error) => void>();

  const getKalshiWsUrl = (environment: Environment): string => {
    const baseUrl = getBaseUrl(environment);
    return baseUrl.replace('https://', 'wss://').replace('http://', 'ws://') + KALSHI_WS_PATH;
  };

  const buildWsAuthHeaders = async (
    accessKey: string,
    key: CryptoKey
  ): Promise<Record<string, string>> => {
    return buildAuthHeaders(accessKey, key, 'GET', KALSHI_WS_PATH, '', '');
  };

  const emitFill = (fill: FillNotification) => {
    fillCallbacks.forEach((cb) => cb(fill));
  };

  const emitOrderEvent = (event: unknown) => {
    orderCallbacks.forEach((cb) => cb(event));
  };

  const emitError = (error: Error) => {
    errorCallbacks.forEach((cb) => cb(error));
  };

  const tryParseFill = (msg: UnknownRecord): FillNotification | null => {
    const payload = asRecord(pickFirst(msg.msg, msg.data, msg.payload, msg.fill, msg)) ?? msg;

    const ticker =
      toString(
        pickFirst(payload.market_ticker, payload.ticker, payload.marketTicker, payload.symbol)
      ) ?? null;
    const side = toSide(pickFirst(payload.side, payload.position, payload.contract));
    const action = toAction(pickFirst(payload.action, payload.direction));
    const qty = toNumber(pickFirst(payload.count, payload.quantity, payload.size));
    const rawPrice = toNumber(pickFirst(payload.price, payload.yes_price, payload.yesPrice));
    const orderId =
      toString(pickFirst(payload.order_id, payload.orderId)) ??
      toString(pickFirst(payload.fill_id, payload.fillId, payload.trade_id, payload.tradeId)) ??
      `fill-${Date.now()}-${Math.random().toString(36).slice(2)}`;

    const price = normalizeKalshiPriceToProbability(rawPrice);
    if (!ticker || !side || !action || qty === null || price === null) return null;

    return {
      ticker,
      side,
      action,
      quantity: qty,
      price,
      timestamp: normalizeTimestampMs(
        pickFirst(
          payload.ts,
          payload.timestamp,
          payload.time,
          payload.created_at,
          payload.createdAt
        )
      ),
      orderId,
    };
  };

  const handleKalshiMessage = (raw: unknown) => {
    const msg = typeof raw === 'string' ? (JSON.parse(raw) as unknown) : raw;
    const rec = asRecord(msg);
    if (!rec) return;

    const type =
      (typeof rec.type === 'string' ? rec.type : undefined) ??
      (typeof rec.channel === 'string' ? rec.channel : undefined) ??
      (typeof rec.event === 'string' ? rec.event : undefined) ??
      undefined;

    // Fill channel (user-specific)
    if (type === 'fill' || type === 'order_fill' || type === 'fills') {
      const fill = tryParseFill(rec);
      if (fill) emitFill(fill);
      return;
    }

    // Position updates (authenticated).
    if (type === 'market_position' || type === 'market_positions') {
      emitOrderEvent(msg);
      return;
    }

    // Order-like events (best-effort; Kalshi doesn't document an "order" channel in v2 WS).
    if (
      type === 'order' ||
      type === 'orders' ||
      type === 'order_confirmation' ||
      type === 'order_failure'
    ) {
      emitOrderEvent(msg);
      return;
    }
  };

  const handleRelayMessage = (frame: {
    id: string;
    type: string;
    data?: unknown;
    error?: string;
  }) => {
    if (frame.type === 'error') {
      const err = new Error(frame.error || 'WebSocket error');
      emitError(err);
      if (!upstreamConnected && rejectUpstreamConnect) {
        rejectUpstreamConnect(err);
        rejectUpstreamConnect = null;
        resolveUpstreamConnect = null;
        upstreamConnectPromise = null;
      }
      return;
    }

    if (frame.type === 'message' && frame.data !== undefined) {
      // Relay "connected" ack when upstream WS opens
      if (
        typeof frame.data === 'object' &&
        frame.data &&
        (frame.data as Record<string, unknown>).connected === true
      ) {
        upstreamConnected = true;
        if (resolveUpstreamConnect) {
          resolveUpstreamConnect();
          resolveUpstreamConnect = null;
          rejectUpstreamConnect = null;
          upstreamConnectPromise = null;
        }
        return;
      }
      handleKalshiMessage(frame.data);
    }

    if (frame.type === 'close') {
      if (!upstreamConnected && rejectUpstreamConnect) {
        rejectUpstreamConnect(new Error('Upstream WebSocket closed during connect'));
        rejectUpstreamConnect = null;
        resolveUpstreamConnect = null;
        upstreamConnectPromise = null;
      }
    }
  };

  return {
    async connect(accessKeyId: string, privateKey: CryptoKey, environment: Environment) {
      if (!useRelay) {
        throw new Error(
          'UserStream currently requires relay (browser WebSocket cannot set auth headers).'
        );
      }

      if (connected) return;

      relayClient = createRelayWs();
      relayClient.onMessage(handleRelayMessage);
      relayClient.onError((e) => emitError(e));

      const headers = await buildWsAuthHeaders(accessKeyId, privateKey);
      streamId = `user-stream-${Date.now()}`;
      const wsUrl = getKalshiWsUrl(environment);

      upstreamConnected = false;
      upstreamConnectPromise = new Promise<void>((resolve, reject) => {
        resolveUpstreamConnect = resolve;
        rejectUpstreamConnect = reject;
        setTimeout(() => {
          if (!upstreamConnected) {
            reject(new Error('Timed out waiting for upstream WebSocket to connect'));
            resolveUpstreamConnect = null;
            rejectUpstreamConnect = null;
            upstreamConnectPromise = null;
          }
        }, 10000);
      });

      await relayClient.connect(streamId, wsUrl, headers);
      await upstreamConnectPromise;
      connected = true;
    },

    subscribe() {
      if (!relayClient || !streamId) return;

      // Use a hybrid subscribe payload to maximize compatibility with differing docs/examples.
      // - Some docs show `cmd: 'subscribe'`
      // - Some show `type: 'subscribe'`
      // Kalshi typically ignores unknown keys; relay forwards as-is.
      const payload = {
        id: 1,
        cmd: 'subscribe',
        type: 'subscribe',
        params: {
          // Kalshi WS channels (docs Jan 2026):
          // - `fill` for user fills
          // - `market_positions` for position changes (authenticated)
          channels: ['fill', 'market_positions'],
        },
        channels: ['fill', 'market_positions'],
      };

      relayClient.subscribe(streamId, payload);
    },

    onFill(callback: (fill: FillNotification) => void) {
      fillCallbacks.add(callback);
    },

    onOrderEvent(callback: (event: unknown) => void) {
      orderCallbacks.add(callback);
    },

    onError(callback: (error: Error) => void) {
      errorCallbacks.add(callback);
    },

    disconnect() {
      if (relayClient && streamId) {
        relayClient.close(streamId);
        relayClient.disconnect();
      }
      relayClient = null;
      streamId = null;
      connected = false;
      upstreamConnected = false;
      upstreamConnectPromise = null;
      resolveUpstreamConnect = null;
      rejectUpstreamConnect = null;
      fillCallbacks.clear();
      orderCallbacks.clear();
      errorCallbacks.clear();
    },

    isConnected() {
      return connected;
    },
  };
}
