import path from 'path';
import { execSync } from 'child_process';
import { defineConfig, loadEnv } from 'vite';
import react from '@vitejs/plugin-react';
import { nodePolyfills } from 'vite-plugin-node-polyfills';

function getGitCommitHash(): string {
  try {
    return execSync('git rev-parse --short HEAD', { encoding: 'utf8' }).trim();
  } catch {
    return 'unknown';
  }
}

export default defineConfig(({ mode }) => {
  // Skip node polyfills in test to avoid node-stdlib-browser ESM punycode resolution error.
  const isTest = process.env.VITEST === 'true';
  const env = loadEnv(mode, __dirname, '');
  const routerBase = env.VITE_ROUTER_BASE?.trim();
  const base = routerBase ? `/${routerBase.replace(/^\/|\/$/g, '')}/` : './';

  return {
    base,
    plugins: [
      react(),
      ...(isTest
        ? []
        : [
            // Needed for browser usage of Polymarket/ethers stack (Buffer/process).
            nodePolyfills({ protocolImports: true }),
          ]),
    ],
    resolve: {
      alias: {
        '@': path.resolve(__dirname, './src'),
        ...(isTest
          ? {}
          : {
              buffer: 'buffer',
              process: 'process/browser',
            }),
      },
    },
    define: {
      global: 'globalThis',
      __BUILD_COMMIT__: JSON.stringify(getGitCommitHash()),
      __BUILD_TIME__: JSON.stringify(new Date().toISOString()),
    },
    // Avoid writing Vite temp/cache files into node_modules (can be read-only depending on how deps were installed).
    cacheDir: '.vite',
    server: {
      port: 5173,
      strictPort: true,
      open: true,
      proxy: {
        '/relay': {
          target: 'http://localhost:8787',
          changeOrigin: true,
        },
      },
    },
    build: {
      outDir: 'dist',
      sourcemap: true,
      rollupOptions: {
        output: {
          manualChunks: {
            // Heavy crypto/blockchain deps — only needed when Polymarket is active
            'vendor-polymarket': ['ethers', '@polymarket/clob-client'],
            // Charting library — only needed on chart views
            'vendor-charts': ['lightweight-charts'],
            // RSA crypto — loaded after auth
            'vendor-crypto': ['node-forge'],
          },
        },
      },
    },
  };
});
