/**
 * Glossary Tooltip
 * Hover tooltip that shows a definition from the glossary
 */

import { useState } from 'react';
import { cn } from '@/lib/utils';

interface GlossaryTerm {
  term: string;
  definition: string;
  category?: string;
}

interface GlossaryTooltipProps {
  term: GlossaryTerm;
  children: React.ReactNode;
  position?: 'top' | 'bottom' | 'left' | 'right';
  className?: string;
}

export function GlossaryTooltip({
  term,
  children,
  position = 'top',
  className,
}: GlossaryTooltipProps) {
  const [isVisible, setIsVisible] = useState(false);

  const positionClasses: Record<string, string> = {
    top: 'bottom-full left-1/2 -translate-x-1/2 mb-2',
    bottom: 'top-full left-1/2 -translate-x-1/2 mt-2',
    left: 'right-full top-1/2 -translate-y-1/2 mr-2',
    right: 'left-full top-1/2 -translate-y-1/2 ml-2',
  };

  const arrowClasses: Record<string, string> = {
    top: 'top-full left-1/2 -translate-x-1/2 border-t-slate-800 border-x-transparent border-b-transparent',
    bottom:
      'bottom-full left-1/2 -translate-x-1/2 border-b-slate-800 border-x-transparent border-t-transparent',
    left: 'left-full top-1/2 -translate-y-1/2 border-l-slate-800 border-y-transparent border-r-transparent',
    right:
      'right-full top-1/2 -translate-y-1/2 border-r-slate-800 border-y-transparent border-l-transparent',
  };

  return (
    <span
      className={cn('relative inline-block', className)}
      onMouseEnter={() => setIsVisible(true)}
      onMouseLeave={() => setIsVisible(false)}
    >
      <span className="cursor-help border-b border-dashed border-slate-500">{children}</span>

      {isVisible && (
        <div
          className={cn(
            'absolute z-50 w-64 rounded-lg border border-slate-700 bg-slate-800 p-3 shadow-xl',
            'animate-in fade-in duration-150',
            positionClasses[position]
          )}
        >
          {/* Arrow */}
          <div className={cn('absolute h-0 w-0 border-4', arrowClasses[position])} />

          {/* Content */}
          <div className="space-y-1">
            <div className="flex items-center justify-between">
              <span className="text-sm font-medium text-white">{term.term}</span>
              {term.category && (
                <span className="rounded bg-slate-700 px-1.5 py-0.5 text-[10px] text-slate-400">
                  {term.category}
                </span>
              )}
            </div>
            <p className="text-xs leading-relaxed text-slate-300">{term.definition}</p>
          </div>
        </div>
      )}
    </span>
  );
}

// Common glossary terms for demo
export const glossaryTerms: Record<string, GlossaryTerm> = {
  yes: {
    term: 'YES',
    definition:
      'A contract that pays $1.00 if the event happens. Price represents market probability.',
    category: 'Contracts',
  },
  no: {
    term: 'NO',
    definition: 'A contract that pays $1.00 if the event does NOT happen. Price = 100 - YES price.',
    category: 'Contracts',
  },
  bid: {
    term: 'Bid',
    definition: 'The highest price a buyer is willing to pay. Also called "best buy" price.',
    category: 'Order Book',
  },
  ask: {
    term: 'Ask',
    definition:
      'The lowest price a seller is willing to accept. Also called "best sell" or "offer" price.',
    category: 'Order Book',
  },
  spread: {
    term: 'Spread',
    definition:
      'The difference between the bid and ask price. Tighter spreads indicate more liquidity.',
    category: 'Order Book',
  },
  liquidity: {
    term: 'Liquidity',
    definition:
      'How easily you can buy/sell without moving the price. More orders = more liquidity.',
    category: 'Market',
  },
  vig: {
    term: 'Vig (Vigorish)',
    definition:
      'The fee or edge built into odds. Also called "juice". The platform\'s cut on trades.',
    category: 'Fees',
  },
  ev: {
    term: 'EV (Expected Value)',
    definition:
      'The average profit/loss per bet over time. Positive EV (+EV) means profitable long-term.',
    category: 'Strategy',
  },
  moneyline: {
    term: 'Moneyline',
    definition: 'A bet on which team/side will win outright, without a point spread.',
    category: 'Bet Types',
  },
  limit: {
    term: 'Limit Order',
    definition: 'An order to buy/sell at a specific price or better. May not fill immediately.',
    category: 'Orders',
  },
  market: {
    term: 'Market Order',
    definition:
      'An order that fills immediately at the best available price. Guaranteed fill, not price.',
    category: 'Orders',
  },
  maker: {
    term: 'Maker',
    definition: 'Someone who adds liquidity by placing limit orders that rest on the book.',
    category: 'Fees',
  },
  taker: {
    term: 'Taker',
    definition:
      'Someone who removes liquidity by filling existing orders. Usually pays higher fees.',
    category: 'Fees',
  },
  pnl: {
    term: 'P&L (Profit & Loss)',
    definition:
      'Your profit or loss from trades. Can be realized (closed) or unrealized (open positions).',
    category: 'Portfolio',
  },
};

export type { GlossaryTerm };
