import { ReactNode } from 'react';

interface LessonLayoutProps {
  children: ReactNode;
}

/**
 * Container for lesson content.
 * Provides consistent max-width and spacing.
 * Use LessonRow for content + aside pairs.
 */
export function LessonLayout({ children }: LessonLayoutProps) {
  return <div className="space-y-8">{children}</div>;
}

interface LessonRowProps {
  children: ReactNode;
  aside?: ReactNode;
}

/**
 * A row in a lesson with main content and optional aside.
 * Main content is ALWAYS the same fixed width.
 * Aside appears to the right on large screens, below on small screens.
 */
export function LessonRow({ children, aside }: LessonRowProps) {
  return (
    <div className="flex flex-col items-start gap-8 lg:flex-row lg:gap-16">
      {/* Main content - always fixed width */}
      <div className="w-full flex-shrink-0 space-y-6 lg:w-[700px]">{children}</div>

      {/* Aside - only rendered if provided */}
      {aside && <aside className="w-full flex-shrink-0 space-y-4 lg:w-64">{aside}</aside>}
    </div>
  );
}

interface ExpandableRowProps {
  children: ReactNode;
  aside?: ReactNode;
  isExpanded: boolean;
}

/**
 * An expandable content row where the aside only appears when expanded.
 * The main content maintains its width; the aside pops out to the right.
 * Use this inside expandable cards/accordions.
 */
export function ExpandableRow({ children, aside, isExpanded }: ExpandableRowProps) {
  return (
    <div className="flex flex-col items-start gap-6 lg:flex-row lg:gap-16">
      {/* Main content - fixed width */}
      <div className="min-w-0 flex-1 space-y-4">{children}</div>

      {/* Aside - only shows when expanded */}
      {isExpanded && aside && (
        <aside className="w-full flex-shrink-0 space-y-4 lg:w-56">{aside}</aside>
      )}
    </div>
  );
}

interface LessonHeaderProps {
  title: string;
  description: string;
  duration?: string;
  category?: string;
}

/**
 * Header for a lesson page with title, description, and metadata.
 */
export function LessonHeader({ title, description, duration, category }: LessonHeaderProps) {
  return (
    <div className="space-y-2">
      <div className="flex items-center gap-3">
        {category && (
          <span className="bg-primary/10 text-primary rounded px-2 py-0.5 text-xs font-medium">
            {category}
          </span>
        )}
        {duration && <span className="text-muted-foreground text-xs">{duration}</span>}
      </div>
      <h1 className="text-2xl font-bold">{title}</h1>
      <p className="text-muted-foreground">{description}</p>
    </div>
  );
}

interface SectionHeaderProps {
  title: string;
  subtitle?: string;
}

/**
 * Section header within a lesson.
 */
export function SectionHeader({ title, subtitle }: SectionHeaderProps) {
  return (
    <div className="space-y-1">
      <h2 className="text-xl font-semibold">{title}</h2>
      {subtitle && <p className="text-muted-foreground text-sm">{subtitle}</p>}
    </div>
  );
}
