/**
 * Lesson: Kalshi API Entity Model
 *
 * A comprehensive guide to the Kalshi API's domain model for developers.
 * Assumes API experience but no Kalshi-specific knowledge.
 */

import { useState } from 'react';
import { getGuide, getNextGuide } from '@/data/guides';
import { LessonLayout, LessonRow, LessonHeader, SectionHeader } from '../LessonLayout';
import {
  ObjectiveBlock,
  ConstraintBlock,
  TipBlock,
  InsightBlock,
  WarningBlock,
  ConceptBlock,
  NextStepBlock,
  Term,
} from '../blocks';
import { GlossaryTooltip } from '@/components/atoms/GlossaryTooltip';

// Custom glossary terms for this lesson
const terms = {
  event: {
    term: 'Event',
    definition:
      'A real-world occurrence that can be traded on. Contains one or more markets representing different outcomes or thresholds.',
    category: 'API Entities' as const,
  },
  market: {
    term: 'Market',
    definition:
      'A specific binary contract within an event. Has a YES and NO side, settles to $1 or $0.',
    category: 'API Entities' as const,
  },
  series: {
    term: 'Series',
    definition:
      'A template for recurring events. Defines the structure, rules, and settlement sources applied to each instance.',
    category: 'API Entities' as const,
  },
  ticker: {
    term: 'Ticker',
    definition:
      'A unique string identifier for an event or market. Follows a hierarchical naming convention.',
    category: 'API Entities' as const,
  },
  position: {
    term: 'Position',
    definition:
      'Your current holdings in a market. Positive means you own YES contracts, negative means you own NO contracts.',
    category: 'API Entities' as const,
  },
  fill: {
    term: 'Fill',
    definition: 'A completed trade. Created when your order matches with a counterparty.',
    category: 'API Entities' as const,
  },
  settlement: {
    term: 'Settlement',
    definition:
      'The final resolution of a market. YES contracts pay $1 if the outcome occurred, $0 otherwise.',
    category: 'API Entities' as const,
  },
};

// Entity property type
type EntityProperty = { name: string; type: string; desc: string };

// Entity type
type Entity = {
  id: string;
  name: string;
  description: string;
  color: string;
  x: number;
  y: number;
  relations: string[];
  properties: EntityProperty[];
};

// Entity data with positions for SVG diagram
const entityData: Record<string, Entity> = {
  series: {
    id: 'series',
    name: 'Series',
    description: 'Template for recurring events',
    color: '#a855f7', // purple
    x: 270,
    y: 20,
    relations: ['event'],
    properties: [
      { name: 'ticker', type: 'string', desc: 'Unique series identifier (e.g., "INXD")' },
      { name: 'title', type: 'string', desc: 'Human-readable name' },
      { name: 'category', type: 'string', desc: 'Classification (economics, politics, sports)' },
      { name: 'frequency', type: 'string', desc: 'How often new events are created' },
      { name: 'settlement_sources', type: 'array', desc: 'Data sources for outcome verification' },
    ],
  },
  event: {
    id: 'event',
    name: 'Event',
    description: 'Real-world occurrence being traded',
    color: '#3b82f6', // blue
    x: 270,
    y: 90,
    relations: ['market'],
    properties: [
      { name: 'event_ticker', type: 'string', desc: 'Unique identifier (e.g., "INXD-24DEC31")' },
      { name: 'series_ticker', type: 'string', desc: 'Parent series reference' },
      { name: 'title', type: 'string', desc: 'Human-readable description' },
      { name: 'strike_date', type: 'datetime', desc: 'When outcome is determined' },
      { name: 'markets', type: 'Market[]', desc: 'Array of markets within this event' },
      {
        name: 'mutually_exclusive',
        type: 'boolean',
        desc: 'If true, exactly one market settles YES',
      },
    ],
  },
  market: {
    id: 'market',
    name: 'Market',
    description: 'Binary contract (YES/NO)',
    color: '#22c55e', // green
    x: 270,
    y: 160,
    relations: ['order', 'position', 'orderbook', 'trade'],
    properties: [
      { name: 'ticker', type: 'string', desc: 'Unique market identifier' },
      { name: 'event_ticker', type: 'string', desc: 'Parent event reference' },
      { name: 'status', type: 'enum', desc: 'unopened | open | closed | settled' },
      { name: 'yes_bid / yes_ask', type: 'number', desc: 'Best bid and ask prices (1-99)' },
      { name: 'last_price', type: 'number', desc: 'Most recent trade price' },
      { name: 'volume', type: 'number', desc: 'Total contracts traded' },
      { name: 'open_interest', type: 'number', desc: 'Outstanding unsettled contracts' },
      { name: 'result', type: 'string?', desc: '"yes" | "no" | null (if not settled)' },
    ],
  },
  order: {
    id: 'order',
    name: 'Order',
    description: 'Your intent to buy/sell',
    color: '#f59e0b', // amber
    x: 70,
    y: 250,
    relations: ['fill'],
    properties: [
      { name: 'order_id', type: 'string', desc: 'Unique order identifier (UUID)' },
      { name: 'client_order_id', type: 'string', desc: 'Your custom identifier' },
      { name: 'ticker', type: 'string', desc: 'Market this order is for' },
      { name: 'side', type: 'enum', desc: '"yes" | "no"' },
      { name: 'action', type: 'enum', desc: '"buy" | "sell"' },
      { name: 'type', type: 'enum', desc: '"limit" | "market"' },
      { name: 'count', type: 'number', desc: 'Total contracts in order' },
      { name: 'remaining_count', type: 'number', desc: 'Contracts not yet filled' },
      { name: 'yes_price', type: 'number', desc: 'Limit price in cents (1-99)' },
      { name: 'status', type: 'enum', desc: 'resting | executed | canceled' },
    ],
  },
  fill: {
    id: 'fill',
    name: 'Fill',
    description: 'Completed transaction',
    color: '#ef4444', // red
    x: 70,
    y: 320,
    relations: [],
    properties: [
      { name: 'trade_id', type: 'string', desc: 'Unique fill identifier' },
      { name: 'order_id', type: 'string', desc: 'Order that generated this fill' },
      { name: 'ticker', type: 'string', desc: 'Market traded' },
      { name: 'side', type: 'enum', desc: '"yes" | "no"' },
      { name: 'action', type: 'enum', desc: '"buy" | "sell"' },
      { name: 'count', type: 'number', desc: 'Number of contracts filled' },
      { name: 'yes_price', type: 'number', desc: 'Execution price' },
      { name: 'created_time', type: 'datetime', desc: 'When the fill occurred' },
      { name: 'is_taker', type: 'boolean', desc: 'True if you crossed the spread' },
    ],
  },
  position: {
    id: 'position',
    name: 'Position',
    description: 'Your holdings in a market',
    color: '#06b6d4', // cyan
    x: 200,
    y: 250,
    relations: [],
    properties: [
      { name: 'ticker', type: 'string', desc: 'Market identifier' },
      { name: 'position', type: 'number', desc: 'Contracts held (+ = YES, - = NO)' },
      { name: 'market_exposure', type: 'number', desc: 'Dollar value at risk' },
      { name: 'realized_pnl', type: 'number', desc: 'Profit/loss from closed trades' },
      { name: 'total_traded', type: 'number', desc: 'Cumulative contracts traded' },
      { name: 'resting_orders_count', type: 'number', desc: 'Open orders in this market' },
    ],
  },
  orderbook: {
    id: 'orderbook',
    name: 'Orderbook',
    description: 'All resting orders in a market',
    color: '#6366f1', // indigo
    x: 340,
    y: 250,
    relations: [],
    properties: [
      { name: 'yes', type: 'array', desc: 'YES side bids: [[price, quantity], ...]' },
      { name: 'no', type: 'array', desc: 'NO side bids: [[price, quantity], ...]' },
    ],
  },
  trade: {
    id: 'trade',
    name: 'Trade',
    description: 'Public record of a matched trade',
    color: '#ec4899', // pink
    x: 470,
    y: 250,
    relations: [],
    properties: [
      { name: 'trade_id', type: 'string', desc: 'Unique trade identifier' },
      { name: 'ticker', type: 'string', desc: 'Market where trade occurred' },
      { name: 'count', type: 'number', desc: 'Number of contracts traded' },
      { name: 'yes_price', type: 'number', desc: 'Execution price (YES side)' },
      { name: 'taker_side', type: 'enum', desc: '"yes" | "no" — who crossed the spread' },
      { name: 'created_time', type: 'datetime', desc: 'Trade timestamp' },
    ],
  },
};

// Relationship lines between entities (using UML notation: 1 = one, * = many)
const relationships: Array<{ from: string; to: string; fromLabel: string; toLabel: string }> = [
  { from: 'series', to: 'event', fromLabel: '1', toLabel: '*' },
  { from: 'event', to: 'market', fromLabel: '1', toLabel: '*' },
  { from: 'market', to: 'order', fromLabel: '1', toLabel: '*' },
  { from: 'market', to: 'position', fromLabel: '1', toLabel: '1' },
  { from: 'market', to: 'orderbook', fromLabel: '1', toLabel: '1' },
  { from: 'market', to: 'trade', fromLabel: '1', toLabel: '*' },
  { from: 'order', to: 'fill', fromLabel: '1', toLabel: '*' },
];

// Interactive entity relationship diagram with SVG
function EntityRelationshipWidget() {
  const [selectedEntity, setSelectedEntity] = useState<string | null>(null);

  const entities = Object.values(entityData);
  const selected = selectedEntity ? entityData[selectedEntity] : null;

  // Check if an entity is related to the selected one
  const isRelated = (entityId: string) => {
    if (!selectedEntity) return false;
    const selectedData = entityData[selectedEntity];
    if (!selectedData) return false;
    // Direct relation from selected
    if (selectedData.relations.includes(entityId)) return true;
    // Reverse relation (entity points to selected)
    const entityDataItem = entityData[entityId];
    if (entityDataItem?.relations.includes(selectedEntity)) return true;
    return false;
  };

  // Check if a relationship line should be highlighted
  const isRelationshipHighlighted = (from: string, to: string) => {
    if (!selectedEntity) return false;
    return from === selectedEntity || to === selectedEntity;
  };

  const boxWidth = 90;
  const boxHeight = 32;

  return (
    <div className="bg-card rounded-lg border p-6">
      <div className="text-muted-foreground mb-4 text-sm font-semibold uppercase">
        Entity Relationship Diagram
      </div>
      <p className="text-muted-foreground mb-4 text-sm">
        Click on any entity to see its properties and relationships.
      </p>

      {/* SVG Diagram */}
      <div className="bg-muted/30 mb-4 rounded-lg p-4">
        <svg
          width="540"
          height="370"
          className="mx-auto block"
        >
          {/* Relationship lines */}
          {relationships.map((rel, i) => {
            const fromEntity = entityData[rel.from];
            const toEntity = entityData[rel.to];
            if (!fromEntity || !toEntity) return null;
            const highlighted = isRelationshipHighlighted(rel.from, rel.to);

            // Calculate line endpoints based on entity positions
            const startX = fromEntity.x;
            let startY = fromEntity.y + boxHeight;
            const endX = toEntity.x;
            let endY = toEntity.y;

            // For lines that go sideways (from Market to children)
            if (Math.abs(fromEntity.x - toEntity.x) > 50) {
              // Draw from bottom of parent, then to top of child
              startY = fromEntity.y + boxHeight;
              endY = toEntity.y;
            }

            // Calculate label positions (near each end of line)
            const dx = endX - startX;
            const dy = endY - startY;
            const len = Math.sqrt(dx * dx + dy * dy);
            const ux = dx / len;
            const uy = dy / len;

            return (
              <g key={i}>
                <line
                  x1={startX}
                  y1={startY}
                  x2={endX}
                  y2={endY}
                  stroke={highlighted ? '#3b82f6' : '#475569'}
                  strokeWidth={highlighted ? 2 : 1}
                />
                {/* UML cardinality labels near each end */}
                <text
                  x={startX + ux * 12 + 6}
                  y={startY + uy * 12 + 4}
                  fontSize="10"
                  fill={highlighted ? '#3b82f6' : '#94a3b8'}
                  fontWeight="600"
                >
                  {rel.fromLabel}
                </text>
                <text
                  x={endX - ux * 12 + 6}
                  y={endY - uy * 12 + 4}
                  fontSize="10"
                  fill={highlighted ? '#3b82f6' : '#94a3b8'}
                  fontWeight="600"
                >
                  {rel.toLabel}
                </text>
              </g>
            );
          })}

          {/* Entity boxes */}
          {entities.map((entity) => {
            const isSelected = selectedEntity === entity.id;
            const related = isRelated(entity.id);

            return (
              <g
                key={entity.id}
                onClick={() => setSelectedEntity(isSelected ? null : entity.id)}
                className="cursor-pointer"
              >
                {/* Box shadow/glow for selected */}
                {isSelected && (
                  <rect
                    x={entity.x - boxWidth / 2 - 3}
                    y={entity.y - 3}
                    width={boxWidth + 6}
                    height={boxHeight + 6}
                    rx={6}
                    fill="none"
                    stroke="#3b82f6"
                    strokeWidth={2}
                  />
                )}
                {/* Highlight for related */}
                {related && !isSelected && (
                  <rect
                    x={entity.x - boxWidth / 2 - 2}
                    y={entity.y - 2}
                    width={boxWidth + 4}
                    height={boxHeight + 4}
                    rx={5}
                    fill="none"
                    stroke="#22c55e"
                    strokeWidth={2}
                    strokeDasharray="4"
                  />
                )}
                {/* Main box */}
                <rect
                  x={entity.x - boxWidth / 2}
                  y={entity.y}
                  width={boxWidth}
                  height={boxHeight}
                  rx={4}
                  fill={isSelected ? entity.color : '#1e293b'}
                  stroke={entity.color}
                  strokeWidth={isSelected ? 0 : 2}
                />
                {/* Entity name */}
                <text
                  x={entity.x}
                  y={entity.y + boxHeight / 2 + 5}
                  textAnchor="middle"
                  fontSize="12"
                  fontWeight="600"
                  fill={isSelected ? '#fff' : entity.color}
                >
                  {entity.name}
                </text>
              </g>
            );
          })}
        </svg>
      </div>

      {/* Selected entity properties */}
      {selected ? (
        <div className="overflow-hidden rounded-lg border">
          <div
            className="flex items-center gap-3 px-4 py-3"
            style={{ backgroundColor: selected.color + '20' }}
          >
            <div
              className="h-4 w-4 rounded"
              style={{ backgroundColor: selected.color }}
            />
            <div>
              <div className="font-semibold">{selected.name}</div>
              <div className="text-muted-foreground text-sm">{selected.description}</div>
            </div>
          </div>
          <div className="p-4">
            <div className="text-muted-foreground mb-3 text-sm font-semibold uppercase">
              Properties
            </div>
            <div className="space-y-2">
              {selected.properties.map((prop, i) => (
                <div
                  key={i}
                  className="flex items-start gap-3 text-sm"
                >
                  <code className="text-primary bg-primary/10 min-w-[140px] rounded px-1.5 py-0.5 font-mono text-xs">
                    {prop.name}
                  </code>
                  <span className="text-muted-foreground min-w-[70px] font-mono text-xs">
                    {prop.type}
                  </span>
                  <span className="text-foreground/80">{prop.desc}</span>
                </div>
              ))}
            </div>
            {selected.relations.length > 0 && (
              <div className="mt-4 border-t pt-4">
                <div className="text-sm">
                  <span className="text-muted-foreground">Has many: </span>
                  <span className="font-medium text-green-400">
                    {selected.relations.map((r) => entityData[r]?.name).join(', ')}
                  </span>
                </div>
              </div>
            )}
          </div>
        </div>
      ) : (
        <div className="text-muted-foreground rounded-lg border p-8 text-center">
          Click on an entity in the diagram above to view its properties
        </div>
      )}
    </div>
  );
}

// Market status state machine widget
function MarketStatusWidget() {
  const [currentStatus, setCurrentStatus] = useState<string>('unopened');

  const statuses = [
    {
      id: 'unopened',
      name: 'unopened',
      description:
        'Market exists but trading has not started. Common for events scheduled in the future.',
      color: 'bg-slate-500',
      next: ['open'],
    },
    {
      id: 'open',
      name: 'open',
      description: 'Active trading is allowed. Orders can be placed, amended, and canceled.',
      color: 'bg-green-500',
      next: ['closed'],
    },
    {
      id: 'closed',
      name: 'closed',
      description:
        'Trading halted, awaiting settlement. No new orders accepted. Existing positions frozen.',
      color: 'bg-amber-500',
      next: ['settled'],
    },
    {
      id: 'settled',
      name: 'settled',
      description: 'Final outcome determined. Positions resolved to $1 or $0. Terminal state.',
      color: 'bg-blue-500',
      next: [],
    },
  ];

  const current = statuses.find((s) => s.id === currentStatus)!;

  return (
    <div className="bg-card rounded-lg border p-6">
      <div className="text-muted-foreground mb-4 text-sm font-semibold uppercase">
        Market Status Lifecycle
      </div>

      <div className="mb-6 flex items-center justify-between">
        {statuses.map((status, i) => (
          <div
            key={status.id}
            className="flex items-center"
          >
            <button
              onClick={() => setCurrentStatus(status.id)}
              className={`relative rounded-lg border-2 px-4 py-2 transition-all ${
                currentStatus === status.id
                  ? 'border-primary bg-primary/10'
                  : 'border-muted hover:border-muted-foreground/50'
              }`}
            >
              <div className={`h-2 w-2 rounded-full ${status.color} absolute -right-1 -top-1`} />
              <span className="font-mono text-sm">{status.name}</span>
            </button>
            {i < statuses.length - 1 && <div className="bg-muted mx-2 h-0.5 w-8" />}
          </div>
        ))}
      </div>

      <div className="bg-muted/50 rounded-lg p-4">
        <div className="mb-2 flex items-center gap-2">
          <div className={`h-3 w-3 rounded-full ${current.color}`} />
          <span className="font-mono font-medium">{current.name}</span>
        </div>
        <p className="text-muted-foreground text-sm">{current.description}</p>
        {current.next.length > 0 && (
          <div className="mt-3 text-sm">
            <span className="text-muted-foreground">Can transition to: </span>
            <span className="text-primary font-mono">{current.next.join(', ')}</span>
          </div>
        )}
      </div>
    </div>
  );
}

// Order status widget
function OrderStatusWidget() {
  const statuses = [
    {
      id: 'resting',
      name: 'resting',
      description: 'Order is on the book, waiting to be matched. Has remaining_count > 0.',
      color: 'bg-blue-500',
      fields: ['order_id', 'price', 'remaining_count', 'created_time'],
    },
    {
      id: 'executed',
      name: 'executed',
      description: 'Order fully filled. remaining_count = 0, all contracts matched.',
      color: 'bg-green-500',
      fields: ['order_id', 'price', 'fill_count', 'executed_time'],
    },
    {
      id: 'canceled',
      name: 'canceled',
      description: 'Order removed from book by user or system. No further matching possible.',
      color: 'bg-red-500',
      fields: ['order_id', 'price', 'remaining_count', 'cancel_reason'],
    },
  ];

  return (
    <div className="bg-card rounded-lg border p-6">
      <div className="text-muted-foreground mb-4 text-sm font-semibold uppercase">
        Order Status States
      </div>

      <div className="grid grid-cols-3 gap-4">
        {statuses.map((status) => (
          <div
            key={status.id}
            className="bg-muted/50 rounded-lg p-4"
          >
            <div className="mb-2 flex items-center gap-2">
              <div className={`h-3 w-3 rounded-full ${status.color}`} />
              <span className="font-mono font-medium">{status.name}</span>
            </div>
            <p className="text-muted-foreground mb-3 text-sm">{status.description}</p>
            <div className="text-xs">
              <span className="text-muted-foreground">Key fields: </span>
              <span className="text-foreground/70 font-mono">{status.fields.join(', ')}</span>
            </div>
          </div>
        ))}
      </div>
    </div>
  );
}

// YES/NO equivalence widget
function YesNoEquivalenceWidget() {
  const [yesPrice, setYesPrice] = useState(65);
  const noPrice = 100 - yesPrice;

  return (
    <div className="bg-card rounded-lg border p-6">
      <div className="text-muted-foreground mb-4 text-sm font-semibold uppercase">
        YES/NO Price Equivalence
      </div>

      <div className="mb-6">
        <label className="text-muted-foreground text-sm">YES Price</label>
        <input
          type="range"
          min={1}
          max={99}
          value={yesPrice}
          onChange={(e) => setYesPrice(Number(e.target.value))}
          className="mt-2 w-full"
        />
      </div>

      <div className="mb-6 grid grid-cols-2 gap-4">
        <div className="rounded-lg border border-green-500/30 bg-green-500/10 p-4 text-center">
          <div className="mb-1 text-sm text-green-400">YES Price</div>
          <div className="text-3xl font-bold text-green-400">{yesPrice}¢</div>
        </div>
        <div className="rounded-lg border border-red-500/30 bg-red-500/10 p-4 text-center">
          <div className="mb-1 text-sm text-red-400">NO Price</div>
          <div className="text-3xl font-bold text-red-400">{noPrice}¢</div>
        </div>
      </div>

      <div className="bg-muted/50 space-y-2 rounded-lg p-4 text-sm">
        <p>
          <span className="text-muted-foreground">Buying YES at </span>
          <span className="font-mono text-green-400">{yesPrice}¢</span>
          <span className="text-muted-foreground"> = Selling NO at </span>
          <span className="font-mono text-red-400">{noPrice}¢</span>
        </p>
        <p>
          <span className="text-muted-foreground">Selling YES at </span>
          <span className="font-mono text-green-400">{yesPrice}¢</span>
          <span className="text-muted-foreground"> = Buying NO at </span>
          <span className="font-mono text-red-400">{noPrice}¢</span>
        </p>
        <p className="text-muted-foreground border-muted border-t pt-2">
          The orderbook shows both YES and NO bids. A bid for YES at X¢ is mathematically equivalent
          to an ask for NO at (100-X)¢.
        </p>
      </div>
    </div>
  );
}

export function KalshiApiEntitiesLesson() {
  const guide = getGuide('kalshi-api-entities')!;
  const nextGuide = getNextGuide('kalshi-api-entities');

  return (
    <LessonLayout>
      {/* Header */}
      <LessonRow>
        <LessonHeader
          title={guide.title}
          description={guide.description}
          duration="25 min"
          category="Developer"
        />
      </LessonRow>

      {/* Objective */}
      <LessonRow
        aside={
          <ConstraintBlock
            title="What We'll Cover"
            items={[
              'Core entities: Events, Markets, Series',
              'Trading entities: Orders, Fills, Positions',
              'Market data: Orderbook, Trades, Candlesticks',
              'Entity relationships and lifecycles',
            ]}
          />
        }
      >
        <ObjectiveBlock>
          By the end of this lesson, you'll understand every major entity in the Kalshi API, what
          their fields mean, how they relate to each other, and the state machines that govern their
          lifecycles. This is your mental model for building on Kalshi.
        </ObjectiveBlock>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 1: THE DOMAIN MODEL */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <InsightBlock title="Hierarchy Matters">
            Understanding the Series → Event → Market hierarchy is essential. Most API operations
            happen at the Market level, but filtering and discovery happen at Event and Series
            levels.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="The Domain Model"
          subtitle="How Kalshi organizes tradeable outcomes"
        />

        <p className="text-foreground leading-relaxed">
          Kalshi's API is organized around a three-level hierarchy:{' '}
          <GlossaryTooltip term={terms.series}>Series</GlossaryTooltip> →{' '}
          <GlossaryTooltip term={terms.event}>Event</GlossaryTooltip> →{' '}
          <GlossaryTooltip term={terms.market}>Market</GlossaryTooltip>. Everything you trade is a
          Market, but Markets are grouped into Events, and Events can be generated from Series
          templates.
        </p>

        <div className="mt-6">
          <EntityRelationshipWidget />
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 2: SERIES */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="Finding Series">
            Use the /series endpoints to discover what types of events exist on Kalshi. Each series
            has a ticker pattern that its events follow.
          </TipBlock>
        }
      >
        <SectionHeader
          title="Series: The Template Layer"
          subtitle="Recurring event patterns"
        />

        <p className="text-muted-foreground leading-relaxed">
          A <strong className="text-foreground">Series</strong> is a template for recurring events
          that follow the same rules and structure. Think of it as a blueprint that defines how
          events of a certain type work.
        </p>

        <ConceptBlock title="Series Examples">
          <ul className="space-y-2">
            <li>
              <strong>INXD</strong> — Daily S&P 500 close price brackets
            </li>
            <li>
              <strong>KXBTC</strong> — Daily Bitcoin price brackets
            </li>
            <li>
              <strong>FOMC</strong> — Federal Reserve rate decisions
            </li>
            <li>
              <strong>NFL-GAME</strong> — NFL game outcomes
            </li>
          </ul>
        </ConceptBlock>

        <div className="bg-muted/30 mt-6 rounded-lg border p-4">
          <div className="mb-3 text-sm font-semibold">Key Series Fields</div>
          <div className="grid gap-3 text-sm">
            <div className="flex gap-4">
              <code className="text-primary w-32 font-mono">ticker</code>
              <span className="text-muted-foreground">
                Unique identifier for the series (e.g., "INXD")
              </span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-32 font-mono">title</code>
              <span className="text-muted-foreground">Human-readable name</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-32 font-mono">category</code>
              <span className="text-muted-foreground">
                Classification (economics, politics, sports, etc.)
              </span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-32 font-mono">frequency</code>
              <span className="text-muted-foreground">
                How often new events are created (daily, weekly, etc.)
              </span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-32 font-mono">settlement_sources</code>
              <span className="text-muted-foreground">
                Where outcome data comes from for settlement
              </span>
            </div>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 3: EVENTS */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <InsightBlock title="Event Tickers">
            Event tickers typically include a date suffix. INXD-24DEC31 means "S&P 500 index event
            for December 31, 2024". The series ticker is the prefix.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="Events: The Container Layer"
          subtitle="Real-world occurrences you can trade on"
        />

        <p className="text-muted-foreground leading-relaxed">
          An <strong className="text-foreground">Event</strong> represents a specific real-world
          occurrence. It contains one or more Markets representing different outcomes or thresholds
          for that occurrence.
        </p>

        <div className="bg-muted/30 mt-6 rounded-lg border p-4">
          <div className="mb-3 text-sm font-semibold">Key Event Fields</div>
          <div className="grid gap-3 text-sm">
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">event_ticker</code>
              <span className="text-muted-foreground">
                Unique identifier (e.g., "INXD-24DEC31")
              </span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">series_ticker</code>
              <span className="text-muted-foreground">Parent series this event belongs to</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">title</code>
              <span className="text-muted-foreground">Human-readable description</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">strike_date</code>
              <span className="text-muted-foreground">
                When the event outcome will be determined
              </span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">markets</code>
              <span className="text-muted-foreground">
                Array of Market objects within this event
              </span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">mutually_exclusive</code>
              <span className="text-muted-foreground">
                If true, exactly one market will settle YES
              </span>
            </div>
          </div>
        </div>

        <div className="mt-6 rounded-lg border border-blue-500/30 bg-blue-500/10 p-4">
          <div className="mb-2 font-medium text-blue-400">Single vs. Multiple Market Events</div>
          <p className="text-muted-foreground text-sm">
            <strong>Single-market events</strong> have one binary outcome: "Will X happen?" (e.g.,
            "Will Lakers beat Celtics?")
          </p>
          <p className="text-muted-foreground mt-2 text-sm">
            <strong>Multi-market events</strong> have multiple mutually exclusive outcomes: "Which
            bracket will the S&P close in?" with markets for 5800-5850, 5850-5900, etc. Exactly one
            will settle YES.
          </p>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 4: MARKETS */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <WarningBlock title="Market vs Event Tickers">
            Don't confuse market_ticker with event_ticker. A market ticker includes the specific
            outcome (e.g., "INXD-24DEC31-B5900" for the 5900 bracket), while the event ticker is the
            parent container.
          </WarningBlock>
        }
      >
        <SectionHeader
          title="Markets: The Trading Layer"
          subtitle="Binary contracts you actually trade"
        />

        <p className="text-muted-foreground leading-relaxed">
          A <strong className="text-foreground">Market</strong> is a binary contract representing a
          specific YES/NO outcome. This is where all trading happens. Every order, position, and
          fill is associated with a specific market.
        </p>

        <div className="mt-6">
          <MarketStatusWidget />
        </div>

        <div className="bg-muted/30 mt-6 rounded-lg border p-4">
          <div className="mb-3 text-sm font-semibold">Key Market Fields</div>
          <div className="grid gap-3 text-sm">
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">ticker</code>
              <span className="text-muted-foreground">Unique market identifier</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">event_ticker</code>
              <span className="text-muted-foreground">Parent event this market belongs to</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">status</code>
              <span className="text-muted-foreground">unopened | open | closed | settled</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">yes_bid / yes_ask</code>
              <span className="text-muted-foreground">Best bid and ask prices for YES side</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">last_price</code>
              <span className="text-muted-foreground">Most recent trade price</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">volume</code>
              <span className="text-muted-foreground">Total contracts traded</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">volume_24h</code>
              <span className="text-muted-foreground">Contracts traded in last 24 hours</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">open_interest</code>
              <span className="text-muted-foreground">Outstanding contracts (not yet settled)</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">result</code>
              <span className="text-muted-foreground">
                Settlement outcome: "yes" | "no" | null (if not settled)
              </span>
            </div>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 5: THE YES/NO DUALITY */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <InsightBlock title="One Orderbook, Two Views">
            The API returns both YES and NO bids in the orderbook response. Internally, they're the
            same book — a YES bid at 60¢ IS a NO ask at 40¢.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="The YES/NO Duality"
          subtitle="Understanding binary market structure"
        />

        <p className="text-muted-foreground leading-relaxed">
          Every Kalshi market has two sides: YES and NO. These aren't separate markets — they're two
          views of the same underlying contract. Understanding this duality is essential for
          interpreting orderbook data and positions.
        </p>

        <div className="mt-6">
          <YesNoEquivalenceWidget />
        </div>

        <div className="mt-6 space-y-4">
          <Term term="Positive Position (Long YES)">
            You own YES contracts. If the market settles YES, each contract pays $1. If it settles
            NO, they're worth $0. Displayed as a positive number in position.
          </Term>

          <Term term="Negative Position (Long NO)">
            You own NO contracts. If the market settles NO, each contract pays $1. If it settles
            YES, they're worth $0. Displayed as a negative number in position (which can be
            confusing — see it as "short YES" or "long NO").
          </Term>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 6: ORDERS */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="Order Limits">
            Kalshi allows up to 200,000 open orders per user. Batch operations can submit up to 20
            orders per request for efficiency.
          </TipBlock>
        }
      >
        <SectionHeader
          title="Orders: Your Trading Intent"
          subtitle="How orders work and their lifecycle"
        />

        <p className="text-muted-foreground leading-relaxed">
          An <strong className="text-foreground">Order</strong> represents your intent to buy or
          sell contracts at a specific price. Orders can be limit orders (resting until filled) or
          market orders (executed immediately at best available price).
        </p>

        <div className="mt-6">
          <OrderStatusWidget />
        </div>

        <div className="bg-muted/30 mt-6 rounded-lg border p-4">
          <div className="mb-3 text-sm font-semibold">Key Order Fields</div>
          <div className="grid gap-3 text-sm">
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">order_id</code>
              <span className="text-muted-foreground">Unique order identifier (UUID)</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">client_order_id</code>
              <span className="text-muted-foreground">Your custom identifier for tracking</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">ticker</code>
              <span className="text-muted-foreground">Market this order is for</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">side</code>
              <span className="text-muted-foreground">
                "yes" | "no" — which side you're buying/selling
              </span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">action</code>
              <span className="text-muted-foreground">"buy" | "sell"</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">type</code>
              <span className="text-muted-foreground">"limit" | "market"</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">count</code>
              <span className="text-muted-foreground">Total contracts in order</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">remaining_count</code>
              <span className="text-muted-foreground">Contracts not yet filled</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">yes_price / no_price</code>
              <span className="text-muted-foreground">Limit price in cents (1-99)</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">status</code>
              <span className="text-muted-foreground">resting | executed | canceled</span>
            </div>
          </div>
        </div>

        <div className="mt-6 rounded-lg border border-amber-500/30 bg-amber-500/10 p-4">
          <div className="mb-2 font-medium text-amber-400">Order Actions Explained</div>
          <div className="text-muted-foreground space-y-2 text-sm">
            <p>
              <strong>buy + yes:</strong> Buying YES contracts. You profit if outcome is YES.
            </p>
            <p>
              <strong>sell + yes:</strong> Selling YES contracts you own (closing a long YES
              position).
            </p>
            <p>
              <strong>buy + no:</strong> Buying NO contracts. You profit if outcome is NO.
            </p>
            <p>
              <strong>sell + no:</strong> Selling NO contracts you own (closing a long NO position).
            </p>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 7: FILLS */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <InsightBlock title="Fills vs Trades">
            A Fill is YOUR record of a transaction. A Trade is the PUBLIC record that appears on the
            tape. Every trade creates two fills — one for each counterparty.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="Fills: Completed Transactions"
          subtitle="Records of your executed trades"
        />

        <p className="text-muted-foreground leading-relaxed">
          A <strong className="text-foreground">Fill</strong> is created when your order matches
          with a counterparty. Each fill represents contracts that have changed hands at a specific
          price.
        </p>

        <div className="bg-muted/30 mt-6 rounded-lg border p-4">
          <div className="mb-3 text-sm font-semibold">Key Fill Fields</div>
          <div className="grid gap-3 text-sm">
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">trade_id</code>
              <span className="text-muted-foreground">Unique fill identifier</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">order_id</code>
              <span className="text-muted-foreground">Order that generated this fill</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">ticker</code>
              <span className="text-muted-foreground">Market traded</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">side</code>
              <span className="text-muted-foreground">"yes" | "no"</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">action</code>
              <span className="text-muted-foreground">"buy" | "sell"</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">count</code>
              <span className="text-muted-foreground">Number of contracts in this fill</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">yes_price / no_price</code>
              <span className="text-muted-foreground">Execution price</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">created_time</code>
              <span className="text-muted-foreground">When the fill occurred (ISO 8601)</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">is_taker</code>
              <span className="text-muted-foreground">
                true if you crossed the spread (market order or aggressive limit)
              </span>
            </div>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 8: POSITIONS */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <WarningBlock title="Position Sign Convention">
            Positive position = long YES. Negative position = long NO (not "short"!). This confuses
            many developers initially.
          </WarningBlock>
        }
      >
        <SectionHeader
          title="Positions: Your Holdings"
          subtitle="Current contract ownership"
        />

        <p className="text-muted-foreground leading-relaxed">
          A <strong className="text-foreground">Position</strong> represents your current holdings
          in a market. The API provides both your position (current holdings) and total_traded
          (cumulative volume, useful for tracking activity).
        </p>

        <div className="bg-muted/30 mt-6 rounded-lg border p-4">
          <div className="mb-3 text-sm font-semibold">Key Position Fields</div>
          <div className="grid gap-3 text-sm">
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">ticker</code>
              <span className="text-muted-foreground">Market identifier</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">position</code>
              <span className="text-muted-foreground">
                Current contracts held (+ = YES, - = NO)
              </span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">market_exposure</code>
              <span className="text-muted-foreground">Dollar value at risk</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">realized_pnl</code>
              <span className="text-muted-foreground">Profit/loss from closed trades</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">total_traded</code>
              <span className="text-muted-foreground">
                Cumulative contracts traded (both directions)
              </span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">resting_orders_count</code>
              <span className="text-muted-foreground">Open orders in this market</span>
            </div>
          </div>
        </div>

        <div className="mt-6 rounded-lg border border-blue-500/30 bg-blue-500/10 p-4">
          <div className="mb-2 font-medium text-blue-400">Position Examples</div>
          <div className="text-muted-foreground space-y-2 text-sm">
            <p>
              <code className="text-foreground">position: 100</code> → You own 100 YES contracts.
              You profit if YES.
            </p>
            <p>
              <code className="text-foreground">position: -50</code> → You own 50 NO contracts. You
              profit if NO.
            </p>
            <p>
              <code className="text-foreground">position: 0</code> → Flat. No exposure to this
              market.
            </p>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 9: ORDERBOOK */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="WebSocket Updates">
            For real-time orderbook data, use the WebSocket feed with the "orderbook_delta" channel.
            REST polling is rate-limited and laggy.
          </TipBlock>
        }
      >
        <SectionHeader
          title="Orderbook: Market Depth"
          subtitle="All resting orders at each price level"
        />

        <p className="text-muted-foreground leading-relaxed">
          The <strong className="text-foreground">Orderbook</strong> shows all resting limit orders
          organized by price level. Kalshi returns both YES and NO sides, though they're
          mathematically equivalent (YES bid at 60 = NO ask at 40).
        </p>

        <div className="bg-muted/30 mt-6 rounded-lg border p-4">
          <div className="mb-3 text-sm font-semibold">Orderbook Response Structure</div>
          <pre className="bg-muted/50 overflow-x-auto rounded p-4 text-xs">
            {`{
  "orderbook": {
    "yes": [
      [60, 150],  // 150 contracts bid at 60¢
      [59, 200],  // 200 contracts bid at 59¢
      [58, 75]    // 75 contracts bid at 58¢
    ],
    "no": [
      [39, 150],  // Equivalent to yes ask at 61¢
      [38, 100],  // Equivalent to yes ask at 62¢
      [37, 50]    // Equivalent to yes ask at 63¢
    ]
  }
}`}
          </pre>
          <p className="text-muted-foreground mt-3 text-sm">
            Each entry is <code>[price, quantity]</code>. YES bids are sorted descending (best bid
            first). NO bids show the other side of the book.
          </p>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 10: TRADES (PUBLIC TAPE) */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <InsightBlock title="Tape Reading">
            The trades endpoint shows public market activity. Use it to gauge sentiment, spot large
            players, or analyze historical price action.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="Trades: The Public Tape"
          subtitle="All executed trades in a market"
        />

        <p className="text-muted-foreground leading-relaxed">
          <strong className="text-foreground">Trades</strong> are the public record of all matched
          transactions. Unlike fills (which are private to you), trades show what everyone is doing.
          Useful for analysis and tape reading.
        </p>

        <div className="bg-muted/30 mt-6 rounded-lg border p-4">
          <div className="mb-3 text-sm font-semibold">Key Trade Fields</div>
          <div className="grid gap-3 text-sm">
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">trade_id</code>
              <span className="text-muted-foreground">Unique trade identifier</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">ticker</code>
              <span className="text-muted-foreground">Market where trade occurred</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">count</code>
              <span className="text-muted-foreground">Number of contracts traded</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">yes_price</code>
              <span className="text-muted-foreground">Execution price (YES side)</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">taker_side</code>
              <span className="text-muted-foreground">
                "yes" | "no" — which side crossed the spread
              </span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">created_time</code>
              <span className="text-muted-foreground">Trade timestamp</span>
            </div>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 11: SETTLEMENTS */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <WarningBlock title="Settlement Timing">
            Settlement doesn't happen instantly at market close. There's typically a delay while
            Kalshi verifies the official outcome from settlement sources.
          </WarningBlock>
        }
      >
        <SectionHeader
          title="Settlements: Position Resolution"
          subtitle="How markets resolve at expiration"
        />

        <p className="text-muted-foreground leading-relaxed">
          <strong className="text-foreground">Settlement</strong> is when a market's outcome is
          determined and positions are resolved. YES contracts pay $1 if the outcome occurred, $0
          otherwise. NO contracts are the inverse.
        </p>

        <div className="bg-muted/30 mt-6 rounded-lg border p-4">
          <div className="mb-3 text-sm font-semibold">Settlement Fields</div>
          <div className="grid gap-3 text-sm">
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">market.result</code>
              <span className="text-muted-foreground">"yes" | "no" | null — the outcome</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">market.status</code>
              <span className="text-muted-foreground">Becomes "settled" when resolved</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">settlement_value</code>
              <span className="text-muted-foreground">100 (YES won) or 0 (NO won)</span>
            </div>
          </div>
        </div>

        <div className="mt-6 rounded-lg border border-green-500/30 bg-green-500/10 p-4">
          <div className="mb-2 font-medium text-green-400">Settlement P&L Calculation</div>
          <div className="text-muted-foreground space-y-2 text-sm">
            <p>
              <strong>If you're long YES (position {'>'} 0) and result is YES:</strong>
            </p>
            <p className="ml-4 font-mono">P&L = position × ($1.00 - avg_entry_price)</p>
            <p>
              <strong>If you're long NO (position {'<'} 0) and result is NO:</strong>
            </p>
            <p className="ml-4 font-mono">P&L = |position| × ($1.00 - avg_entry_price)</p>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 12: ACCOUNT ENTITIES */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="Subaccounts">
            Kalshi supports up to 32 subaccounts per user. Useful for separating strategies or
            managing different bankrolls independently.
          </TipBlock>
        }
      >
        <SectionHeader
          title="Account Entities"
          subtitle="Balance, portfolio, and subaccounts"
        />

        <p className="text-muted-foreground leading-relaxed">
          The API provides several endpoints for managing your account state, including balance
          queries, portfolio summaries, and subaccount management.
        </p>

        <div className="bg-muted/30 mt-6 rounded-lg border p-4">
          <div className="mb-3 text-sm font-semibold">Balance & Portfolio Fields</div>
          <div className="grid gap-3 text-sm">
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">balance</code>
              <span className="text-muted-foreground">Available cash in cents</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">portfolio_value</code>
              <span className="text-muted-foreground">Total account value including positions</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">payout</code>
              <span className="text-muted-foreground">Pending payouts from settlements</span>
            </div>
          </div>
        </div>

        <div className="bg-muted/30 mt-4 rounded-lg border p-4">
          <div className="mb-3 text-sm font-semibold">Subaccount Fields</div>
          <div className="grid gap-3 text-sm">
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">member_id</code>
              <span className="text-muted-foreground">Your main account ID</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">subaccount_id</code>
              <span className="text-muted-foreground">1-32 (0 is primary account)</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">subaccount_balance</code>
              <span className="text-muted-foreground">Balance for this subaccount</span>
            </div>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 13: REAL-TIME DATA */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <InsightBlock title="WebSocket Channels">
            Key channels: "ticker" for price updates, "orderbook_delta" for book changes, "fill" for
            your executions, "trade" for public tape.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="Real-Time Data"
          subtitle="WebSocket streaming entities"
        />

        <p className="text-muted-foreground leading-relaxed">
          For real-time updates, Kalshi provides a WebSocket API. After authentication, subscribe to
          channels to receive streaming updates.
        </p>

        <div className="bg-muted/30 mt-6 rounded-lg border p-4">
          <div className="mb-3 text-sm font-semibold">WebSocket Message Types</div>
          <div className="grid gap-3 text-sm">
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">ticker</code>
              <span className="text-muted-foreground">
                Price, volume, and open interest updates
              </span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">orderbook_delta</code>
              <span className="text-muted-foreground">Incremental orderbook changes</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">trade</code>
              <span className="text-muted-foreground">Public trade notifications</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">fill</code>
              <span className="text-muted-foreground">Your order executions</span>
            </div>
            <div className="flex gap-4">
              <code className="text-primary w-40 font-mono">order</code>
              <span className="text-muted-foreground">Your order status changes</span>
            </div>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* KEY TAKEAWAYS */}
      {/* =================================================================== */}
      <LessonRow>
        <SectionHeader
          title="Key Takeaways"
          subtitle="The mental model for Kalshi's API"
        />

        <div className="space-y-4">
          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="bg-primary/20 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
              <span className="text-primary font-bold">1</span>
            </div>
            <div>
              <div className="font-medium">Hierarchy: Series → Event → Market</div>
              <p className="text-muted-foreground mt-1 text-sm">
                Series are templates, Events are instances, Markets are the tradeable contracts.
                Most operations happen at the Market level.
              </p>
            </div>
          </div>

          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="bg-primary/20 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
              <span className="text-primary font-bold">2</span>
            </div>
            <div>
              <div className="font-medium">YES/NO Duality</div>
              <p className="text-muted-foreground mt-1 text-sm">
                Every market has two equivalent views. YES at 60¢ = NO at 40¢. Positions can be
                positive (long YES) or negative (long NO).
              </p>
            </div>
          </div>

          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="bg-primary/20 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
              <span className="text-primary font-bold">3</span>
            </div>
            <div>
              <div className="font-medium">Order → Fill → Position Flow</div>
              <p className="text-muted-foreground mt-1 text-sm">
                Orders express intent, Fills record executions, Positions track current holdings.
                Each step updates your account state.
              </p>
            </div>
          </div>

          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="bg-primary/20 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
              <span className="text-primary font-bold">4</span>
            </div>
            <div>
              <div className="font-medium">
                Market Lifecycle: unopened → open → closed → settled
              </div>
              <p className="text-muted-foreground mt-1 text-sm">
                Markets progress through states. Trading only happens in "open". Settlement resolves
                positions to $1 or $0.
              </p>
            </div>
          </div>

          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="bg-primary/20 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
              <span className="text-primary font-bold">5</span>
            </div>
            <div>
              <div className="font-medium">REST for Operations, WebSocket for Real-Time</div>
              <p className="text-muted-foreground mt-1 text-sm">
                Use REST endpoints for placing orders and querying state. Use WebSocket for
                streaming price updates, fills, and orderbook changes.
              </p>
            </div>
          </div>
        </div>
      </LessonRow>

      {/* Next Step */}
      {nextGuide && (
        <LessonRow>
          <NextStepBlock
            title={nextGuide.title}
            description={nextGuide.description}
            href={`/${nextGuide.slug}`}
          />
        </LessonRow>
      )}
    </LessonLayout>
  );
}
