import { useState } from 'react';
import { LessonRow, SectionHeader } from '../LessonLayout';
import { TipBlock, InsightBlock } from '../blocks';

/**
 * MM Trade Identification Practice Quiz
 *
 * Interactive scenarios where users evaluate market opportunities
 * and check their answers against expert analysis.
 */

type Scenario = {
  id: string;
  market: string;
  yesPrice: number;
  noPrice: number;
  spread: number;
  queueDepth: number;
  flowRate: string;
  flowDescription: string;
  timeToEvent: string;
  additionalInfo: string;
  correctAnswer: 'take' | 'pass' | 'small';
  difficulty: 'easy' | 'medium' | 'hard';
  explanation: string[];
  keyFactors: { factor: string; impact: 'positive' | 'negative' | 'neutral' }[];
};

const scenarios: Scenario[] = [
  {
    id: '1',
    market: 'Bucks vs Pacers - Bucks Moneyline',
    yesPrice: 67,
    noPrice: 31,
    spread: 2,
    queueDepth: 80,
    flowRate: 'High',
    flowDescription: '50+ contracts/min, game starts in 20 min',
    timeToEvent: '20 min',
    additionalInfo: 'Normal pre-game rush. No news pending.',
    correctAnswer: 'take',
    difficulty: 'easy',
    explanation: [
      '2¢ spread is healthy profit margin',
      'High flow means fast fills',
      '20 min to game = peak retail activity',
      'No information risk (no pending news)',
      'Queue depth of 80 with 50+/min flow = fill in ~2 min',
    ],
    keyFactors: [
      { factor: '2¢ spread', impact: 'positive' },
      { factor: 'High flow', impact: 'positive' },
      { factor: 'Short time to event', impact: 'positive' },
      { factor: 'No news pending', impact: 'positive' },
    ],
  },
  {
    id: '2',
    market: 'Suns vs Nuggets - Suns Moneyline',
    yesPrice: 42,
    noPrice: 57,
    spread: 1,
    queueDepth: 300,
    flowRate: 'Low',
    flowDescription: '8 contracts/min, game not until tomorrow',
    timeToEvent: '18 hours',
    additionalInfo: 'Overnight market. Very quiet.',
    correctAnswer: 'pass',
    difficulty: 'easy',
    explanation: [
      '1¢ spread is too thin after fees',
      'Low flow = would take 37+ min to fill',
      '18 hours to event = no urgency from retail',
      'Capital would be tied up for uncertain return',
      'Better to wait until closer to game time',
    ],
    keyFactors: [
      { factor: '1¢ spread', impact: 'negative' },
      { factor: 'Low flow', impact: 'negative' },
      { factor: 'Deep queue (300)', impact: 'negative' },
      { factor: 'Far from event', impact: 'negative' },
    ],
  },
  {
    id: '3',
    market: 'Celtics vs Heat - Celtics Moneyline',
    yesPrice: 85,
    noPrice: 12,
    spread: 3,
    queueDepth: 25,
    flowRate: 'Medium',
    flowDescription: '15 contracts/min',
    timeToEvent: '1 hour',
    additionalInfo: 'Heavy favorite. Most bets are small YES bets on Celtics.',
    correctAnswer: 'small',
    difficulty: 'medium',
    explanation: [
      '3¢ spread is attractive',
      'Short queue + medium flow = fast fills',
      'BUT: Heavy favorite creates asymmetric risk',
      'You win 3¢ per contract, but if Celtics lose you lose 88¢',
      'Worth taking but SIZE DOWN significantly',
    ],
    keyFactors: [
      { factor: '3¢ spread', impact: 'positive' },
      { factor: 'Short queue', impact: 'positive' },
      { factor: 'Heavy favorite (85%)', impact: 'negative' },
      { factor: 'Asymmetric risk', impact: 'negative' },
    ],
  },
  {
    id: '4',
    market: 'Lakers vs Clippers - Lakers Moneyline',
    yesPrice: 52,
    noPrice: 46,
    spread: 2,
    queueDepth: 150,
    flowRate: 'Very Low',
    flowDescription: '2 contracts/min - LeBron injury report pending',
    timeToEvent: '2 hours',
    additionalInfo: "Market is frozen waiting for LeBron's status update expected in ~30 min.",
    correctAnswer: 'pass',
    difficulty: 'medium',
    explanation: [
      'Spread looks fine at 2¢',
      'BUT: Low flow is a warning sign',
      'Injury report pending = major information risk',
      'When report drops, price will jump 5-10¢ instantly',
      "If you're in queue, you'll get adversely selected",
      'Wait for news, then re-evaluate',
    ],
    keyFactors: [
      { factor: '2¢ spread', impact: 'positive' },
      { factor: 'Very low flow', impact: 'negative' },
      { factor: 'Pending injury report', impact: 'negative' },
      { factor: 'Adverse selection risk', impact: 'negative' },
    ],
  },
  {
    id: '5',
    market: 'Mavs vs Spurs - Mavs Moneyline',
    yesPrice: 74,
    noPrice: 24,
    spread: 2,
    queueDepth: 45,
    flowRate: 'High',
    flowDescription: '40 contracts/min, 5 min to tip-off',
    timeToEvent: '5 min',
    additionalInfo: 'Last-minute betting rush. Market closes at tip-off.',
    correctAnswer: 'take',
    difficulty: 'medium',
    explanation: [
      '5 min to close = maximum urgency',
      'High flow + short queue = fills in ~1 min',
      '2¢ spread is acceptable',
      'Yes, Mavs are favorites, but short exposure time limits risk',
      'Classic pre-game opportunity - act fast',
    ],
    keyFactors: [
      { factor: '2¢ spread', impact: 'positive' },
      { factor: 'High flow', impact: 'positive' },
      { factor: '5 min to close', impact: 'positive' },
      { factor: 'Short queue (45)', impact: 'positive' },
    ],
  },
  {
    id: '6',
    market: 'Warriors vs Thunder - Warriors Moneyline',
    yesPrice: 58,
    noPrice: 40,
    spread: 2,
    queueDepth: 200,
    flowRate: 'Medium-Low',
    flowDescription: '12 contracts/min',
    timeToEvent: '3 hours',
    additionalInfo: 'Normal afternoon market. Warriors slightly favored.',
    correctAnswer: 'pass',
    difficulty: 'hard',
    explanation: [
      '2¢ spread is borderline acceptable',
      'Queue of 200 at 12/min = 17 min wait',
      '3 hours to event means flow will increase later',
      'Current opportunity cost is high',
      'Better strategy: wait until 1 hour before when flow picks up',
      'Pass now, revisit closer to game time',
    ],
    keyFactors: [
      { factor: '2¢ spread', impact: 'neutral' },
      { factor: 'Deep queue (200)', impact: 'negative' },
      { factor: '3 hours to event', impact: 'negative' },
      { factor: 'Flow will improve later', impact: 'negative' },
    ],
  },
  {
    id: '7',
    market: 'Raptors vs Magic - Raptors Moneyline',
    yesPrice: 63,
    noPrice: 34,
    spread: 3,
    queueDepth: 15,
    flowRate: 'Medium',
    flowDescription: '20 contracts/min',
    timeToEvent: '45 min',
    additionalInfo: 'Moderate interest game. Spread just widened from 2¢ to 3¢.',
    correctAnswer: 'take',
    difficulty: 'medium',
    explanation: [
      '3¢ spread is good profit margin',
      'Very short queue (15) = fills in under 1 min',
      'Spread widening could signal opportunity',
      '45 min to event is good timing',
      'Take the trade while spread is wide',
    ],
    keyFactors: [
      { factor: '3¢ spread', impact: 'positive' },
      { factor: 'Very short queue (15)', impact: 'positive' },
      { factor: 'Spread just widened', impact: 'positive' },
      { factor: 'Good timing (45 min)', impact: 'positive' },
    ],
  },
  {
    id: '8',
    market: 'Kings vs Pelicans - Kings Moneyline',
    yesPrice: 51,
    noPrice: 48,
    spread: 1,
    queueDepth: 100,
    flowRate: 'Very High',
    flowDescription: '80+ contracts/min, lots of action',
    timeToEvent: '10 min',
    additionalInfo: "Pick'em game. Very active market.",
    correctAnswer: 'take',
    difficulty: 'hard',
    explanation: [
      '1¢ spread is normally too thin...',
      'BUT: 80+/min flow is exceptional',
      "You'll fill 100 contracts in ~1 min, then refill",
      'At this volume, even 1¢ adds up quickly',
      "Pick'em means balanced risk (no heavy favorite)",
      'High-frequency, low-margin opportunity',
    ],
    keyFactors: [
      { factor: '1¢ spread', impact: 'negative' },
      { factor: 'Very high flow (80+/min)', impact: 'positive' },
      { factor: "Pick'em game (balanced risk)", impact: 'positive' },
      { factor: '10 min to event', impact: 'positive' },
    ],
  },
  {
    id: '9',
    market: 'Pistons vs Hornets - Pistons Moneyline',
    yesPrice: 44,
    noPrice: 54,
    spread: 2,
    queueDepth: 250,
    flowRate: 'Very Low',
    flowDescription: '3 contracts/min',
    timeToEvent: '4 hours',
    additionalInfo: 'Low-interest game. Neither team in playoff hunt.',
    correctAnswer: 'pass',
    difficulty: 'easy',
    explanation: [
      'Low-interest games have low flow',
      '250 queue at 3/min = 83 min wait',
      '4 hours to event with no expected pick-up',
      "2¢ spread isn't enough to justify the wait",
      'This game might never get enough action',
    ],
    keyFactors: [
      { factor: '2¢ spread', impact: 'neutral' },
      { factor: 'Deep queue (250)', impact: 'negative' },
      { factor: 'Very low flow', impact: 'negative' },
      { factor: 'Low-interest game', impact: 'negative' },
    ],
  },
  {
    id: '10',
    market: '76ers vs Knicks - 76ers Moneyline',
    yesPrice: 56,
    noPrice: 42,
    spread: 2,
    queueDepth: 60,
    flowRate: 'Medium',
    flowDescription: '25 contracts/min',
    timeToEvent: '30 min',
    additionalInfo:
      'Big market rivalry. Joel Embiid just confirmed to play after being questionable.',
    correctAnswer: 'take',
    difficulty: 'hard',
    explanation: [
      'News just resolved in a predictable direction',
      'Market has already adjusted (price moved)',
      'Now safe to quote - no more information risk',
      '2¢ spread, medium flow, 30 min to event = good setup',
      'Rivalry game means good retail interest',
      'Queue of 60 at 25/min = 2-3 min wait',
    ],
    keyFactors: [
      { factor: '2¢ spread', impact: 'positive' },
      { factor: 'News just resolved', impact: 'positive' },
      { factor: 'Big market rivalry', impact: 'positive' },
      { factor: '30 min to event', impact: 'positive' },
    ],
  },
];

function ScenarioCard({
  scenario,
  questionNumber,
  totalQuestions,
  onAnswer,
  onNext,
}: {
  scenario: Scenario;
  questionNumber: number;
  totalQuestions: number;
  onAnswer: (answer: string, isCorrect: boolean) => void;
  onNext: () => void;
}) {
  const [userAnswer, setUserAnswer] = useState<string | null>(null);
  const [answered, setAnswered] = useState(false);

  const handleAnswer = (answer: string) => {
    if (answered) return;
    setUserAnswer(answer);
    setAnswered(true);
    onAnswer(answer, answer === scenario.correctAnswer);
  };

  const isCorrect = userAnswer === scenario.correctAnswer;
  const isLastQuestion = questionNumber === totalQuestions;

  const difficultyColors: Record<string, string> = {
    easy: 'bg-green-900/50 text-green-400',
    medium: 'bg-amber-900/50 text-amber-400',
    hard: 'bg-red-900/50 text-red-400',
  };

  return (
    <div className="overflow-hidden rounded-lg border border-slate-700 bg-slate-800/50">
      {/* Header */}
      <div className="flex items-start justify-between border-b border-slate-700 bg-slate-900/50 p-4">
        <div>
          <div className="mb-1 text-xs text-slate-500">
            Question {questionNumber} of {totalQuestions}
          </div>
          <div className="font-semibold text-white">{scenario.market}</div>
          <div className="mt-1 text-xs text-slate-500">
            <span className={`rounded px-2 py-0.5 ${difficultyColors[scenario.difficulty]}`}>
              {scenario.difficulty}
            </span>
          </div>
        </div>
        {answered && (
          <div
            className={`rounded px-3 py-1 text-sm font-bold ${
              isCorrect ? 'bg-green-600 text-white' : 'bg-red-600 text-white'
            }`}
          >
            {isCorrect ? 'CORRECT' : 'INCORRECT'}
          </div>
        )}
      </div>

      {/* Market Data */}
      <div className="p-4">
        <div className="mb-4 grid grid-cols-2 gap-4 md:grid-cols-4">
          <div>
            <div className="text-xs text-slate-500">YES Price</div>
            <div className="text-xl font-bold text-green-400">{scenario.yesPrice}¢</div>
          </div>
          <div>
            <div className="text-xs text-slate-500">NO Price</div>
            <div className="text-xl font-bold text-red-400">{scenario.noPrice}¢</div>
          </div>
          <div>
            <div className="text-xs text-slate-500">Spread</div>
            <div className="text-xl font-bold text-white">{scenario.spread}¢</div>
          </div>
          <div>
            <div className="text-xs text-slate-500">Time to Event</div>
            <div className="text-xl font-bold text-amber-400">{scenario.timeToEvent}</div>
          </div>
        </div>

        <div className="mb-4 grid grid-cols-2 gap-4">
          <div className="rounded bg-slate-900/50 p-3">
            <div className="text-xs text-slate-500">Queue Depth (NO bid)</div>
            <div className="text-lg text-white">{scenario.queueDepth} contracts</div>
          </div>
          <div className="rounded bg-slate-900/50 p-3">
            <div className="text-xs text-slate-500">Flow Rate</div>
            <div className="text-lg text-white">{scenario.flowRate}</div>
            <div className="text-xs text-slate-400">{scenario.flowDescription}</div>
          </div>
        </div>

        <div className="mb-4 rounded border border-blue-700/50 bg-blue-900/20 p-3">
          <div className="mb-1 text-xs text-blue-400">Additional Context</div>
          <p className="text-sm text-slate-300">{scenario.additionalInfo}</p>
        </div>

        {/* Answer buttons - only show when not answered */}
        {!answered && (
          <div className="flex gap-3">
            <button
              onClick={() => handleAnswer('take')}
              className="flex-1 rounded-lg bg-slate-700 py-3 font-semibold text-slate-300 transition-colors hover:bg-green-600 hover:text-white"
            >
              TAKE
            </button>
            <button
              onClick={() => handleAnswer('small')}
              className="flex-1 rounded-lg bg-slate-700 py-3 font-semibold text-slate-300 transition-colors hover:bg-amber-600 hover:text-white"
            >
              SMALL SIZE
            </button>
            <button
              onClick={() => handleAnswer('pass')}
              className="flex-1 rounded-lg bg-slate-700 py-3 font-semibold text-slate-300 transition-colors hover:bg-red-600 hover:text-white"
            >
              PASS
            </button>
          </div>
        )}

        {/* Immediate Feedback */}
        {answered && (
          <div className="space-y-4">
            <div
              className={`rounded-lg p-4 ${
                isCorrect
                  ? 'border border-green-700/50 bg-green-900/20'
                  : 'border border-red-700/50 bg-red-900/20'
              }`}
            >
              <div className="text-sm">
                <span className="text-slate-400">Your answer: </span>
                <span className="font-semibold text-white">{userAnswer?.toUpperCase()}</span>
                {!isCorrect && (
                  <>
                    <span className="text-slate-400"> → Correct: </span>
                    <span className="font-semibold text-white">
                      {scenario.correctAnswer.toUpperCase()}
                    </span>
                  </>
                )}
              </div>
            </div>

            <div className="rounded-lg bg-slate-900/50 p-4">
              <div className="mb-2 text-sm text-slate-400">Why?</div>
              <ul className="space-y-1">
                {scenario.explanation.map((e, i) => (
                  <li
                    key={i}
                    className="text-sm text-slate-300"
                  >
                    • {e}
                  </li>
                ))}
              </ul>
            </div>

            <div className="rounded-lg bg-slate-900/50 p-4">
              <div className="mb-2 text-sm text-slate-400">Key Factors:</div>
              <div className="flex flex-wrap gap-2">
                {scenario.keyFactors.map((kf, i) => {
                  const impactColors: Record<string, string> = {
                    positive: 'bg-green-900/50 text-green-400',
                    negative: 'bg-red-900/50 text-red-400',
                    neutral: 'bg-slate-700 text-slate-400',
                  };
                  return (
                    <span
                      key={i}
                      className={`rounded px-2 py-1 text-xs ${impactColors[kf.impact]}`}
                    >
                      {kf.factor}
                    </span>
                  );
                })}
              </div>
            </div>

            {/* Next button */}
            <button
              onClick={onNext}
              className="w-full rounded-lg bg-purple-600 py-3 font-semibold text-white transition-colors hover:bg-purple-700"
            >
              {isLastQuestion ? 'See Final Results' : 'Next Question →'}
            </button>
          </div>
        )}
      </div>
    </div>
  );
}

export function MMPracticeQuizLesson() {
  const [currentIndex, setCurrentIndex] = useState(0);
  const [correctCount, setCorrectCount] = useState(0);
  const [finished, setFinished] = useState(false);

  const handleAnswer = (_answer: string, isCorrect: boolean) => {
    if (isCorrect) {
      setCorrectCount((prev) => prev + 1);
    }
  };

  const handleNext = () => {
    if (currentIndex >= scenarios.length - 1) {
      setFinished(true);
      return;
    }
    setCurrentIndex((prev) => prev + 1);
  };

  const handleReset = () => {
    setCurrentIndex(0);
    setCorrectCount(0);
    setFinished(false);
  };

  const currentScenario = scenarios[currentIndex];

  return (
    <div className="space-y-8">
      {/* Introduction */}
      <section>
        <SectionHeader
          title="MM Trade Identification: Practice"
          subtitle="Test your pattern recognition skills"
        />
        <LessonRow
          aside={
            <TipBlock title="How to Approach">
              Use the 5-question framework from the previous lesson. Check spread, queue, flow, info
              risk, and opportunity cost before deciding.
            </TipBlock>
          }
        >
          <p className="leading-relaxed text-slate-300">
            For each scenario, decide whether to <strong className="text-green-400">TAKE</strong>{' '}
            the trade, <strong className="text-amber-400">SMALL SIZE</strong> (take but reduced), or{' '}
            <strong className="text-red-400">PASS</strong>. You&apos;ll get immediate feedback after
            each answer.
          </p>
        </LessonRow>
      </section>

      {/* Progress bar */}
      <section>
        <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-4">
          <div className="mb-2 flex items-center justify-between">
            <span className="text-slate-400">Progress</span>
            <span className="font-bold text-white">
              {finished ? scenarios.length : currentIndex} / {scenarios.length}
              {!finished && (
                <span className="ml-2 text-green-400">({correctCount} correct so far)</span>
              )}
            </span>
          </div>
          <div className="h-2 overflow-hidden rounded-full bg-slate-700">
            <div
              className="h-full bg-purple-600 transition-all"
              style={{
                width: `${((finished ? scenarios.length : currentIndex) / scenarios.length) * 100}%`,
              }}
            />
          </div>
        </div>
      </section>

      {/* Current Scenario or Final Results */}
      {!finished ? (
        <section>
          <ScenarioCard
            key={currentScenario.id}
            scenario={currentScenario}
            questionNumber={currentIndex + 1}
            totalQuestions={scenarios.length}
            onAnswer={handleAnswer}
            onNext={handleNext}
          />
        </section>
      ) : (
        <section>
          <SectionHeader
            title="Final Results"
            subtitle="How you did"
          />
          <LessonRow
            aside={
              <InsightBlock title="Practice Makes Permanent">
                Retake the quiz until you can identify patterns instantly. Real trading requires
                split-second decisions.
              </InsightBlock>
            }
          >
            <div className="grid gap-4 md:grid-cols-3">
              <div className="rounded-lg border border-green-700/50 bg-green-900/20 p-4 text-center">
                <div className="text-3xl font-bold text-green-400">{correctCount}</div>
                <div className="text-sm text-slate-400">Correct</div>
              </div>
              <div className="rounded-lg border border-red-700/50 bg-red-900/20 p-4 text-center">
                <div className="text-3xl font-bold text-red-400">
                  {scenarios.length - correctCount}
                </div>
                <div className="text-sm text-slate-400">Incorrect</div>
              </div>
              <div className="rounded-lg border border-purple-700/50 bg-purple-900/20 p-4 text-center">
                <div className="text-3xl font-bold text-purple-400">
                  {Math.round((correctCount / scenarios.length) * 100)}%
                </div>
                <div className="text-sm text-slate-400">Accuracy</div>
              </div>
            </div>

            <div className="mt-6 rounded-lg border border-slate-700 bg-slate-800/50 p-6">
              <h4 className="mb-4 font-semibold text-white">Key Patterns to Remember</h4>
              <ul className="space-y-2 text-sm text-slate-300">
                <li>
                  <span className="text-green-400">✓</span> Wide spread + high flow + short queue =
                  almost always TAKE
                </li>
                <li>
                  <span className="text-green-400">✓</span> Tight spread + low flow = almost always
                  PASS
                </li>
                <li>
                  <span className="text-amber-400">!</span> Pending news = PASS until news resolves
                </li>
                <li>
                  <span className="text-amber-400">!</span> Heavy favorite = SIZE DOWN due to
                  asymmetric risk
                </li>
                <li>
                  <span className="text-amber-400">!</span> Tight spread CAN work if flow is
                  exceptional
                </li>
              </ul>
            </div>

            <div className="mt-6 flex justify-center">
              <button
                onClick={handleReset}
                className="rounded-lg bg-purple-600 px-8 py-3 font-semibold text-white transition-colors hover:bg-purple-700"
              >
                Try Again
              </button>
            </div>
          </LessonRow>
        </section>
      )}
    </div>
  );
}
