import { useState } from 'react';
import { Link } from 'react-router-dom';
import { LessonRow, SectionHeader } from '../LessonLayout';
import { TipBlock, InsightBlock, WarningBlock, ConstraintBlock } from '../blocks';
import { Formula } from '@/components/atoms/Formula';

/**
 * How Market Makers Identify Good Trades
 *
 * Teaching lesson that walks through the MM decision-making process
 * with concrete examples. Builds intuition for what makes a trade attractive.
 */

type TradeExample = {
  id: string;
  title: string;
  setup: {
    market: string;
    yesPrice: number;
    noPrice: number;
    bidAskSpread: number;
    queueDepth: number;
    retailFlow: string;
    timeToEvent: string;
  };
  analysis: {
    spreadProfit: number;
    queuePosition: string;
    flowQuality: string;
    riskAssessment: string;
  };
  decision: 'take' | 'pass' | 'conditional';
  reasoning: string[];
  whatToWatch: string[];
};

function MarketDisplay({ example }: { example: TradeExample }) {
  const { setup } = example;

  return (
    <div className="rounded-lg bg-slate-900/50 p-4">
      <div className="mb-2 text-sm text-slate-400">{setup.market}</div>
      <div className="grid grid-cols-2 gap-4 md:grid-cols-4">
        <div>
          <div className="text-xs text-slate-500">YES Price</div>
          <div className="text-xl font-bold text-green-400">{setup.yesPrice}¢</div>
        </div>
        <div>
          <div className="text-xs text-slate-500">NO Price</div>
          <div className="text-xl font-bold text-red-400">{setup.noPrice}¢</div>
        </div>
        <div>
          <div className="text-xs text-slate-500">Spread</div>
          <div className="text-xl font-bold text-white">{setup.bidAskSpread}¢</div>
        </div>
        <div>
          <div className="text-xs text-slate-500">Time to Event</div>
          <div className="text-xl font-bold text-amber-400">{setup.timeToEvent}</div>
        </div>
      </div>
      <div className="mt-4 grid grid-cols-2 gap-4">
        <div>
          <div className="text-xs text-slate-500">Queue Depth at Best NO Bid</div>
          <div className="text-lg text-white">{setup.queueDepth} contracts</div>
        </div>
        <div>
          <div className="text-xs text-slate-500">Retail Flow</div>
          <div className="text-lg text-white">{setup.retailFlow}</div>
        </div>
      </div>
    </div>
  );
}

function AnalysisDisplay({ example }: { example: TradeExample }) {
  const { analysis, decision, reasoning, whatToWatch } = example;

  return (
    <div className="space-y-4">
      <div className="grid gap-4 md:grid-cols-2">
        <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-4">
          <div className="mb-2 text-xs text-slate-400">Spread Profit (if filled)</div>
          <div className="text-2xl font-bold text-green-400">{analysis.spreadProfit}¢</div>
          <div className="text-xs text-slate-500">per contract</div>
        </div>
        <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-4">
          <div className="mb-2 text-xs text-slate-400">Queue Assessment</div>
          <div className="text-lg text-white">{analysis.queuePosition}</div>
        </div>
        <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-4">
          <div className="mb-2 text-xs text-slate-400">Flow Quality</div>
          <div className="text-lg text-white">{analysis.flowQuality}</div>
        </div>
        <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-4">
          <div className="mb-2 text-xs text-slate-400">Risk Assessment</div>
          <div className="text-lg text-white">{analysis.riskAssessment}</div>
        </div>
      </div>

      <div
        className={`rounded-lg border p-4 ${
          decision === 'take'
            ? 'border-green-700/50 bg-green-900/20'
            : decision === 'pass'
              ? 'border-red-700/50 bg-red-900/20'
              : 'border-amber-700/50 bg-amber-900/20'
        }`}
      >
        <div
          className={`mb-2 text-lg font-bold ${
            decision === 'take'
              ? 'text-green-400'
              : decision === 'pass'
                ? 'text-red-400'
                : 'text-amber-400'
          }`}
        >
          Decision:{' '}
          {decision === 'take' ? 'TAKE THE TRADE' : decision === 'pass' ? 'PASS' : 'CONDITIONAL'}
        </div>
        <div className="space-y-1 text-sm text-slate-300">
          {reasoning.map((r, i) => (
            <p key={i}>• {r}</p>
          ))}
        </div>
      </div>

      <div className="rounded-lg border border-blue-700/50 bg-blue-900/20 p-4">
        <div className="mb-2 font-semibold text-blue-400">What to Watch</div>
        <ul className="space-y-1 text-sm text-slate-300">
          {whatToWatch.map((w, i) => (
            <li key={i}>→ {w}</li>
          ))}
        </ul>
      </div>
    </div>
  );
}

const tradeExamples: TradeExample[] = [
  {
    id: 'good-spread',
    title: 'The Good Spread Opportunity',
    setup: {
      market: 'Lakers vs Celtics - Lakers Moneyline',
      yesPrice: 54,
      noPrice: 44,
      bidAskSpread: 2,
      queueDepth: 150,
      retailFlow: 'Moderate - 20 contracts/min',
      timeToEvent: '2 hours',
    },
    analysis: {
      spreadProfit: 2,
      queuePosition: 'Can get near front with patience',
      flowQuality: 'Good retail flow, mostly small orders',
      riskAssessment: 'Low - 2 hours to adjust if needed',
    },
    decision: 'take',
    reasoning: [
      '2¢ spread is healthy profit margin',
      'Moderate flow means orders will get filled',
      '2 hours to event gives time to manage position',
      'Queue depth is manageable - can get to front within 30 min',
    ],
    whatToWatch: [
      'Queue position as you wait for fills',
      'Any news that might move the price',
      'Flow rate - if it dries up, consider canceling',
    ],
  },
  {
    id: 'tight-spread',
    title: 'The Too-Tight Spread',
    setup: {
      market: 'Heat vs Knicks - Heat Moneyline',
      yesPrice: 62,
      noPrice: 37,
      bidAskSpread: 1,
      queueDepth: 400,
      retailFlow: 'Low - 5 contracts/min',
      timeToEvent: '4 hours',
    },
    analysis: {
      spreadProfit: 1,
      queuePosition: 'Deep queue - would take 80+ min to reach front',
      flowQuality: 'Low flow - fills will be slow',
      riskAssessment: 'Time risk - sitting in queue for hours',
    },
    decision: 'pass',
    reasoning: [
      '1¢ spread is too thin after fees',
      'Deep queue means long wait for uncertain fills',
      'Low flow = high opportunity cost of capital',
      'Better opportunities elsewhere',
    ],
    whatToWatch: [
      'If spread widens to 2¢+, reconsider',
      'If flow picks up closer to game time',
      'If queue thins out',
    ],
  },
  {
    id: 'high-flow',
    title: 'The High-Flow Opportunity',
    setup: {
      market: 'Warriors vs Mavericks - Warriors Moneyline',
      yesPrice: 71,
      noPrice: 27,
      bidAskSpread: 2,
      queueDepth: 50,
      retailFlow: 'High - 100+ contracts/min',
      timeToEvent: '15 min',
    },
    analysis: {
      spreadProfit: 2,
      queuePosition: 'Short queue - fills in seconds',
      flowQuality: 'Excellent - heavy retail before game',
      riskAssessment: 'Medium - fast market, less time to adjust',
    },
    decision: 'take',
    reasoning: [
      'High flow = fast fills',
      'Short queue = immediate execution',
      '15 min to event - classic pre-game rush',
      'Need to be ready to cancel quickly if price moves',
    ],
    whatToWatch: [
      "Position size - don't overload in fast market",
      'Price movements - be ready to cancel instantly',
      'Your total exposure across markets',
    ],
  },
  {
    id: 'lopsided-book',
    title: 'The Lopsided Book',
    setup: {
      market: 'Celtics vs 76ers - Celtics Moneyline',
      yesPrice: 78,
      noPrice: 18,
      bidAskSpread: 4,
      queueDepth: 20,
      retailFlow: 'Low - 3 contracts/min',
      timeToEvent: '1 hour',
    },
    analysis: {
      spreadProfit: 4,
      queuePosition: 'Very short queue',
      flowQuality: 'Low - heavy favorite means less betting interest',
      riskAssessment: 'High - if wrong, you lose 82¢ per contract',
    },
    decision: 'conditional',
    reasoning: [
      '4¢ spread is attractive, BUT...',
      'Low flow means you might not get filled much',
      'Heavy favorite = asymmetric risk (you win 4¢, can lose 82¢)',
      'Only worth it if you can manage position size carefully',
    ],
    whatToWatch: [
      'Position size - keep small due to asymmetric risk',
      'Any upset news that could swing the price',
      'Set strict loss limits',
    ],
  },
  {
    id: 'news-pending',
    title: 'The News-Pending Situation',
    setup: {
      market: 'Nets vs Bulls - Nets Moneyline',
      yesPrice: 48,
      noPrice: 50,
      bidAskSpread: 2,
      queueDepth: 80,
      retailFlow: 'Very Low - waiting for injury report',
      timeToEvent: '3 hours',
    },
    analysis: {
      spreadProfit: 2,
      queuePosition: 'Medium queue',
      flowQuality: 'Stalled - everyone waiting for news',
      riskAssessment: 'Very High - price will jump when news hits',
    },
    decision: 'pass',
    reasoning: [
      'Spread looks fine, but market is frozen for a reason',
      'When injury report drops, price will move fast',
      "If you're in queue, you'll get toxic fills (adverse selection)",
      'Better to wait for news, then re-evaluate',
    ],
    whatToWatch: [
      'The injury report - when it drops, react fast',
      'Any early leaks or rumors',
      'Once news is priced in, then consider entering',
    ],
  },
];

function DecisionFramework() {
  return (
    <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-6">
      <h4 className="mb-4 text-lg font-semibold text-white">The MM Decision Framework</h4>
      <p className="mb-6 text-sm text-slate-400">
        Every trade decision follows this mental checklist. Learn to run through it quickly.
      </p>

      <div className="space-y-4">
        <div className="flex items-start gap-4">
          <div className="flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full bg-purple-600 font-bold text-white">
            1
          </div>
          <div>
            <div className="font-semibold text-white">Is the spread worth it?</div>
            <p className="text-sm text-slate-400">
              After fees (~1-2¢ round trip), is there enough profit margin? Generally need 2¢+
              spread to be worthwhile.
            </p>
            <div className="mt-2 rounded bg-slate-900/50 p-2 text-sm">
              <Formula>{'\\text{Net Profit} = \\text{Spread} - \\text{Fees}'}</Formula>
            </div>
          </div>
        </div>

        <div className="flex items-start gap-4">
          <div className="flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full bg-purple-600 font-bold text-white">
            2
          </div>
          <div>
            <div className="font-semibold text-white">Can I get filled?</div>
            <p className="text-sm text-slate-400">
              What&apos;s the queue depth? What&apos;s the flow rate? How long will I wait? A great
              price doesn&apos;t matter if you never get filled.
            </p>
            <div className="mt-2 rounded bg-slate-900/50 p-2 text-sm">
              <Formula>
                {'\\text{Est. Wait} = \\frac{\\text{Queue Depth}}{\\text{Flow Rate}}'}
              </Formula>
            </div>
          </div>
        </div>

        <div className="flex items-start gap-4">
          <div className="flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full bg-purple-600 font-bold text-white">
            3
          </div>
          <div>
            <div className="font-semibold text-white">What&apos;s my risk?</div>
            <p className="text-sm text-slate-400">
              If price moves against me, how bad is it? How much time do I have to react? Asymmetric
              payoffs (heavy favorites/underdogs) need smaller sizes.
            </p>
            <div className="mt-2 rounded bg-slate-900/50 p-2 text-sm">
              <Formula>
                {'\\text{Max Loss} = \\text{Position} \\times (100 - \\text{Entry Price})'}
              </Formula>
            </div>
          </div>
        </div>

        <div className="flex items-start gap-4">
          <div className="flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full bg-purple-600 font-bold text-white">
            4
          </div>
          <div>
            <div className="font-semibold text-white">Is there information risk?</div>
            <p className="text-sm text-slate-400">
              Is news pending? Could the market move suddenly? If so, you might get adversely
              selected (filled right before bad news).
            </p>
          </div>
        </div>

        <div className="flex items-start gap-4">
          <div className="flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full bg-purple-600 font-bold text-white">
            5
          </div>
          <div>
            <div className="font-semibold text-white">What&apos;s my opportunity cost?</div>
            <p className="text-sm text-slate-400">
              If I tie up capital here, what am I missing elsewhere? Sometimes passing on a decent
              trade frees you for a better one.
            </p>
          </div>
        </div>
      </div>
    </div>
  );
}

function PatternRecognition() {
  const patterns = [
    {
      pattern: 'Wide spread + High flow',
      signal: 'STRONG BUY',
      color: 'green',
      explanation: 'Best case: good profit margin and fast fills',
    },
    {
      pattern: 'Wide spread + Low flow',
      signal: 'CAUTIOUS',
      color: 'amber',
      explanation: 'Good margin but may sit unfilled - size smaller',
    },
    {
      pattern: 'Tight spread + High flow',
      signal: 'MAYBE',
      color: 'amber',
      explanation: 'Thin margins but high volume can make up for it',
    },
    {
      pattern: 'Tight spread + Low flow',
      signal: 'PASS',
      color: 'red',
      explanation: 'Worst case: no margin and no fills',
    },
    {
      pattern: 'Deep queue at your price',
      signal: 'CONSIDER REPRICING',
      color: 'amber',
      explanation: 'You might wait forever - consider worse price for faster fill',
    },
    {
      pattern: 'Short queue + good spread',
      signal: 'STRONG BUY',
      color: 'green',
      explanation: 'Fast fills at good prices - act quickly',
    },
    {
      pattern: 'News pending',
      signal: 'WAIT',
      color: 'red',
      explanation: "Price will jump - don't get adversely selected",
    },
    {
      pattern: 'Heavy favorite (80%+)',
      signal: 'SIZE DOWN',
      color: 'amber',
      explanation: 'Asymmetric risk - small wins, big potential loss',
    },
    {
      pattern: 'Close to event (<30 min)',
      signal: 'HIGH ALERT',
      color: 'amber',
      explanation: 'Fast market - be ready to react quickly',
    },
  ];

  return (
    <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-6">
      <h4 className="mb-4 text-lg font-semibold text-white">Pattern Recognition Cheat Sheet</h4>
      <p className="mb-6 text-sm text-slate-400">
        Experienced MMs recognize these patterns instantly. Train your eye.
      </p>

      <div className="grid gap-3 md:grid-cols-2 lg:grid-cols-3">
        {patterns.map((p) => (
          <div
            key={p.pattern}
            className={`rounded-lg border p-3 ${
              p.color === 'green'
                ? 'border-green-700/50 bg-green-900/20'
                : p.color === 'red'
                  ? 'border-red-700/50 bg-red-900/20'
                  : 'border-amber-700/50 bg-amber-900/20'
            }`}
          >
            <div className="text-sm font-medium text-white">{p.pattern}</div>
            <div
              className={`mt-1 text-xs font-bold ${
                p.color === 'green'
                  ? 'text-green-400'
                  : p.color === 'red'
                    ? 'text-red-400'
                    : 'text-amber-400'
              }`}
            >
              {p.signal}
            </div>
            <div className="mt-1 text-xs text-slate-400">{p.explanation}</div>
          </div>
        ))}
      </div>
    </div>
  );
}

export function MMTradeIdentificationLesson() {
  const [activeExample, setActiveExample] = useState(0);

  return (
    <div className="space-y-8">
      {/* Introduction */}
      <section>
        <SectionHeader
          title="How Market Makers Identify Good Trades"
          subtitle="The decision framework that separates good opportunities from bad ones"
        />
        <LessonRow
          aside={
            <ConstraintBlock
              title="What You'll Learn"
              items={[
                'The 5-question decision framework',
                'Pattern recognition shortcuts',
                'Worked examples with analysis',
                'Speed and timing requirements',
              ]}
            />
          }
        >
          <p className="leading-relaxed text-slate-300">
            A market maker&apos;s job is to constantly evaluate: &quot;Is this opportunity worth
            taking?&quot; They&apos;re not predicting who wins—they&apos;re assessing whether the
            risk/reward of providing liquidity is favorable right now.
          </p>
          <p className="mt-4 leading-relaxed text-slate-300">
            This lesson teaches you to see markets the way MMs see them. By the end, you&apos;ll
            understand the patterns they recognize and the decisions they make.
          </p>
        </LessonRow>
      </section>

      {/* The Decision Framework */}
      <section>
        <SectionHeader
          title="The Decision Framework"
          subtitle="Five questions every MM asks before taking a trade"
        />
        <LessonRow
          aside={
            <InsightBlock title="Systematic, Not Intuitive">
              Good MMs don't rely on gut feel. They run through the same checklist every time.
              That's what makes them consistent.
            </InsightBlock>
          }
        >
          <DecisionFramework />
        </LessonRow>
      </section>

      {/* Pattern Recognition */}
      <section>
        <SectionHeader
          title="Pattern Recognition"
          subtitle="Common situations and what they signal"
        />
        <LessonRow
          aside={
            <TipBlock title="Cheat Sheet">
              Print this pattern list and reference it while practicing. Over time, recognition
              becomes automatic.
            </TipBlock>
          }
        >
          <PatternRecognition />
        </LessonRow>
      </section>

      {/* Worked Examples */}
      <section>
        <SectionHeader
          title="Worked Examples"
          subtitle="Walk through real trade decisions step by step"
        />
        <LessonRow
          aside={
            <WarningBlock title="No Shortcuts">
              Don't skip to the decision. Work through each example using the framework first, then
              compare your reasoning.
            </WarningBlock>
          }
        >
          <p className="mb-4 text-sm text-slate-400">
            Click each example to see the full analysis. Notice how the same framework applies to
            different situations.
          </p>

          <div className="mb-6 flex flex-wrap gap-2">
            {tradeExamples.map((ex, i) => (
              <button
                key={ex.id}
                onClick={() => setActiveExample(i)}
                className={`rounded-lg px-4 py-2 text-sm ${
                  activeExample === i
                    ? 'bg-purple-600 text-white'
                    : 'bg-slate-700 text-slate-300 hover:bg-slate-600'
                }`}
              >
                {ex.title}
              </button>
            ))}
          </div>

          <div className="space-y-6">
            <MarketDisplay example={tradeExamples[activeExample]} />
            <AnalysisDisplay example={tradeExamples[activeExample]} />
          </div>
        </LessonRow>
      </section>

      {/* The Meta-Skill */}
      <section>
        <SectionHeader
          title="The Meta-Skill: Speed"
          subtitle="How fast MMs need to make these decisions"
        />
        <LessonRow
          aside={
            <InsightBlock title="Design Implication">
              If your UI requires more than 1 second to assess a market, you've added too much
              friction. Information at a glance.
            </InsightBlock>
          }
        >
          <p className="leading-relaxed text-slate-300">
            Everything above needs to happen <strong className="text-white">fast</strong>. An
            experienced MM scans a market and decides in seconds, not minutes. The framework becomes
            internalized—pattern recognition, not calculation.
          </p>

          <div className="my-6 rounded-lg border border-slate-700 bg-slate-800/50 p-6">
            <h4 className="mb-4 text-lg font-semibold text-white">Decision Speed by Context</h4>
            <div className="space-y-3">
              <div className="flex items-center gap-4">
                <div className="w-32 text-sm text-slate-400">Slow market scan:</div>
                <div className="h-2 flex-1 rounded-full bg-slate-700">
                  <div
                    className="h-2 rounded-full bg-green-500"
                    style={{ width: '20%' }}
                  />
                </div>
                <div className="w-20 text-sm text-slate-400">5-10 sec</div>
              </div>
              <div className="flex items-center gap-4">
                <div className="w-32 text-sm text-slate-400">Active quoting:</div>
                <div className="h-2 flex-1 rounded-full bg-slate-700">
                  <div
                    className="h-2 rounded-full bg-amber-500"
                    style={{ width: '10%' }}
                  />
                </div>
                <div className="w-20 text-sm text-slate-400">1-2 sec</div>
              </div>
              <div className="flex items-center gap-4">
                <div className="w-32 text-sm text-slate-400">Price move reaction:</div>
                <div className="h-2 flex-1 rounded-full bg-slate-700">
                  <div
                    className="h-2 rounded-full bg-red-500"
                    style={{ width: '5%' }}
                  />
                </div>
                <div className="w-20 text-sm text-slate-400">&lt;1 sec</div>
              </div>
            </div>
          </div>

          <p className="leading-relaxed text-slate-300">
            <strong className="text-white">UI implication:</strong> Your software needs to present
            all decision-relevant information in a single glance. If the MM has to click, scroll, or
            navigate to find information, you&apos;ve failed.
          </p>
        </LessonRow>
      </section>

      {/* Summary */}
      <section>
        <SectionHeader
          title="Key Takeaways"
          subtitle="What to remember about MM trade identification"
        />
        <LessonRow
          aside={
            <TipBlock title="Reference Card">
              Keep these takeaways visible when building your dashboard. They define what
              information needs to be prominent.
            </TipBlock>
          }
        >
          <div className="rounded-lg border border-purple-800/50 bg-gradient-to-r from-purple-900/30 to-blue-900/30 p-6">
            <ul className="space-y-3 text-slate-300">
              <li className="flex gap-2">
                <span className="text-green-400">✓</span>
                <span>
                  <strong className="text-white">Spread is the profit margin.</strong> Need 2¢+
                  after fees to be worthwhile. Tight spreads need high volume to compensate.
                </span>
              </li>
              <li className="flex gap-2">
                <span className="text-green-400">✓</span>
                <span>
                  <strong className="text-white">Queue + Flow = Fill probability.</strong> Deep
                  queue and low flow means you&apos;ll wait forever. Short queue + high flow is
                  ideal.
                </span>
              </li>
              <li className="flex gap-2">
                <span className="text-green-400">✓</span>
                <span>
                  <strong className="text-white">Information risk is real.</strong> Pending news
                  means the market can move against you instantly. Wait for clarity.
                </span>
              </li>
              <li className="flex gap-2">
                <span className="text-green-400">✓</span>
                <span>
                  <strong className="text-white">Heavy favorites have asymmetric risk.</strong> You
                  win small, can lose big. Size down on lopsided markets.
                </span>
              </li>
              <li className="flex gap-2">
                <span className="text-green-400">✓</span>
                <span>
                  <strong className="text-white">Pattern recognition beats calculation.</strong>{' '}
                  Train your eye to spot good/bad setups instantly. Speed matters.
                </span>
              </li>
            </ul>
          </div>
        </LessonRow>
      </section>

      {/* Transition to Practice */}
      <section>
        <SectionHeader
          title="Ready to Practice?"
          subtitle="Test your pattern recognition"
        />
        <LessonRow>
          <div className="rounded-lg border border-purple-700/50 bg-purple-900/20 p-6 text-center">
            <p className="mb-4 text-slate-300">
              Now that you understand the framework, practice applying it. The next lesson presents
              market scenarios for you to evaluate—no hints, just you and the data.
            </p>
            <Link
              to="/mm-practice-quiz"
              className="inline-block font-semibold text-purple-400 transition-colors hover:text-purple-300"
            >
              → Continue to: MM Trade Identification Practice
            </Link>
          </div>
        </LessonRow>
      </section>
    </div>
  );
}
