import { useState } from 'react';
import { LessonRow, SectionHeader } from '../LessonLayout';
import { TipBlock, InsightBlock, WarningBlock, ConstraintBlock } from '../blocks';

/**
 * Day-in-the-Life Scenarios for Market Makers
 *
 * Shows different situations an MM encounters and how they respond.
 * Helps product builders understand different "modes" the UI needs to support.
 */

type TimelineEvent = {
  time: string;
  event: string;
  mmThinks: string;
  mmDoes: string;
  uiNeeds: string;
};

type Scenario = {
  id: string;
  title: string;
  description: string;
  context: string;
  timeline: TimelineEvent[];
  keyInsight: string;
  uiImplications: string[];
};

function ScenarioViewer({ scenario }: { scenario: Scenario }) {
  const [activeEvent, setActiveEvent] = useState(0);

  return (
    <div className="overflow-hidden rounded-lg border border-slate-700 bg-slate-800/50">
      {/* Header */}
      <div className="border-b border-slate-700 bg-slate-900/50 p-4">
        <h4 className="text-lg font-semibold text-white">{scenario.title}</h4>
        <p className="mt-1 text-sm text-slate-400">{scenario.description}</p>
      </div>

      {/* Context */}
      <div className="border-b border-slate-700 bg-blue-900/20 p-4">
        <div className="mb-1 text-xs font-medium text-blue-400">CONTEXT</div>
        <p className="text-sm text-slate-300">{scenario.context}</p>
      </div>

      {/* Timeline */}
      <div className="p-4">
        <div className="mb-3 text-xs font-medium text-slate-400">TIMELINE</div>
        <div className="mb-4 flex gap-2 overflow-x-auto pb-2">
          {scenario.timeline.map((event, i) => (
            <button
              key={i}
              onClick={() => setActiveEvent(i)}
              className={`whitespace-nowrap rounded-lg px-3 py-2 text-sm ${
                activeEvent === i
                  ? 'bg-purple-600 text-white'
                  : 'bg-slate-700 text-slate-300 hover:bg-slate-600'
              }`}
            >
              {event.time}
            </button>
          ))}
        </div>

        {/* Active event details */}
        <div className="space-y-3">
          <div className="rounded-lg bg-slate-900/50 p-4">
            <div className="mb-1 text-xs text-slate-400">What happens:</div>
            <p className="text-white">{scenario.timeline[activeEvent].event}</p>
          </div>

          <div className="grid gap-3 md:grid-cols-2">
            <div className="rounded-lg border border-amber-700/50 bg-amber-900/20 p-4">
              <div className="mb-1 text-xs text-amber-400">MM thinks:</div>
              <p className="text-sm text-slate-300">{scenario.timeline[activeEvent].mmThinks}</p>
            </div>
            <div className="rounded-lg border border-green-700/50 bg-green-900/20 p-4">
              <div className="mb-1 text-xs text-green-400">MM does:</div>
              <p className="text-sm text-slate-300">{scenario.timeline[activeEvent].mmDoes}</p>
            </div>
          </div>

          <div className="rounded-lg border border-purple-700/50 bg-purple-900/20 p-4">
            <div className="mb-1 text-xs text-purple-400">UI needs to support:</div>
            <p className="text-sm text-slate-300">{scenario.timeline[activeEvent].uiNeeds}</p>
          </div>
        </div>
      </div>

      {/* Key insight */}
      <div className="border-t border-slate-700 bg-slate-900/50 p-4">
        <div className="mb-2 text-xs font-medium text-slate-400">KEY INSIGHT FOR BUILDERS</div>
        <p className="text-slate-300">{scenario.keyInsight}</p>
      </div>

      {/* UI Implications */}
      <div className="border-t border-slate-700 p-4">
        <div className="mb-2 text-xs font-medium text-slate-400">UI IMPLICATIONS</div>
        <ul className="space-y-1">
          {scenario.uiImplications.map((imp, i) => (
            <li
              key={i}
              className="flex gap-2 text-sm text-slate-300"
            >
              <span className="text-purple-400">→</span>
              {imp}
            </li>
          ))}
        </ul>
      </div>
    </div>
  );
}

const scenarios: Scenario[] = [
  {
    id: 'slow-market',
    title: 'The Slow Market',
    description: 'No retail flow, wide spreads, nothing happening',
    context:
      "It's 2pm on a Tuesday. The Lakers-Celtics game is tonight at 7pm. The market has been quiet all morning. Spreads are wide, volume is low.",
    timeline: [
      {
        time: '2:00 PM',
        event: 'MM checks the market. Best bid on Lakers NO is 44¢, best ask is 48¢. 4¢ spread.',
        mmThinks:
          'Wide spread means opportunity, but also means low activity. I could post at 45¢ and try to capture the spread.',
        mmDoes: 'Posts a bid at 45¢ for 50 contracts on Lakers NO. Joins the queue.',
        uiNeeds:
          'Quick view of current spread. One-click order entry. Confirmation of order placed and queue position.',
      },
      {
        time: '2:15 PM',
        event: 'Still no fills. Queue position is #12 out of 200 contracts at 45¢.',
        mmThinks:
          "I'm near the front of the queue, which is good. But no flow means I might sit here for hours. Is this worth my time?",
        mmDoes: 'Monitors passively. Maybe opens other markets to check for better opportunities.',
        uiNeeds:
          'Passive monitoring mode. Multi-market view. Ability to see queue position without clicking.',
      },
      {
        time: '3:30 PM',
        event: 'A small retail order comes in. 20 contracts filled at market. MM gets 12 of them.',
        mmThinks:
          'Finally some flow. I got filled at 45¢. If I think fair value is 46¢, I just made 1¢ per contract. Not much, but it adds up.',
        mmDoes: 'Checks position. Decides whether to re-quote immediately or wait.',
        uiNeeds:
          'Clear fill notification. Updated position display. Updated P&L. Quick ability to re-post order.',
      },
      {
        time: '4:00 PM',
        event: 'Market still quiet. MM has 12 contracts of Lakers NO position.',
        mmThinks:
          'I have a small position. Do I want to add more? The game is in 3 hours. Closer to game time usually brings more flow.',
        mmDoes: 'Posts another order at 45¢. Continues passive monitoring.',
        uiNeeds: 'Position summary. Time until event. Historical flow patterns (nice-to-have).',
      },
    ],
    keyInsight:
      "In slow markets, the MM is in 'passive monitoring' mode. The UI needs to support low-attention watching of multiple markets simultaneously. They're not actively clicking—they're waiting.",
    uiImplications: [
      'Multi-market dashboard that updates without interaction',
      'Visual indicators when flow picks up (highlight, sound)',
      'Queue position visible at a glance',
      'Time-until-event countdown',
    ],
  },
  {
    id: 'fast-market',
    title: 'The Fast Market',
    description: 'High volume, tight spreads, rapid flow',
    context:
      "It's 6:45pm, 15 minutes before Lakers-Celtics tip-off. Retail bettors are flooding in with last-minute bets. Volume is 10x normal.",
    timeline: [
      {
        time: '6:45 PM',
        event:
          "Orders flying in. Spread is tight (1¢). MM's orders are getting filled every few seconds.",
        mmThinks:
          "This is the action I've been waiting for. Tight spread means less profit per trade, but high volume makes up for it. Need to stay focused.",
        mmDoes: 'Actively re-quoting after every fill. Monitoring position size closely.',
        uiNeeds:
          'Instant fill notifications. One-key re-quote. Real-time position tracking. Speed is everything.',
      },
      {
        time: '6:47 PM',
        event: "MM's position is growing. Now has 200 contracts of Lakers NO.",
        mmThinks:
          'Position is getting large. If Lakers win, I lose money on this inventory. Need to manage risk—maybe widen my quote or pause.',
        mmDoes: 'Pulls orders temporarily to assess. Checks exposure across all markets.',
        uiNeeds: 'Total exposure view. One-click cancel all. Position limits/warnings.',
      },
      {
        time: '6:50 PM',
        event: 'A large retail order sweeps through. 500 contracts at market.',
        mmThinks:
          "Big sweep! That was aggressive. Someone really wants in. This might signal information—or just a whale. Either way, I'm filled on a lot.",
        mmDoes: 'Checks tape to understand the flow. Re-evaluates fair value. May adjust quote.',
        uiNeeds:
          "Tape highlighting large orders. Ability to see who's aggressive. Quick price adjustment.",
      },
      {
        time: '6:55 PM',
        event: 'Game about to start. Flow is slowing as market approaches close.',
        mmThinks: 'Time to wind down. I have my position. Now I just wait for the game result.',
        mmDoes: 'Pulls remaining orders. Accepts current position.',
        uiNeeds: "Cancel all orders quickly. Position summary. Clear 'end of trading' state.",
      },
    ],
    keyInsight:
      "In fast markets, the MM is in 'active trading' mode. Speed is critical. They need to react in seconds, not minutes. The UI must minimize friction for every action.",
    uiImplications: [
      'Keyboard shortcuts for all actions',
      'Sub-second order execution',
      'Real-time position and P&L updates',
      'Audio alerts for large fills',
      'One-click cancel-all button',
    ],
  },
  {
    id: 'price-move',
    title: 'The Price Move Against',
    description: 'Market moves before the MM can adjust',
    context:
      'MM has a position in Lakers NO at an average price of 45¢. Suddenly, news breaks that LeBron is playing (was questionable). Lakers YES jumps from 55¢ to 62¢.',
    timeline: [
      {
        time: '5:30 PM',
        event: 'News breaks: LeBron confirmed to play. Lakers YES immediately jumps to 62¢.',
        mmThinks:
          "My Lakers NO position just got worse. Fair value moved against me. Lakers NO should now be around 38¢, not 45¢. I'm underwater on this position.",
        mmDoes: 'Immediately checks position P&L. Assesses damage.',
        uiNeeds:
          'Real-time P&L updates. Alert when prices move significantly. Clear indication of position vs. current market.',
      },
      {
        time: '5:31 PM',
        event: "MM's existing orders at 45¢ are now way off-market. Getting toxic fills.",
        mmThinks:
          "Anyone selling to me at 45¢ knows they're getting a great deal. These are informed sellers. I need to cancel NOW.",
        mmDoes: 'Frantically cancels all orders. Stops the bleeding.',
        uiNeeds:
          'ONE CLICK cancel all. Immediate order cancellation. No confirmation dialogs—speed matters.',
      },
      {
        time: '5:32 PM',
        event: 'Orders cancelled. MM is stuck with a position at a bad price.',
        mmThinks:
          'Okay, bleeding stopped. I have a loss on this position. Do I hold and hope Lakers lose, or cut and move on?',
        mmDoes: 'Evaluates: is there still edge here? Decides whether to exit or hold.',
        uiNeeds: 'Clear P&L by position. Exit buttons. Current market price vs. my average entry.',
      },
      {
        time: '5:35 PM',
        event: 'MM decides to hold the position and adjust quotes to new market.',
        mmThinks:
          "I'll hold. Lakers might still lose. I'll adjust my quotes to the new reality and try to make some of it back.",
        mmDoes: 'Posts new orders at appropriate prices (around 38-40¢ for Lakers NO).',
        uiNeeds:
          'Quick re-pricing. Suggested fair value based on YES price. Easy order modification.',
      },
    ],
    keyInsight:
      'Price moves are the most dangerous moment for MMs. The UI needs to support instant reaction—cancel everything, assess damage, and recover. Every second of delay costs money.',
    uiImplications: [
      'PROMINENT cancel-all button (or hotkey)',
      'Price alerts when market moves more than X%',
      'Real-time P&L that updates instantly',
      "Visual warning when orders are 'off-market'",
      'No confirmation dialogs for cancels',
    ],
  },
  {
    id: 'fat-finger',
    title: 'The Fat-Finger Error',
    description: 'MM accidentally submits wrong order',
    context:
      'MM intends to post a bid at 45¢ for 50 contracts. Accidentally types 54¢ instead of 45¢. This order would buy at a terrible price.',
    timeline: [
      {
        time: '3:00 PM',
        event: "MM types '54' instead of '45' and hits enter. Order submitted at 54¢.",
        mmThinks: "Wait, that wasn't right. I meant 45, not 54. Oh no.",
        mmDoes: 'Immediately tries to cancel.',
        uiNeeds:
          "If order hasn't been matched yet, instant cancel. Clear order confirmation before submit.",
      },
      {
        time: '3:00:01 PM',
        event: 'Order at 54¢ gets immediately filled (it was above the ask).',
        mmThinks:
          "Too late. I just bought at 54¢ when I should have bought at 45¢. That's a 9¢ loss per contract.",
        mmDoes: 'Accepts the mistake. Logs it. Considers how to avoid next time.',
        uiNeeds: 'Accurate fill reporting. P&L impact shown immediately.',
      },
      {
        time: '3:01 PM',
        event: 'MM has 50 contracts at a bad price. Decides what to do.',
        mmThinks:
          'I made a mistake. Do I exit immediately and lock in the loss, or hold and hope the market moves my way?',
        mmDoes: 'Likely exits at market to limit damage and move on.',
        uiNeeds: 'Quick exit button. Clear loss display. Ability to move on.',
      },
    ],
    keyInsight:
      'Fat-finger errors are inevitable. The best defense is prevention (price sanity checks, clear confirmation), not confirmation dialogs (too slow). The UI should catch obvious errors before submission.',
    uiImplications: [
      'Price sanity check: warn if order is significantly off-market',
      'Clear visual preview of order before submit',
      'Highlight when price entered is on wrong side of spread',
      "Consider 'undo' for very recent orders (if not yet filled)",
      'DO NOT add confirmation dialogs—too slow',
    ],
  },
  {
    id: 'end-of-session',
    title: 'End of Session Wind-Down',
    description: 'Closing out the trading day',
    context:
      "It's 10pm. Games are finishing. MM needs to close out positions and assess performance.",
    timeline: [
      {
        time: '10:00 PM',
        event: "Last game just ended. Lakers won. MM's positions are settling.",
        mmThinks:
          'Okay, Lakers won. My Lakers NO positions lost. Let me see the total damage and how the night went overall.',
        mmDoes: 'Reviews P&L across all markets. Cancels any remaining open orders.',
        uiNeeds: 'End-of-day P&L summary. All positions with outcomes. Easy bulk cancel.',
      },
      {
        time: '10:05 PM',
        event: 'Settlements processing. Some positions paying out, others worthless.',
        mmThinks:
          'Net P&L for tonight: +$340. Not bad. The Celtics game went well, covered the Lakers loss.',
        mmDoes: 'Reviews trade log. Notes any mistakes or learnings.',
        uiNeeds: 'Trade history. P&L breakdown by market. Export capability for record-keeping.',
      },
      {
        time: '10:15 PM',
        event: 'MM logs off for the night.',
        mmThinks: "Good session. I'll review tomorrow and think about what to adjust.",
        mmDoes: "Closes the application. Maybe sets alerts for tomorrow's markets.",
        uiNeeds: 'Session summary. Ability to set alerts for next day. Clean logout.',
      },
    ],
    keyInsight:
      "End-of-session is about review and record-keeping. The MM switches from 'trading mode' to 'analysis mode'. Different information needs—more about history and summary, less about real-time action.",
    uiImplications: [
      'Daily P&L summary view',
      'Trade history with filters',
      'Performance metrics over time',
      'Export to CSV/spreadsheet',
      'Saved settings and alerts for next session',
    ],
  },
];

function ModeComparison() {
  const modes = [
    {
      name: 'Passive Monitoring',
      when: 'Slow markets, waiting for flow',
      attention: 'Low',
      actions: 'Occasional',
      uiMode: 'Dashboard/overview',
      keyFeatures: ['Multi-market view', 'Alerts when flow picks up', 'Queue position at a glance'],
    },
    {
      name: 'Active Trading',
      when: 'Fast markets, high volume',
      attention: 'High',
      actions: 'Continuous',
      uiMode: 'Single-market focus',
      keyFeatures: ['Keyboard shortcuts', 'Instant execution', 'Real-time everything'],
    },
    {
      name: 'Crisis Response',
      when: 'Price moves against, errors',
      attention: 'Critical',
      actions: 'Immediate',
      uiMode: 'Emergency controls',
      keyFeatures: ['Cancel-all button', 'Position exit', 'No dialogs'],
    },
    {
      name: 'Analysis/Review',
      when: 'End of session, planning',
      attention: 'Reflective',
      actions: 'Browsing',
      uiMode: 'Reports/history',
      keyFeatures: ['P&L summaries', 'Trade history', 'Performance charts'],
    },
  ];

  return (
    <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-6">
      <h4 className="mb-4 text-lg font-semibold text-white">The Four UI Modes</h4>
      <p className="mb-6 text-sm text-slate-400">
        Your UI needs to support all four of these modes. The same user shifts between them
        throughout a session.
      </p>

      <div className="grid gap-4 md:grid-cols-2">
        {modes.map((mode) => (
          <div
            key={mode.name}
            className="rounded-lg border border-slate-600 bg-slate-900/50 p-4"
          >
            <div className="mb-2 flex items-start justify-between">
              <h5 className="font-semibold text-white">{mode.name}</h5>
              <span
                className={`rounded px-2 py-1 text-xs ${
                  mode.attention === 'Critical'
                    ? 'bg-red-900/50 text-red-400'
                    : mode.attention === 'High'
                      ? 'bg-amber-900/50 text-amber-400'
                      : mode.attention === 'Low'
                        ? 'bg-green-900/50 text-green-400'
                        : 'bg-blue-900/50 text-blue-400'
                }`}
              >
                {mode.attention} attention
              </span>
            </div>
            <p className="mb-3 text-sm text-slate-400">{mode.when}</p>
            <div className="mb-2 text-xs text-slate-500">Key features needed:</div>
            <ul className="space-y-1">
              {mode.keyFeatures.map((f, i) => (
                <li
                  key={i}
                  className="text-sm text-slate-300"
                >
                  • {f}
                </li>
              ))}
            </ul>
          </div>
        ))}
      </div>
    </div>
  );
}

export function MarketMakerScenariosLesson() {
  const [activeScenario, setActiveScenario] = useState(0);

  return (
    <div className="space-y-8">
      {/* Introduction */}
      <section>
        <SectionHeader
          title="Day-in-the-Life Scenarios"
          subtitle="Understanding what market makers actually experience"
        />
        <LessonRow
          aside={
            <ConstraintBlock
              title="Scenarios Covered"
              items={[
                'Slow Market (passive monitoring)',
                'Fast Market (active trading)',
                'Price Move Against (crisis response)',
                'Fat-Finger Error (error recovery)',
                'End of Session (wind-down)',
              ]}
            />
          }
        >
          <p className="leading-relaxed text-slate-300">
            To build great trading software, you need to understand not just <em>what</em> your
            users do, but <em>when</em> and <em>why</em>. A market maker&apos;s day isn&apos;t
            uniform—they shift between different modes depending on market conditions.
          </p>
          <p className="mt-4 leading-relaxed text-slate-300">
            This lesson walks through real scenarios showing how market conditions change the
            workflow, what decisions get made, and what the UI needs to support in each case.
          </p>
        </LessonRow>
      </section>

      {/* Mode Overview */}
      <section>
        <SectionHeader
          title="The Four UI Modes"
          subtitle="Your software needs to support all of these"
        />
        <LessonRow
          aside={
            <InsightBlock title="Mode-Switching">
              MMs don't navigate to different screens—they stay on one screen and their mode changes
              based on market conditions.
            </InsightBlock>
          }
        >
          <ModeComparison />
        </LessonRow>
      </section>

      {/* Scenario Selector */}
      <section>
        <SectionHeader
          title="Scenario Walkthroughs"
          subtitle="Click through each scenario to understand the workflow"
        />
        <LessonRow
          aside={
            <TipBlock title="Read Actively">
              For each scenario, note what data the MM needs and when. This maps directly to UI
              requirements.
            </TipBlock>
          }
        >
          <div className="mb-6 flex flex-wrap gap-2">
            {scenarios.map((scenario, i) => (
              <button
                key={scenario.id}
                onClick={() => setActiveScenario(i)}
                className={`rounded-lg px-4 py-2 text-sm ${
                  activeScenario === i
                    ? 'bg-purple-600 text-white'
                    : 'bg-slate-700 text-slate-300 hover:bg-slate-600'
                }`}
              >
                {scenario.title}
              </button>
            ))}
          </div>
          <ScenarioViewer scenario={scenarios[activeScenario]} />
        </LessonRow>
      </section>

      {/* Summary */}
      <section>
        <SectionHeader
          title="Key Takeaways for Builders"
          subtitle="What these scenarios tell you about UI design"
        />
        <LessonRow
          aside={
            <WarningBlock title="Test Under Stress">
              Test your UI during fast market conditions, not just slow ones. Usability breaks down
              when speed matters most.
            </WarningBlock>
          }
        >
          <div className="space-y-4">
            <div className="rounded-lg border border-purple-700/50 bg-purple-900/20 p-4">
              <h4 className="mb-2 font-semibold text-purple-400">
                1. The UI is not one-size-fits-all
              </h4>
              <p className="text-sm text-slate-300">
                Different market conditions require different UI modes. Passive monitoring needs a
                dashboard. Active trading needs speed. Crises need emergency controls. Design for
                all four modes.
              </p>
            </div>

            <div className="rounded-lg border border-purple-700/50 bg-purple-900/20 p-4">
              <h4 className="mb-2 font-semibold text-purple-400">
                2. Speed requirements vary dramatically
              </h4>
              <p className="text-sm text-slate-300">
                In slow markets, a 2-second delay doesn&apos;t matter. In fast markets or during
                price moves, 2 seconds can cost real money. The UI must be fast enough for the worst
                case.
              </p>
            </div>

            <div className="rounded-lg border border-purple-700/50 bg-purple-900/20 p-4">
              <h4 className="mb-2 font-semibold text-purple-400">
                3. Cancel-all is a critical safety feature
              </h4>
              <p className="text-sm text-slate-300">
                When things go wrong, the MM needs to stop everything instantly. This button (or
                hotkey) should be prominent and fast. No confirmation dialogs.
              </p>
            </div>

            <div className="rounded-lg border border-purple-700/50 bg-purple-900/20 p-4">
              <h4 className="mb-2 font-semibold text-purple-400">4. Context transitions matter</h4>
              <p className="text-sm text-slate-300">
                The shift from &quot;nothing happening&quot; to &quot;everything happening&quot; can
                occur in seconds. The UI should help the user transition smoothly—alerts when flow
                picks up, easy focus on a single market when needed.
              </p>
            </div>

            <div className="rounded-lg border border-purple-700/50 bg-purple-900/20 p-4">
              <h4 className="mb-2 font-semibold text-purple-400">
                5. Post-session is a different product
              </h4>
              <p className="text-sm text-slate-300">
                End-of-day review is more like an analytics tool than a trading tool. Different
                information architecture, different pace. Could be a separate view or even a
                separate interface.
              </p>
            </div>
          </div>
        </LessonRow>
      </section>
    </div>
  );
}
