/**
 * Lesson: Reading a Sportsbook Odds Screen
 *
 * Comprehensive guide to understanding traditional sportsbook interfaces,
 * including how to decode the numbers, columns, and formatting.
 * Includes an interactive mockup of what we want to build.
 */

import { useState, Fragment } from 'react';
import { Monitor, Hash, ArrowUpDown, Trophy } from 'lucide-react';
import { LessonLayout, LessonRow, LessonHeader, SectionHeader } from '../LessonLayout';
import {
  ObjectiveBlock,
  ConstraintBlock,
  TipBlock,
  InsightBlock,
  WarningBlock,
  ConceptCard,
  ExampleBox,
  Term,
} from '../blocks';
import { cn } from '@/lib/utils';

// =============================================================================
// PM-STYLE MOCKUP (Multiple Strikes per Game)
// =============================================================================

/**
 * THE KEY INSIGHT (from Jon):
 *
 * Traditional sportsbooks: ONE line per game is relevant
 *   - Vegas sets "Lakers -6.5" and all books use that same line
 *   - You're just shopping for the best PRICE on that ONE line
 *
 * Prediction markets: MULTIPLE lines are ALL relevant
 *   - Kalshi offers: -3.5, -4.5, -5.5, -6.5, -7.5, -8.5...
 *   - Each is a separate market with its own price
 *   - A trader might prefer "Lakers -4.5 at 62¢" over "Lakers -6.5 at 52¢"
 *   - ALL lines are tradeable opportunities
 *
 * Solution: Show "main" line (closest to 50¢) + expand to see all others
 */

interface Strike {
  strike: number; // e.g., 220.5 for totals, -6.5 for spreads
  yesPrice: number; // cents (0-100) - "Over" for totals, "Covers" for spreads
  noPrice: number; // cents (0-100) - "Under" for totals, "Doesn't Cover" for spreads
  yesLiq: number;
  noLiq: number;
  isMain?: boolean; // closest to 50¢
}

interface PMGame {
  date: string;
  time: string;
  awayTeam: string;
  homeTeam: string;
  awayCode: string;
  homeCode: string;
  // Moneyline markets
  moneyline: {
    awayPrice: number;
    homePrice: number;
    awayLiq: number;
    homeLiq: number;
  };
  // All total strikes for this game (Over/Under combined score)
  totalStrikes: Strike[];
  // All spread strikes for this game (point spread on favorite)
  spreadStrikes: Strike[];
  favorite: 'away' | 'home'; // which team is favored
}

const PM_MOCK_GAMES: PMGame[] = [
  {
    date: '02/02',
    time: '4:10 PM',
    awayTeam: 'Milwaukee Bucks',
    homeTeam: 'San Antonio Spurs',
    awayCode: 'MIL',
    homeCode: 'SAS',
    favorite: 'away',
    moneyline: { awayPrice: 72, homePrice: 28, awayLiq: 5420, homeLiq: 3210 },
    totalStrikes: [
      { strike: 211.5, yesPrice: 92, noPrice: 8, yesLiq: 12, noLiq: 295 },
      { strike: 214.5, yesPrice: 65, noPrice: 35, yesLiq: 186, noLiq: 802 },
      { strike: 217.5, yesPrice: 58, noPrice: 42, yesLiq: 320, noLiq: 1100 },
      { strike: 220.5, yesPrice: 52, noPrice: 48, yesLiq: 890, noLiq: 920, isMain: true },
      { strike: 223.5, yesPrice: 45, noPrice: 55, yesLiq: 750, noLiq: 680 },
      { strike: 226.5, yesPrice: 38, noPrice: 62, yesLiq: 420, noLiq: 510 },
      { strike: 229.5, yesPrice: 28, noPrice: 72, yesLiq: 180, noLiq: 340 },
    ],
    spreadStrikes: [
      { strike: -3.5, yesPrice: 82, noPrice: 18, yesLiq: 45, noLiq: 120 },
      { strike: -5.5, yesPrice: 68, noPrice: 32, yesLiq: 210, noLiq: 380 },
      { strike: -7.5, yesPrice: 52, noPrice: 48, yesLiq: 620, noLiq: 580, isMain: true },
      { strike: -9.5, yesPrice: 38, noPrice: 62, yesLiq: 340, noLiq: 720 },
      { strike: -11.5, yesPrice: 25, noPrice: 75, yesLiq: 150, noLiq: 890 },
    ],
  },
  {
    date: '02/02',
    time: '7:00 PM',
    awayTeam: 'Boston Celtics',
    homeTeam: 'Miami Heat',
    awayCode: 'BOS',
    homeCode: 'MIA',
    favorite: 'away',
    moneyline: { awayPrice: 65, homePrice: 35, awayLiq: 8900, homeLiq: 4200 },
    totalStrikes: [
      { strike: 205.5, yesPrice: 78, noPrice: 22, yesLiq: 45, noLiq: 180 },
      { strike: 208.5, yesPrice: 62, noPrice: 38, yesLiq: 210, noLiq: 450 },
      { strike: 211.5, yesPrice: 51, noPrice: 49, yesLiq: 680, noLiq: 710, isMain: true },
      { strike: 214.5, yesPrice: 42, noPrice: 58, yesLiq: 520, noLiq: 890 },
      { strike: 217.5, yesPrice: 31, noPrice: 69, yesLiq: 290, noLiq: 1200 },
    ],
    spreadStrikes: [
      { strike: -2.5, yesPrice: 72, noPrice: 28, yesLiq: 180, noLiq: 290 },
      { strike: -4.5, yesPrice: 55, noPrice: 45, yesLiq: 450, noLiq: 520 },
      { strike: -5.5, yesPrice: 48, noPrice: 52, yesLiq: 720, noLiq: 680, isMain: true },
      { strike: -6.5, yesPrice: 41, noPrice: 59, yesLiq: 380, noLiq: 610 },
      { strike: -8.5, yesPrice: 28, noPrice: 72, yesLiq: 120, noLiq: 450 },
    ],
  },
  {
    date: '02/02',
    time: '7:30 PM',
    awayTeam: 'New York Knicks',
    homeTeam: 'Golden State Warriors',
    awayCode: 'NYK',
    homeCode: 'GSW',
    favorite: 'home',
    moneyline: { awayPrice: 45, homePrice: 55, awayLiq: 6100, homeLiq: 7800 },
    totalStrikes: [
      { strike: 218.5, yesPrice: 71, noPrice: 29, yesLiq: 120, noLiq: 380 },
      { strike: 221.5, yesPrice: 59, noPrice: 41, yesLiq: 340, noLiq: 520 },
      { strike: 224.5, yesPrice: 48, noPrice: 52, yesLiq: 780, noLiq: 820, isMain: true },
      { strike: 227.5, yesPrice: 38, noPrice: 62, yesLiq: 450, noLiq: 690 },
      { strike: 230.5, yesPrice: 25, noPrice: 75, yesLiq: 180, noLiq: 950 },
    ],
    spreadStrikes: [
      { strike: -1.5, yesPrice: 68, noPrice: 32, yesLiq: 290, noLiq: 410 },
      { strike: -2.5, yesPrice: 58, noPrice: 42, yesLiq: 480, noLiq: 520 },
      { strike: -3.5, yesPrice: 49, noPrice: 51, yesLiq: 650, noLiq: 620, isMain: true },
      { strike: -4.5, yesPrice: 41, noPrice: 59, yesLiq: 380, noLiq: 540 },
      { strike: -5.5, yesPrice: 32, noPrice: 68, yesLiq: 180, noLiq: 720 },
    ],
  },
];

/**
 * Combined mockup showing Totals, Spreads, and Moneylines
 *
 * KEY REQUIREMENTS (from Jon/Professor):
 * - "Main" market = the strike closest to 50¢ (even odds)
 * - Main market is dynamic — updates if a different strike becomes closest to 50¢
 * - Expandable rows are optional (traders rarely use alternate strikes)
 *   but "fine with it" if simple to add (per Professor)
 */
function PMTotalsMockup() {
  const [expandedGames, setExpandedGames] = useState<Set<string>>(new Set());
  const [viewMode, setViewMode] = useState<'totals' | 'spreads' | 'moneylines'>('totals');

  const toggleExpand = (gameKey: string) => {
    setExpandedGames((prev) => {
      const next = new Set(prev);
      if (next.has(gameKey)) {
        next.delete(gameKey);
      } else {
        next.add(gameKey);
      }
      return next;
    });
  };

  // Convert cents to American odds format
  // 50¢ = even odds
  // >50¢ = favorite (negative odds): -(cents / (100 - cents)) * 100
  // <50¢ = underdog (positive odds): +((100 - cents) / cents) * 100
  const formatPrice = (cents: number) => {
    if (cents === 50) return '+100';
    if (cents > 50) {
      const odds = Math.round(-(cents / (100 - cents)) * 100);
      return `${odds}`;
    }
    const odds = Math.round(((100 - cents) / cents) * 100);
    return `+${odds}`;
  };

  const formatLiq = (contracts: number) => {
    if (contracts >= 1000) return `$${((contracts * 50) / 1000).toFixed(1)}K`;
    return `$${Math.round(contracts * 0.5)}`;
  };

  // Get appropriate strikes based on view mode
  const getStrikes = (game: PMGame) => {
    return viewMode === 'totals' ? game.totalStrikes : game.spreadStrikes;
  };

  const isExpandable = viewMode === 'totals' || viewMode === 'spreads';

  return (
    <div className="overflow-hidden rounded-lg border bg-slate-900 text-slate-100">
      {/* Header */}
      <div className="flex items-center gap-2 border-b border-slate-700 bg-slate-800 px-3 py-2">
        <span className="text-sm font-semibold">NBA Value Dashboard</span>
        <div className="flex-1" />
        <div className="flex gap-1">
          <button
            onClick={() => setViewMode('spreads')}
            className={cn(
              'rounded px-2 py-1 text-xs',
              viewMode === 'spreads' ? 'bg-blue-600' : 'bg-slate-700'
            )}
          >
            Spreads
          </button>
          <button
            onClick={() => setViewMode('totals')}
            className={cn(
              'rounded px-2 py-1 text-xs',
              viewMode === 'totals' ? 'bg-blue-600' : 'bg-slate-700'
            )}
          >
            Totals
          </button>
          <button
            onClick={() => setViewMode('moneylines')}
            className={cn(
              'rounded px-2 py-1 text-xs',
              viewMode === 'moneylines' ? 'bg-blue-600' : 'bg-slate-700'
            )}
          >
            Moneylines
          </button>
        </div>
      </div>

      {/* Explanation banner */}
      <div className="border-b border-blue-800 bg-blue-900/30 px-3 py-2 text-xs text-blue-200">
        {viewMode === 'moneylines' && (
          <>
            <strong>Moneylines:</strong> Simple "who wins" bets. One market per team. Odds shown in
            American format (-150 = bet $150 to win $100, +150 = bet $100 to win $150).
          </>
        )}
        {viewMode === 'totals' && (
          <>
            <strong>PM Approach:</strong> Shows the "main" line (closest to even odds) for each
            game. <span className="text-amber-300">Click a row</span> to see all 7 available
            strikes. <strong>Over</strong> = combined score exceeds line. <strong>Under</strong> =
            combined score below line.
          </>
        )}
        {viewMode === 'spreads' && (
          <>
            <strong>PM Approach:</strong> Shows the "main" spread (closest to even odds) for each
            game. <span className="text-amber-300">Click a row</span> to see all 5 available
            strikes. <strong>Cover</strong> = favorite wins by more than spread.{' '}
            <strong>Miss</strong> = favorite fails to cover.
          </>
        )}
      </div>

      {/* Table */}
      <div className="overflow-x-auto">
        <table className="w-full text-xs">
          <thead>
            <tr className="border-b border-slate-700 bg-slate-800/50">
              <th className="w-8 px-3 py-2 text-left font-medium text-slate-400"></th>
              <th className="px-3 py-2 text-left font-medium text-slate-400">Date</th>
              <th className="px-3 py-2 text-left font-medium text-slate-400">Time</th>
              <th className="px-3 py-2 text-left font-medium text-slate-400">Matchup</th>
              {viewMode === 'totals' && (
                <>
                  <th className="px-3 py-2 text-center font-medium text-slate-400">Line</th>
                  <th className="px-3 py-2 text-center font-medium text-slate-400">Over</th>
                  <th className="px-3 py-2 text-center font-medium text-slate-400">Under</th>
                  <th className="px-3 py-2 text-center font-medium text-slate-400">O Liq</th>
                  <th className="px-3 py-2 text-center font-medium text-slate-400">U Liq</th>
                </>
              )}
              {viewMode === 'spreads' && (
                <>
                  <th className="px-3 py-2 text-center font-medium text-slate-400">Fav</th>
                  <th className="px-3 py-2 text-center font-medium text-slate-400">Spread</th>
                  <th className="px-3 py-2 text-center font-medium text-slate-400">Cover</th>
                  <th className="px-3 py-2 text-center font-medium text-slate-400">Miss</th>
                  <th className="px-3 py-2 text-center font-medium text-slate-400">Liq</th>
                </>
              )}
              {viewMode === 'moneylines' && (
                <>
                  <th className="px-3 py-2 text-center font-medium text-slate-400">Away</th>
                  <th className="px-3 py-2 text-center font-medium text-slate-400">Home</th>
                  <th className="px-3 py-2 text-center font-medium text-slate-400">A Liq</th>
                  <th className="px-3 py-2 text-center font-medium text-slate-400">H Liq</th>
                </>
              )}
            </tr>
          </thead>
          <tbody>
            {PM_MOCK_GAMES.map((game) => {
              const gameKey = `${game.date}-${game.awayCode}-${game.homeCode}`;
              const isExpanded = expandedGames.has(gameKey);
              const strikes = getStrikes(game);
              const mainStrike = strikes.find((s) => s.isMain) ?? strikes[0];
              const adjacentStrikes = strikes.filter((s) => !s.isMain);

              return (
                <Fragment key={gameKey}>
                  {/* Main row */}
                  <tr
                    className={cn(
                      'border-b border-slate-800',
                      isExpandable && 'cursor-pointer',
                      isExpanded ? 'bg-slate-800/50' : 'hover:bg-slate-800/30'
                    )}
                    onClick={() => isExpandable && toggleExpand(gameKey)}
                  >
                    <td className="px-3 py-2 text-slate-500">
                      {isExpandable && (
                        <span
                          className={cn(
                            'inline-block transition-transform',
                            isExpanded && 'rotate-90'
                          )}
                        >
                          ▶
                        </span>
                      )}
                    </td>
                    <td className="px-3 py-2 text-slate-400">{game.date}</td>
                    <td className="px-3 py-2 text-blue-400">{game.time}</td>
                    <td className="px-3 py-2">
                      <span
                        className={cn('font-medium', game.favorite === 'away' && 'text-amber-400')}
                      >
                        {game.awayCode}
                      </span>
                      <span className="mx-1 text-slate-500">@</span>
                      <span
                        className={cn('font-medium', game.favorite === 'home' && 'text-amber-400')}
                      >
                        {game.homeCode}
                      </span>
                    </td>
                    {/* TOTALS VIEW */}
                    {viewMode === 'totals' && (
                      <>
                        <td className="px-3 py-2 text-center font-mono font-bold text-amber-400">
                          {mainStrike.strike}
                        </td>
                        <td className="px-3 py-2 text-center font-mono text-emerald-400">
                          {formatPrice(mainStrike.yesPrice)}
                        </td>
                        <td className="px-3 py-2 text-center font-mono text-rose-400">
                          {formatPrice(mainStrike.noPrice)}
                        </td>
                        <td className="px-3 py-2 text-center font-mono text-slate-400">
                          {formatLiq(mainStrike.yesLiq)}
                        </td>
                        <td className="px-3 py-2 text-center font-mono text-slate-400">
                          {formatLiq(mainStrike.noLiq)}
                        </td>
                      </>
                    )}

                    {/* SPREADS VIEW */}
                    {viewMode === 'spreads' && (
                      <>
                        <td className="px-3 py-2 text-center font-mono text-amber-400">
                          {game.favorite === 'away' ? game.awayCode : game.homeCode}
                        </td>
                        <td className="px-3 py-2 text-center font-mono font-bold text-amber-400">
                          {mainStrike.strike}
                        </td>
                        <td className="px-3 py-2 text-center font-mono text-emerald-400">
                          {formatPrice(mainStrike.yesPrice)}
                        </td>
                        <td className="px-3 py-2 text-center font-mono text-rose-400">
                          {formatPrice(mainStrike.noPrice)}
                        </td>
                        <td className="px-3 py-2 text-center font-mono text-slate-400">
                          {formatLiq(mainStrike.yesLiq + mainStrike.noLiq)}
                        </td>
                      </>
                    )}

                    {/* MONEYLINES VIEW */}
                    {viewMode === 'moneylines' && (
                      <>
                        <td className="px-3 py-2 text-center font-mono">
                          {formatPrice(game.moneyline.awayPrice)}
                        </td>
                        <td className="px-3 py-2 text-center font-mono">
                          {formatPrice(game.moneyline.homePrice)}
                        </td>
                        <td className="px-3 py-2 text-center font-mono text-slate-400">
                          {formatLiq(game.moneyline.awayLiq)}
                        </td>
                        <td className="px-3 py-2 text-center font-mono text-slate-400">
                          {formatLiq(game.moneyline.homeLiq)}
                        </td>
                      </>
                    )}
                  </tr>

                  {/* Expanded strikes (totals and spreads) */}
                  {isExpandable &&
                    isExpanded &&
                    adjacentStrikes.map((strike) => (
                      <tr
                        key={`${gameKey}-${strike.strike}`}
                        className="border-b border-slate-800/50 bg-slate-900/50"
                      >
                        <td></td>
                        <td></td>
                        <td></td>
                        <td className="px-3 py-1.5 text-right text-slate-500">└</td>
                        {viewMode === 'totals' && (
                          <>
                            <td className="px-3 py-1.5 text-center font-mono text-slate-300">
                              {strike.strike}
                            </td>
                            <td className="px-3 py-1.5 text-center font-mono text-emerald-400/70">
                              {formatPrice(strike.yesPrice)}
                            </td>
                            <td className="px-3 py-1.5 text-center font-mono text-rose-400/70">
                              {formatPrice(strike.noPrice)}
                            </td>
                            <td className="px-3 py-1.5 text-center font-mono text-slate-500">
                              {formatLiq(strike.yesLiq)}
                            </td>
                            <td className="px-3 py-1.5 text-center font-mono text-slate-500">
                              {formatLiq(strike.noLiq)}
                            </td>
                          </>
                        )}
                        {viewMode === 'spreads' && (
                          <>
                            <td className="px-3 py-1.5 text-center font-mono text-slate-500">
                              {game.favorite === 'away' ? game.awayCode : game.homeCode}
                            </td>
                            <td className="px-3 py-1.5 text-center font-mono text-slate-300">
                              {strike.strike}
                            </td>
                            <td className="px-3 py-1.5 text-center font-mono text-emerald-400/70">
                              {formatPrice(strike.yesPrice)}
                            </td>
                            <td className="px-3 py-1.5 text-center font-mono text-rose-400/70">
                              {formatPrice(strike.noPrice)}
                            </td>
                            <td className="px-3 py-1.5 text-center font-mono text-slate-500">
                              {formatLiq(strike.yesLiq + strike.noLiq)}
                            </td>
                          </>
                        )}
                      </tr>
                    ))}
                </Fragment>
              );
            })}
          </tbody>
        </table>
      </div>

      {/* Legend */}
      <div className="border-t border-slate-700 bg-slate-800/30 px-3 py-2 text-xs text-slate-400">
        {viewMode === 'moneylines' && (
          <>
            <strong>Moneylines:</strong> Simple "who wins" markets. No expansion needed.{' '}
            <strong>American odds:</strong> -150 = bet $150 to win $100 (favorite), +150 = bet $100
            to win $150 (underdog).
          </>
        )}
        {viewMode === 'totals' && (
          <>
            <strong>Main Line:</strong> The strike closest to even odds (most balanced market).
            Click to expand and see all {PM_MOCK_GAMES[0].totalStrikes.length} available strikes.
          </>
        )}
        {viewMode === 'spreads' && (
          <>
            <strong>Main Line:</strong> The strike closest to even odds. <strong>Cover:</strong>{' '}
            Favorite wins by MORE than the spread. <strong>Miss:</strong> Favorite fails to cover
            (wins by less or loses). Click to expand and see all{' '}
            {PM_MOCK_GAMES[0].spreadStrikes.length} strikes.
          </>
        )}
      </div>
    </div>
  );
}

// =============================================================================
// TRADITIONAL SPORTSBOOK MOCKUP (for comparison)
// =============================================================================

interface OddsCell {
  spread?: string;
  spreadJuice?: string;
  total?: string;
  totalJuice?: string;
  moneyline?: string;
  isHighlighted?: boolean;
  highlightColor?: 'green' | 'yellow';
}

interface GameRow {
  time: string;
  rotationAway: number;
  rotationHome: number;
  awayTeam: string;
  homeTeam: string;
  score?: { away: number; home: number; quarter: string };
  books: Record<string, { away: OddsCell; home: OddsCell }>;
}

const MOCK_GAMES: GameRow[] = [
  {
    time: '02/02 12:10p',
    rotationAway: 551,
    rotationHome: 552,
    awayTeam: 'New Orleans',
    homeTeam: 'Charlotte',
    score: { away: 72, home: 62, quarter: '3rd Q' },
    books: {
      Consensus: {
        away: { total: '230½', spread: '-6½', spreadJuice: '-108' },
        home: { spread: '+6½', spreadJuice: '-112' },
      },
      Superbook: {
        away: { total: '230½', spread: '-6½', spreadJuice: '-109' },
        home: { spread: '+6½', spreadJuice: '-111' },
      },
      'South Point': {
        away: { total: '230½', spread: '-6½', spreadJuice: '-110' },
        home: { spread: '+6½', spreadJuice: '-110' },
      },
      Circa: {
        away: {
          total: '230½',
          spread: '-6½',
          spreadJuice: '-110',
          isHighlighted: true,
          highlightColor: 'yellow',
        },
        home: { spread: '+6½', spreadJuice: '-110' },
      },
    },
  },
  {
    time: '02/02 4:10p',
    rotationAway: 553,
    rotationHome: 554,
    awayTeam: 'Houston',
    homeTeam: 'Indiana',
    books: {
      Consensus: {
        away: { spread: '-6½', spreadJuice: '-110' },
        home: { total: '218½', spread: '+6½', spreadJuice: '-110' },
      },
      Superbook: {
        away: { spread: '-6½', spreadJuice: '-109' },
        home: { total: '218½', spread: '+6½', spreadJuice: '-111' },
      },
      'South Point': {
        away: { spread: '-6½', spreadJuice: '-110' },
        home: { total: '218½', spread: '+6½', spreadJuice: '-110' },
      },
      Circa: {
        away: { spread: '-6½', spreadJuice: '-110' },
        home: { total: '218½', spread: '+6½', spreadJuice: '-110' },
      },
    },
  },
  {
    time: '02/02 4:40p',
    rotationAway: 555,
    rotationHome: 556,
    awayTeam: 'Minnesota',
    homeTeam: 'Memphis',
    books: {
      Consensus: {
        away: { spread: '-6', spreadJuice: '-110' },
        home: { total: '229½', spread: '+6', spreadJuice: '-110' },
      },
      Superbook: {
        away: { spread: '-6', spreadJuice: '-109' },
        home: { total: '229½', spread: '+6', spreadJuice: '-111' },
      },
      'South Point': {
        away: { spread: '-6', spreadJuice: '-110' },
        home: { total: '229½', spread: '+6', spreadJuice: '-110' },
      },
      Circa: {
        away: { spread: '-6', spreadJuice: '-110' },
        home: { total: '229½', spread: '+6', spreadJuice: '-110' },
      },
    },
  },
  {
    time: '02/02 7:10p',
    rotationAway: 557,
    rotationHome: 558,
    awayTeam: 'Philadelphia',
    homeTeam: 'L.A. Clippers',
    books: {
      Consensus: {
        away: { spread: '-1½', spreadJuice: '-109' },
        home: { total: '219½', spread: '+1½', spreadJuice: '-111' },
      },
      Superbook: {
        away: { spread: '-1½', spreadJuice: '-109', isHighlighted: true, highlightColor: 'green' },
        home: { total: '219½', spread: '+1½', spreadJuice: '-111' },
      },
      'South Point': {
        away: { spread: '-2', spreadJuice: '-110' },
        home: { total: '220½', spread: '+2', spreadJuice: '-110' },
      },
      Circa: {
        away: { spread: '-1', spreadJuice: '-105', isHighlighted: true, highlightColor: 'green' },
        home: {
          total: '218',
          spread: '+1',
          spreadJuice: '-115',
          isHighlighted: true,
          highlightColor: 'green',
        },
      },
    },
  },
];

const BOOK_NAMES = ['Consensus', 'Superbook', 'South Point', 'Circa'];

function OddsCellDisplay({ cell, showTotal }: { cell: OddsCell; showTotal?: boolean }) {
  const bgClass = cell.isHighlighted
    ? cell.highlightColor === 'green'
      ? 'bg-emerald-500/20'
      : 'bg-yellow-500/20'
    : '';

  if (showTotal && cell.total) {
    return <div className={cn('px-2 py-1 font-mono text-xs', bgClass)}>{cell.total}</div>;
  }

  if (cell.spread) {
    return (
      <div className={cn('px-2 py-1 font-mono text-xs', bgClass)}>
        {cell.spread}
        {cell.spreadJuice && <span className="text-muted-foreground">{cell.spreadJuice}</span>}
      </div>
    );
  }

  return <div className="text-muted-foreground px-2 py-1 text-xs">—</div>;
}

/**
 * Interactive mockup of a traditional sportsbook odds screen.
 * This shows what the client wants us to build.
 */
function SportsbookMockup() {
  const [viewMode, setViewMode] = useState<'spreads' | 'moneylines'>('spreads');

  return (
    <div className="overflow-hidden rounded-lg border bg-slate-900 text-slate-100">
      {/* Header toolbar */}
      <div className="flex items-center gap-2 border-b border-slate-700 bg-slate-800 px-3 py-2">
        <select className="rounded bg-slate-700 px-2 py-1 text-xs">
          <option>NBA</option>
          <option>NFL</option>
          <option>CBB</option>
          <option>NHL</option>
        </select>
        <select className="rounded bg-slate-700 px-2 py-1 text-xs">
          <option>GAME — DEFAULT</option>
          <option>1ST HALF</option>
          <option>2ND HALF</option>
        </select>
        <div className="flex-1" />
        <div className="flex gap-1">
          {['TODAY', 'NFL', 'NCAAF', 'NBA', 'NCAABK', 'MLB', 'NHL'].map((sport) => (
            <button
              key={sport}
              className={cn(
                'rounded px-2 py-1 text-xs',
                sport === 'NBA' ? 'bg-blue-600' : 'bg-slate-700 hover:bg-slate-600'
              )}
            >
              {sport}
            </button>
          ))}
        </div>
      </div>

      {/* Toggle for spreads vs moneylines */}
      <div className="flex items-center gap-2 border-b border-slate-700 bg-slate-800/50 px-3 py-2">
        <span className="text-xs text-slate-400">View:</span>
        <button
          onClick={() => setViewMode('spreads')}
          className={cn(
            'rounded px-2 py-1 text-xs',
            viewMode === 'spreads' ? 'bg-blue-600' : 'bg-slate-700'
          )}
        >
          Spreads & Totals
        </button>
        <button
          onClick={() => setViewMode('moneylines')}
          className={cn(
            'rounded px-2 py-1 text-xs',
            viewMode === 'moneylines' ? 'bg-blue-600' : 'bg-slate-700'
          )}
        >
          Moneylines
        </button>
      </div>

      {/* Table */}
      <div className="overflow-x-auto">
        <table className="w-full text-xs">
          <thead>
            <tr className="border-b border-slate-700 bg-slate-800/30">
              <th className="px-2 py-2 text-left font-medium text-slate-400">Time</th>
              <th className="px-2 py-2 text-center font-medium text-slate-400">#</th>
              <th className="px-2 py-2 text-left font-medium text-slate-400">Teams</th>
              <th className="px-2 py-2 text-center font-medium text-slate-400">Score</th>
              {BOOK_NAMES.map((book) => (
                <th
                  key={book}
                  className="px-2 py-2 text-center font-medium text-slate-400"
                >
                  {book}
                </th>
              ))}
            </tr>
          </thead>
          <tbody>
            {MOCK_GAMES.map((game, gameIdx) => (
              <>
                {/* Away team row */}
                <tr
                  key={`${gameIdx}-away`}
                  className="border-b border-slate-800 hover:bg-slate-800/30"
                >
                  <td
                    className="px-2 py-1 text-blue-400"
                    rowSpan={2}
                  >
                    {game.time}
                  </td>
                  <td className="px-2 py-1 text-center text-slate-500">{game.rotationAway}</td>
                  <td className="px-2 py-1 font-medium">{game.awayTeam}</td>
                  <td className="px-2 py-1 text-center">
                    {game.score ? (
                      <span className="rounded bg-slate-700 px-1.5 py-0.5">{game.score.away}</span>
                    ) : null}
                  </td>
                  {BOOK_NAMES.map((book) => (
                    <td
                      key={book}
                      className="px-1 py-1 text-center"
                    >
                      <OddsCellDisplay
                        cell={game.books[book]?.away ?? {}}
                        showTotal={true}
                      />
                    </td>
                  ))}
                </tr>
                {/* Home team row */}
                <tr
                  key={`${gameIdx}-home`}
                  className="border-b border-slate-700 hover:bg-slate-800/30"
                >
                  <td className="px-2 py-1 text-center text-slate-500">{game.rotationHome}</td>
                  <td className="px-2 py-1 font-medium">{game.homeTeam}</td>
                  <td className="px-2 py-1 text-center">
                    {game.score ? (
                      <>
                        <span className="rounded bg-slate-700 px-1.5 py-0.5">
                          {game.score.home}
                        </span>
                        <span className="ml-1 text-slate-500">{game.score.quarter}</span>
                      </>
                    ) : null}
                  </td>
                  {BOOK_NAMES.map((book) => (
                    <td
                      key={book}
                      className="px-1 py-1 text-center"
                    >
                      <OddsCellDisplay cell={game.books[book]?.home ?? {}} />
                    </td>
                  ))}
                </tr>
              </>
            ))}
          </tbody>
        </table>
      </div>

      {/* Legend */}
      <div className="flex items-center gap-4 border-t border-slate-700 bg-slate-800/30 px-3 py-2 text-xs">
        <span className="text-slate-400">Legend:</span>
        <div className="flex items-center gap-1">
          <div className="h-3 w-3 rounded bg-emerald-500/40" />
          <span>Best odds</span>
        </div>
        <div className="flex items-center gap-1">
          <div className="h-3 w-3 rounded bg-yellow-500/40" />
          <span>Notable movement</span>
        </div>
      </div>
    </div>
  );
}

/**
 * Annotated breakdown of a single odds cell
 */
function OddsCellBreakdown() {
  return (
    <div className="bg-card rounded-lg border p-4">
      <div className="text-muted-foreground mb-3 text-xs font-semibold uppercase">
        Anatomy of an Odds Cell
      </div>

      <div className="flex items-center justify-center gap-8 py-4">
        {/* The cell */}
        <div className="relative">
          <div className="border-primary bg-primary/5 rounded border-2 border-dashed px-4 py-3 font-mono text-lg">
            <span className="text-primary font-bold">-6½</span>
            <span className="text-muted-foreground">-110</span>
          </div>

          {/* Annotations */}
          <div className="text-primary absolute -top-6 left-0 text-xs">← Spread</div>
          <div className="text-muted-foreground absolute -top-6 right-0 text-xs">Juice →</div>
        </div>

        {/* Explanation */}
        <div className="space-y-2 text-sm">
          <div>
            <span className="text-primary font-bold">-6½</span> = Team is favored by 6.5 points
          </div>
          <div>
            <span className="text-muted-foreground font-bold">-110</span> = Bet $110 to win $100
          </div>
        </div>
      </div>

      <div className="bg-muted/30 mt-4 rounded p-3 text-sm">
        <strong>Reading it together:</strong> "This team is favored by 6.5 points. To bet on them
        covering this spread, you'd pay $110 to win $100."
      </div>
    </div>
  );
}

/**
 * Visual showing how totals work
 */
function TotalsExplainer() {
  const [awayScore, setAwayScore] = useState(112);
  const [homeScore, setHomeScore] = useState(105);
  const total = awayScore + homeScore;
  const line = 220.5;

  return (
    <div className="bg-card rounded-lg border p-4">
      <div className="text-muted-foreground mb-3 text-xs font-semibold uppercase">
        Interactive: How Totals Work
      </div>

      <div className="mb-4 grid grid-cols-2 gap-4">
        <div>
          <label className="text-muted-foreground text-sm">Away Score</label>
          <input
            type="range"
            min={80}
            max={140}
            value={awayScore}
            onChange={(e) => setAwayScore(Number(e.target.value))}
            className="w-full"
          />
          <div className="text-center font-mono text-lg">{awayScore}</div>
        </div>
        <div>
          <label className="text-muted-foreground text-sm">Home Score</label>
          <input
            type="range"
            min={80}
            max={140}
            value={homeScore}
            onChange={(e) => setHomeScore(Number(e.target.value))}
            className="w-full"
          />
          <div className="text-center font-mono text-lg">{homeScore}</div>
        </div>
      </div>

      <div className="flex items-center justify-center gap-4 py-4">
        <div className="text-center">
          <div className="text-muted-foreground text-sm">Combined Score</div>
          <div className="text-3xl font-bold">{total}</div>
        </div>
        <div className="text-muted-foreground text-2xl">vs</div>
        <div className="text-center">
          <div className="text-muted-foreground text-sm">Line</div>
          <div className="text-3xl font-bold">{line}</div>
        </div>
      </div>

      <div
        className={cn(
          'rounded p-3 text-center font-semibold',
          total > line ? 'bg-emerald-500/20 text-emerald-400' : 'bg-rose-500/20 text-rose-400'
        )}
      >
        {total > line ? `OVER wins! (${total} > ${line})` : `UNDER wins! (${total} < ${line})`}
      </div>
    </div>
  );
}

/**
 * Visual showing how spreads work
 */
function SpreadExplainer() {
  const [favoriteScore, setFavoriteScore] = useState(108);
  const [underdogScore, setUnderdogScore] = useState(102);
  const spread = -6.5;
  const margin = favoriteScore - underdogScore;
  const coverSpread = Math.abs(spread);

  return (
    <div className="bg-card rounded-lg border p-4">
      <div className="text-muted-foreground mb-3 text-xs font-semibold uppercase">
        Interactive: How Spreads Work
      </div>

      <div className="mb-4 text-center">
        <span className="bg-primary/20 text-primary rounded px-2 py-1 font-mono">Lakers -6.5</span>
        <span className="text-muted-foreground mx-2">vs</span>
        <span className="bg-muted rounded px-2 py-1 font-mono">Celtics +6.5</span>
      </div>

      <div className="mb-4 grid grid-cols-2 gap-4">
        <div>
          <label className="text-muted-foreground text-sm">Lakers (Favorite)</label>
          <input
            type="range"
            min={90}
            max={130}
            value={favoriteScore}
            onChange={(e) => setFavoriteScore(Number(e.target.value))}
            className="w-full"
          />
          <div className="text-center font-mono text-lg">{favoriteScore}</div>
        </div>
        <div>
          <label className="text-muted-foreground text-sm">Celtics (Underdog)</label>
          <input
            type="range"
            min={90}
            max={130}
            value={underdogScore}
            onChange={(e) => setUnderdogScore(Number(e.target.value))}
            className="w-full"
          />
          <div className="text-center font-mono text-lg">{underdogScore}</div>
        </div>
      </div>

      <div className="flex items-center justify-center gap-4 py-4">
        <div className="text-center">
          <div className="text-muted-foreground text-sm">Actual Margin</div>
          <div className="text-3xl font-bold">{margin > 0 ? `+${margin}` : margin}</div>
          <div className="text-muted-foreground text-xs">
            {margin > 0 ? 'Lakers by' : margin < 0 ? 'Celtics by' : 'Tie'}
          </div>
        </div>
        <div className="text-muted-foreground text-2xl">vs</div>
        <div className="text-center">
          <div className="text-muted-foreground text-sm">Spread</div>
          <div className="text-3xl font-bold">-6.5</div>
          <div className="text-muted-foreground text-xs">Lakers need to win by 7+</div>
        </div>
      </div>

      <div
        className={cn(
          'rounded p-3 text-center font-semibold',
          margin > coverSpread
            ? 'bg-emerald-500/20 text-emerald-400'
            : 'bg-rose-500/20 text-rose-400'
        )}
      >
        {margin > coverSpread
          ? `Lakers COVER! (won by ${margin}, needed ${coverSpread + 0.5}+)`
          : margin > 0
            ? `Celtics cover! (Lakers only won by ${margin}, needed 7+)`
            : `Celtics cover! (Lakers lost or tied)`}
      </div>
    </div>
  );
}

// =============================================================================
// MAIN LESSON COMPONENT
// =============================================================================

export function ReadingOddsScreenLesson() {
  return (
    <LessonLayout>
      {/* Header */}
      <LessonRow>
        <LessonHeader
          title="Reading a Sportsbook Odds Screen"
          description="Learn to decode the numbers, columns, and formatting used by professional sports bettors"
          duration="20 min"
          category="Sports Betting"
        />
      </LessonRow>

      {/* Objective */}
      <LessonRow
        aside={
          <ConstraintBlock
            title="What We'll Cover"
            items={[
              'How to read the stacked numbers format',
              'What rotation numbers are',
              'Understanding spread + juice notation',
              'How professionals compare odds across books',
            ]}
          />
        }
      >
        <ObjectiveBlock>
          By the end of this lesson, you'll be able to look at a traditional sportsbook odds screen
          and understand exactly what every number means. We'll also show a mockup of the interface
          we're planning to build.
        </ObjectiveBlock>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 1: THE BIG PICTURE */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="Why This Matters">
            Professional sports bettors use screens like this to quickly spot the best odds across
            multiple sportsbooks. If Circa offers -110 but Superbook offers -105 on the same bet,
            that's free money.
          </TipBlock>
        }
      >
        <SectionHeader
          title="The Big Picture"
          subtitle="What you're looking at and why it looks this way"
        />

        <p className="text-muted-foreground leading-relaxed">
          The screenshot you saw is from a <strong>professional odds comparison tool</strong>. These
          tools aggregate odds from multiple sportsbooks so traders can quickly find the best
          prices. Think of it like a flight comparison site, but for sports bets.
        </p>

        <div className="mt-6 grid gap-4 sm:grid-cols-2">
          <ConceptCard
            title="Rows = Games"
            description="Each pair of rows represents one game. Away team on top, home team below."
            icon={<Trophy className="h-4 w-4" />}
          />
          <ConceptCard
            title="Columns = Sportsbooks"
            description="Each column shows odds from a different sportsbook (Circa, Superbook, etc.)"
            icon={<Monitor className="h-4 w-4" />}
          />
          <ConceptCard
            title="Stacked Numbers"
            description="The top number is usually the TOTAL (over/under). The bottom is the SPREAD."
            icon={<ArrowUpDown className="h-4 w-4" />}
          />
          <ConceptCard
            title="Rotation Numbers"
            description="Standardized IDs (like 551, 552) that all sportsbooks use to identify each bet."
            icon={<Hash className="h-4 w-4" />}
          />
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 2: THE KEY PROBLEM */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <WarningBlock title="Why Traditional Won't Work">
            Traditional sportsbooks show ONE line per game. Prediction markets have 10+ lines per
            game, each with its own order book. We need a different approach.
          </WarningBlock>
        }
      >
        <SectionHeader
          title="The Key Problem"
          subtitle="Why prediction markets are different"
        />

        <p className="text-muted-foreground mb-4 leading-relaxed">
          Jon realized the traditional sportsbook layout won't work for prediction markets. Here's
          why:
        </p>

        <div className="mb-6 grid gap-4 md:grid-cols-2">
          <div className="rounded-lg border p-4">
            <h4 className="mb-2 font-semibold text-amber-400">Traditional Sportsbook</h4>
            <ul className="text-muted-foreground space-y-1 text-sm">
              <li>• ONE total line per game (e.g., O/U 220.5)</li>
              <li>• ONE spread per game (e.g., LAL -6.5)</li>
              <li>• All books converge on same lines</li>
              <li>• Simple: one row = one game</li>
            </ul>
          </div>
          <div className="rounded-lg border border-blue-500/30 bg-blue-500/5 p-4">
            <h4 className="mb-2 font-semibold text-blue-400">Prediction Markets (Kalshi)</h4>
            <ul className="text-muted-foreground space-y-1 text-sm">
              <li>• MANY total lines: 211.5, 214.5, 217.5, 220.5, 223.5...</li>
              <li>• MANY spread lines: -3.5, -4.5, -5.5, -6.5, -7.5...</li>
              <li>• Each line has its own order book</li>
              <li>• Complex: one game = 10+ tradeable markets</li>
            </ul>
          </div>
        </div>

        <ExampleBox title="Real Data: MIL vs SAS Totals">
          <p className="mb-2">
            For one game, Kalshi offers <strong>11 different total lines</strong>:
          </p>
          <div className="grid grid-cols-4 gap-2 font-mono text-xs">
            <span>211.5</span>
            <span>214.5</span>
            <span>217.5</span>
            <span>220.5</span>
            <span>223.5</span>
            <span>226.5</span>
            <span>229.5</span>
            <span>232.5</span>
            <span>235.5</span>
            <span>238.5</span>
            <span>241.5</span>
          </div>
          <p className="text-muted-foreground mt-2 text-sm">
            Each of these is a separate market with its own order book, prices, and liquidity.
          </p>
        </ExampleBox>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 3: JON'S SOLUTION */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <InsightBlock title="The 50¢ Rule">
            The strike closest to 50¢ is the most "balanced" market — roughly equal action on both
            sides. This is typically the most liquid line and closest to the true consensus.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="Jon's Solution: Main Line + Expand"
          subtitle="Show one line, expand for more"
        />

        <p className="text-muted-foreground mb-4 leading-relaxed">
          From Jon's FEATURES.txt, here's the approach:
        </p>

        <ol className="text-muted-foreground mb-6 list-inside list-decimal space-y-3">
          <li>
            <strong>Find the "main" strike</strong> — For each game, find the total/spread line
            where the price is closest to 50¢. This is the most balanced, liquid market.
          </li>
          <li>
            <strong>Show main line in parent row</strong> — The table shows one row per game with
            the main strike's data.
          </li>
          <li>
            <strong>Click to expand</strong> — Clicking reveals adjacent strikes (like the existing
            moneyline ladder expansion).
          </li>
          <li>
            <strong>Dynamic updates</strong> — If order book shifts and a different strike becomes
            closest to 50¢, the main line updates automatically.
          </li>
        </ol>

        <p className="text-muted-foreground mb-6 leading-relaxed">
          <strong>Try the mockup below</strong> — Click on a game row to expand and see all the
          available strike lines:
        </p>

        <PMTotalsMockup />

        <div className="mt-4 rounded-lg border border-dashed border-emerald-500/50 bg-emerald-500/5 p-4">
          <p className="text-sm">
            <strong className="text-emerald-500">Key Insight:</strong> The "main line" (shown in
            amber) is dynamically selected as the strike closest to 50¢. Click a row to see all 7
            available strikes for that game. This is Jon's proposed solution from nba-1.6.
          </p>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 4: TRADITIONAL COMPARISON */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="For Reference">
            This traditional view is what Brendan's screenshot showed. It works for DraftKings but
            not for Kalshi because of the multiple-strikes issue.
          </TipBlock>
        }
      >
        <SectionHeader
          title="Traditional Sportsbook View (For Comparison)"
          subtitle="What Brendan's screenshot showed"
        />

        <p className="text-muted-foreground mb-6 leading-relaxed">
          For comparison, here's the traditional sportsbook format where each game has ONE spread
          and ONE total. This is what the original screenshot showed, but it won't work for
          prediction markets with multiple strikes per game.
        </p>

        <SportsbookMockup />
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 3: ROTATION NUMBERS */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="Historical Context">
            Before computers, bettors called bookies and said "I want $100 on 551." The rotation
            number system prevented miscommunication about which team you meant.
          </TipBlock>
        }
      >
        <SectionHeader
          title="Rotation Numbers"
          subtitle="The universal ID system for sports bets"
        />

        <Term term="Rotation Number">
          A standardized numerical identifier assigned to each team/side in a sporting event. Every
          licensed sportsbook in the US uses the same rotation numbers, set by Las Vegas.
        </Term>

        <div className="mt-4 space-y-3">
          <p className="text-muted-foreground leading-relaxed">
            When you see numbers like <code className="bg-muted rounded px-1">551</code> and{' '}
            <code className="bg-muted rounded px-1">552</code>, those are rotation numbers. They
            serve two purposes:
          </p>

          <ol className="text-muted-foreground list-inside list-decimal space-y-2">
            <li>
              <strong>Universal identification</strong> — Every sportsbook uses the same numbers for
              the same teams
            </li>
            <li>
              <strong>Quick reference</strong> — Instead of saying "New Orleans Pelicans at
              Charlotte Hornets", you say "551-552"
            </li>
          </ol>
        </div>

        <ExampleBox title="Reading Rotation Numbers">
          <div className="space-y-1 font-mono">
            <div>551 = New Orleans (away team)</div>
            <div>552 = Charlotte (home team)</div>
          </div>
          <p className="text-muted-foreground mt-2">
            By convention, the <strong>away team</strong> gets the lower (odd) number, and the{' '}
            <strong>home team</strong> gets the higher (even) number.
          </p>
        </ExampleBox>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 4: DECODING THE NUMBERS */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <WarningBlock title="Common Confusion">
            The "½" in spreads and totals is intentional — it eliminates ties (pushes). There's no
            such thing as a half point in basketball, so the bet must resolve one way or the other.
          </WarningBlock>
        }
      >
        <SectionHeader
          title="Decoding the Numbers"
          subtitle="What -6½-110 actually means"
        />

        <p className="text-muted-foreground mb-6 leading-relaxed">
          The most confusing part of these screens is the notation. Let's break down what{' '}
          <code className="bg-muted rounded px-1">-6½-110</code> means.
        </p>

        <OddsCellBreakdown />
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 5: SPREADS DEEP DIVE */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <InsightBlock title="Why Spreads Exist">
            If the Lakers play a much weaker team, almost everyone would bet on the Lakers to win.
            The spread "handicaps" the Lakers so both sides of the bet are attractive.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="Understanding Spreads"
          subtitle="Betting on the margin of victory"
        />

        <Term term="Point Spread">
          A handicap applied to the favored team. The favorite must win by more than the spread to
          "cover." The underdog can lose by less than the spread (or win outright) and still
          "cover."
        </Term>

        <div className="mt-4 grid gap-4 sm:grid-cols-2">
          <Term term="Cover">
            The favorite <strong>"covers"</strong> when they win by <em>more</em> than the spread.
            If Lakers are -7.5, they cover by winning by 8 or more points. The bet wins.
          </Term>
          <Term term="Miss">
            The favorite <strong>"misses"</strong> when they fail to cover — winning by less than
            the spread, or losing outright. If Lakers -7.5 win by only 5, they "missed" the spread.
          </Term>
        </div>

        <div className="mt-4 space-y-4">
          <p className="text-muted-foreground leading-relaxed">
            When you see <code className="bg-muted rounded px-1">Lakers -6.5</code>, it means:
          </p>

          <ul className="text-muted-foreground list-inside list-disc space-y-2">
            <li>
              The Lakers are the <strong>favorite</strong> (minus sign = giving points)
            </li>
            <li>
              They must win by <strong>7 or more points</strong> for the bet to win
            </li>
            <li>
              If they win by exactly 6, the bet <strong>loses</strong> (they didn't "cover")
            </li>
          </ul>

          <p className="text-muted-foreground leading-relaxed">
            The opposite side would be <code className="bg-muted rounded px-1">Celtics +6.5</code>:
          </p>

          <ul className="text-muted-foreground list-inside list-disc space-y-2">
            <li>
              The Celtics are the <strong>underdog</strong> (plus sign = receiving points)
            </li>
            <li>
              They can lose by <strong>up to 6 points</strong> and the bet still wins
            </li>
            <li>If they win outright, the bet definitely wins</li>
          </ul>
        </div>

        <div className="mt-6">
          <SpreadExplainer />
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 6: TOTALS DEEP DIVE */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="When to Bet Totals">
            Totals are popular when you have an opinion about the game's "nature" but not its
            winner. Is it going to be a defensive grind or a high-scoring shootout?
          </TipBlock>
        }
      >
        <SectionHeader
          title="Understanding Totals"
          subtitle="Betting on the combined score"
        />

        <Term term="Total (Over/Under)">
          A bet on whether the combined final score of both teams will be higher (over) or lower
          (under) than a set number.
        </Term>

        <div className="mt-4 space-y-4">
          <p className="text-muted-foreground leading-relaxed">
            When you see <code className="bg-muted rounded px-1">220½</code> in the total column:
          </p>

          <ul className="text-muted-foreground list-inside list-disc space-y-2">
            <li>
              <strong>Over 220.5</strong> wins if the combined score is 221 or higher
            </li>
            <li>
              <strong>Under 220.5</strong> wins if the combined score is 220 or lower
            </li>
            <li>You're betting on the game's pace and scoring, not who wins</li>
          </ul>
        </div>

        <div className="mt-6">
          <TotalsExplainer />
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 7: THE JUICE (VIG) */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <WarningBlock title="The House Edge">
            At -110/-110, you need to win 52.4% of your bets just to break even. The juice is how
            sportsbooks make money regardless of outcomes.
          </WarningBlock>
        }
      >
        <SectionHeader
          title="Understanding the Juice"
          subtitle="The sportsbook's fee built into every bet"
        />

        <Term term="Juice (Vig)">
          The commission charged by the sportsbook on losing bets. Standard juice is -110, meaning
          you must bet $110 to win $100. Also called "vigorish" or "vig."
        </Term>

        <div className="mt-4 space-y-4">
          <p className="text-muted-foreground leading-relaxed">
            The second number in <code className="bg-muted rounded px-1">-6½-110</code> is the
            juice. Here's how to read common juice numbers:
          </p>

          <div className="grid gap-3 sm:grid-cols-2">
            <div className="rounded border p-3">
              <div className="font-mono text-lg">-110</div>
              <div className="text-muted-foreground text-sm">Bet $110 to win $100 (standard)</div>
            </div>
            <div className="rounded border p-3">
              <div className="font-mono text-lg">-105</div>
              <div className="text-muted-foreground text-sm">
                Bet $105 to win $100 (better deal)
              </div>
            </div>
            <div className="rounded border p-3">
              <div className="font-mono text-lg">-115</div>
              <div className="text-muted-foreground text-sm">Bet $115 to win $100 (worse deal)</div>
            </div>
            <div className="rounded border p-3">
              <div className="font-mono text-lg">+100</div>
              <div className="text-muted-foreground text-sm">Bet $100 to win $100 (no juice)</div>
            </div>
          </div>

          <p className="text-muted-foreground leading-relaxed">
            When you see <strong className="text-emerald-400">green highlights</strong> on a screen,
            it usually means that sportsbook is offering better juice (lower cost) than competitors.
            That's why odds comparison matters — finding -105 instead of -110 saves you $5 per $100
            bet.
          </p>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 8: PUTTING IT ALL TOGETHER */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <InsightBlock title="Key Takeaway">
            The default view shows spreads and totals because that's where most professional action
            happens. Moneylines are popular with casual bettors but less useful for finding edges.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="Putting It All Together"
          subtitle="Reading a full row"
        />

        <p className="text-muted-foreground mb-6 leading-relaxed">
          Let's read a complete game entry step by step.
        </p>

        <ExampleBox title="Complete Row Analysis">
          <div className="space-y-2 font-mono text-sm">
            <div className="grid grid-cols-[80px_40px_100px_60px_1fr] items-center gap-2">
              <div className="text-blue-400">02/02</div>
              <div className="text-slate-500">551</div>
              <div>New Orleans</div>
              <div className="text-center">72</div>
              <div>230½ / -6½-108</div>
            </div>
            <div className="grid grid-cols-[80px_40px_100px_60px_1fr] items-center gap-2">
              <div className="text-blue-400">12:10p</div>
              <div className="text-slate-500">552</div>
              <div>Charlotte</div>
              <div className="text-center">62 3Q</div>
              <div>+6½-112</div>
            </div>
          </div>

          <div className="mt-4 space-y-2 text-sm">
            <p>
              <strong>Date/Time:</strong> February 2nd at 12:10 PM
            </p>
            <p>
              <strong>Game:</strong> New Orleans (551, away) at Charlotte (552, home)
            </p>
            <p>
              <strong>Live Score:</strong> New Orleans 72, Charlotte 62, 3rd Quarter
            </p>
            <p>
              <strong>Total:</strong> 230.5 points (combined score over/under)
            </p>
            <p>
              <strong>Spread:</strong> Charlotte -6.5 (favored by 6.5 at home). But wait — look
              again! Charlotte is LOSING 72-62. This spread was set before the game started.
            </p>
            <p>
              <strong>Juice:</strong> Charlotte -108 (slightly better), New Orleans would be the
              opposite
            </p>
          </div>
        </ExampleBox>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 9: WHAT WE'RE BUILDING */}
      {/* =================================================================== */}
      <LessonRow>
        <SectionHeader
          title="What We're Building"
          subtitle="Translating this to our dashboard"
        />

        <p className="text-muted-foreground mb-4 leading-relaxed">
          The client wants our dashboard to work like this traditional interface, but showing Kalshi
          and Polymarket odds instead of traditional sportsbooks. Here's the plan:
        </p>

        <div className="space-y-4">
          <div className="rounded border p-4">
            <h4 className="mb-2 font-semibold">1. Default View: Spreads & Totals</h4>
            <p className="text-muted-foreground text-sm">
              When you load the dashboard, you see spreads and totals by default (not moneylines
              like now). Each game shows the spread line and total line from Kalshi and Polymarket.
            </p>
          </div>

          <div className="rounded border p-4">
            <h4 className="mb-2 font-semibold">2. Toggle to Moneylines</h4>
            <p className="text-muted-foreground text-sm">
              A toggle button switches the view to moneylines (who wins). This is what we currently
              show.
            </p>
          </div>

          <div className="rounded border p-4">
            <h4 className="mb-2 font-semibold">3. Multi-Sport Support</h4>
            <p className="text-muted-foreground text-sm">
              Sport tabs at the top: NBA, CBB, NFL, NHL, Tennis. Each sport gets its own view.
            </p>
          </div>

          <div className="rounded border p-4">
            <h4 className="mb-2 font-semibold">4. Click to Trade</h4>
            <p className="text-muted-foreground text-sm">
              Clicking on any odds cell opens the trade modal, pre-filled with that market and
              price.
            </p>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 10: WHAT'S OUT OF SCOPE */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="Props = Player Bets">
            "Will LeBron score 30+ points?" or "Will Patrick Mahomes throw 3+ TDs?" are props. They
            require much more data and a different UI paradigm.
          </TipBlock>
        }
      >
        <SectionHeader
          title="What's NOT in Scope"
          subtitle="Player props and exotic bets"
        />

        <p className="text-muted-foreground mb-4 leading-relaxed">
          The client explicitly said: <em>"We don't need any of these. These are props."</em>
        </p>

        <div className="rounded-lg border border-rose-500/30 bg-rose-500/5 p-4">
          <h4 className="mb-2 font-semibold text-rose-400">Not Building (Props)</h4>
          <ul className="text-muted-foreground list-inside list-disc space-y-1 text-sm">
            <li>First touchdown scorer bets</li>
            <li>Player points/rebounds/assists markets</li>
            <li>2+ touchdowns prop bets</li>
            <li>Any bet on individual player performance</li>
          </ul>
        </div>

        <div className="mt-4 rounded-lg border border-emerald-500/30 bg-emerald-500/5 p-4">
          <h4 className="mb-2 font-semibold text-emerald-400">Building (Team-Level)</h4>
          <ul className="text-muted-foreground list-inside list-disc space-y-1 text-sm">
            <li>Moneylines (who wins)</li>
            <li>Spreads (margin of victory)</li>
            <li>Totals (combined score over/under)</li>
          </ul>
        </div>
      </LessonRow>

      {/* Summary */}
      <LessonRow>
        <div className="bg-muted/30 rounded-lg border p-6">
          <h3 className="mb-4 font-semibold">Quick Reference Summary</h3>
          <div className="grid gap-4 sm:grid-cols-2">
            <div>
              <h4 className="text-muted-foreground mb-2 text-sm font-medium">
                Reading the Numbers
              </h4>
              <ul className="space-y-1 text-sm">
                <li>
                  <code className="bg-muted rounded px-1">-6½</code> = Favored by 6.5 points
                </li>
                <li>
                  <code className="bg-muted rounded px-1">+6½</code> = Underdog by 6.5 points
                </li>
                <li>
                  <code className="bg-muted rounded px-1">-110</code> = Bet $110 to win $100
                </li>
                <li>
                  <code className="bg-muted rounded px-1">220½</code> = Total line of 220.5
                </li>
              </ul>
            </div>
            <div>
              <h4 className="text-muted-foreground mb-2 text-sm font-medium">Key Terms</h4>
              <ul className="space-y-1 text-sm">
                <li>
                  <strong>Spread:</strong> Margin of victory handicap
                </li>
                <li>
                  <strong>Total:</strong> Combined score over/under
                </li>
                <li>
                  <strong>Cover:</strong> Favorite wins by more than spread
                </li>
                <li>
                  <strong>Miss:</strong> Favorite fails to cover spread
                </li>
                <li>
                  <strong>Juice:</strong> Sportsbook's commission
                </li>
              </ul>
            </div>
          </div>
        </div>
      </LessonRow>
    </LessonLayout>
  );
}
