/**
 * Lesson: Reading Order Flow
 *
 * Understanding what the order book tells you about market sentiment
 * and anticipating price movements.
 */

import {
  Activity,
  TrendingUp,
  TrendingDown,
  Eye,
  Zap,
  BarChart3,
  Clock,
  Users,
} from 'lucide-react';
import { getGuide, getNextGuide } from '@/data/guides';
import { LessonLayout, LessonRow, LessonHeader, SectionHeader } from '../LessonLayout';
import {
  ObjectiveBlock,
  ConstraintBlock,
  TipBlock,
  InsightBlock,
  WarningBlock,
  ConceptCard,
  ConceptBlock,
  ExampleBox,
  NextStepBlock,
  Term,
} from '../blocks';
import { AnnotatedOrderBook } from '../widgets/AnnotatedOrderBook';
import { GlossaryTooltip } from '@/components/atoms/GlossaryTooltip';

// Custom glossary terms for this lesson
const terms = {
  orderFlow: {
    term: 'Order Flow',
    definition:
      'The stream of buy and sell orders hitting the market. Analyzing flow reveals where money is going.',
    category: 'Order Book',
  },
  aggressor: {
    term: 'Aggressor',
    definition:
      'The party who initiates a trade by hitting existing orders. Aggressors remove liquidity and signal urgency.',
    category: 'Order Book',
  },
  passive: {
    term: 'Passive Order',
    definition:
      'An order that waits in the book to be filled. Passive orders add liquidity and signal patience.',
    category: 'Order Book',
  },
  sweeps: {
    term: 'Sweep',
    definition:
      'A large order that clears multiple price levels. Sweeps indicate strong conviction and urgency.',
    category: 'Order Book',
  },
  bookPressure: {
    term: 'Book Pressure',
    definition:
      'The imbalance between bid and ask sizes. More bids = buy pressure. More asks = sell pressure.',
    category: 'Order Book',
  },
  depth: {
    term: 'Market Depth',
    definition:
      'The total volume available at each price level. Deeper markets can absorb larger orders without moving price.',
    category: 'Order Book',
  },
  spoofing: {
    term: 'Spoofing',
    definition:
      'Placing fake orders to manipulate others, then canceling before execution. Illegal but still occurs.',
    category: 'Order Book',
  },
  tape: {
    term: 'Tape',
    definition:
      'The real-time record of executed trades. Reading the tape shows actual transactions, not just intentions.',
    category: 'Order Book',
  },
};

export function ReadingOrderFlowLesson() {
  const guide = getGuide('reading-order-flow')!;
  const nextGuide = getNextGuide('reading-order-flow');

  return (
    <LessonLayout>
      {/* Header */}
      <LessonRow>
        <LessonHeader
          title={guide.title}
          description={guide.description}
          duration="18 min"
          category="Strategies"
        />
      </LessonRow>

      {/* Objective */}
      <LessonRow
        aside={
          <ConstraintBlock
            title="What We'll Cover"
            items={[
              'What order flow is and why it matters',
              'Reading the order book for signals',
              'Identifying aggressive vs passive flow',
              'Common order flow patterns',
              'Limitations and traps to avoid',
            ]}
          />
        }
      >
        <ObjectiveBlock>
          By the end of this lesson, you'll understand how to read order flow - the stream of buy
          and sell orders that reveals where money is going and what traders are thinking. This is a
          key skill for anticipating short-term price movements.
        </ObjectiveBlock>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 1: WHAT IS ORDER FLOW */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <InsightBlock title="Beyond Prices">
            Price tells you what happened. Order flow tells you what's happening and hints at what
            might happen next.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="What is Order Flow?"
          subtitle="The heartbeat of the market"
        />

        <p className="text-muted-foreground leading-relaxed">
          <GlossaryTooltip term={terms.orderFlow}>Order flow</GlossaryTooltip> is the real-time
          stream of buy and sell orders entering and exiting the market. While most people watch
          price charts, professionals watch order flow - it shows you the market's intentions, not
          just its history.
        </p>

        <div className="mt-6 grid gap-4 sm:grid-cols-2">
          <div className="bg-card rounded-lg border p-4">
            <div className="mb-3 flex items-center gap-2">
              <BarChart3 className="text-muted-foreground h-5 w-5" />
              <h4 className="font-semibold">What Price Tells You</h4>
            </div>
            <ul className="text-muted-foreground space-y-1 text-sm">
              <li>• Where trades happened</li>
              <li>• Historical movement</li>
              <li>• Support/resistance levels</li>
              <li>• Lagging information</li>
            </ul>
          </div>
          <div className="bg-card rounded-lg border p-4">
            <div className="mb-3 flex items-center gap-2">
              <Activity className="text-primary h-5 w-5" />
              <h4 className="font-semibold">What Order Flow Tells You</h4>
            </div>
            <ul className="text-muted-foreground space-y-1 text-sm">
              <li>• Who's buying/selling right now</li>
              <li>• Urgency and conviction</li>
              <li>• Where liquidity sits</li>
              <li>• Leading information</li>
            </ul>
          </div>
        </div>

        <div className="from-primary/10 via-primary/5 border-primary/30 mt-6 rounded-lg border bg-gradient-to-br to-transparent p-4">
          <h4 className="mb-2 font-semibold">The Core Insight</h4>
          <p className="text-muted-foreground text-sm">
            Every price movement is caused by order flow imbalance. More aggressive buyers than
            sellers → price goes up. More aggressive sellers than buyers → price goes down.
            Understanding flow means understanding why prices move.
          </p>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 2: AGGRESSIVE VS PASSIVE */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <ConceptBlock title="Who Moves Price?">
            Passive orders provide liquidity. Aggressive orders remove it. Price only moves when
            someone is willing to cross the spread.
          </ConceptBlock>
        }
      >
        <SectionHeader
          title="Aggressive vs. Passive Orders"
          subtitle="Understanding who's driving the market"
        />

        <p className="text-muted-foreground leading-relaxed">
          Not all orders are equal.{' '}
          <GlossaryTooltip term={terms.aggressor}>Aggressive orders</GlossaryTooltip> hit existing
          quotes and cause immediate execution.{' '}
          <GlossaryTooltip term={terms.passive}>Passive orders</GlossaryTooltip> wait in the book
          for someone else to trade with them.
        </p>

        <div className="mt-6 space-y-4">
          <Term term="Aggressive (Taker)">
            An order that immediately executes against existing orders. If you submit a market order
            or a limit order that crosses the spread, you're aggressive. Aggressors signal urgency
            and conviction - they're willing to pay the spread to get filled now.
          </Term>

          <Term term="Passive (Maker)">
            An order that adds to the order book and waits. If you submit a limit order that doesn't
            immediately fill, you're passive. Passive orders signal patience - you're willing to
            wait for your price.
          </Term>
        </div>

        <ExampleBox title="Aggressive vs Passive in Action">
          <div className="space-y-4">
            <div className="bg-muted/50 rounded-lg p-4">
              <div className="mb-3 text-sm font-medium">Current Order Book</div>
              <div className="grid gap-4 font-mono text-sm sm:grid-cols-2">
                <div>
                  <div className="mb-2 text-emerald-400">Bids</div>
                  <div>55¢ - 500 contracts</div>
                  <div>54¢ - 300 contracts</div>
                </div>
                <div>
                  <div className="mb-2 text-rose-400">Asks</div>
                  <div>58¢ - 400 contracts</div>
                  <div>59¢ - 250 contracts</div>
                </div>
              </div>
              <div className="text-muted-foreground mt-3 text-xs">
                Spread: 3¢ (55¢ bid / 58¢ ask)
              </div>
            </div>

            <div className="grid gap-3 sm:grid-cols-2">
              <div className="rounded border border-emerald-500/30 bg-emerald-500/5 p-3">
                <div className="mb-2 text-sm font-medium text-emerald-400">Aggressive Buy</div>
                <p className="text-muted-foreground text-xs">
                  "Buy 100 at market" → Fills at 58¢ (best ask). You crossed the spread, signaling
                  urgency. You're the aggressor.
                </p>
              </div>
              <div className="rounded border border-violet-500/30 bg-violet-500/5 p-3">
                <div className="mb-2 text-sm font-medium text-violet-400">Passive Buy</div>
                <p className="text-muted-foreground text-xs">
                  "Buy 100 at 56¢" → Sits in book, waits. You didn't cross the spread. You're adding
                  liquidity and waiting.
                </p>
              </div>
            </div>

            <p className="text-muted-foreground border-t pt-3 text-xs">
              <strong>Key insight:</strong> When you see lots of aggressive buying (trades happening
              at the ask), buyers are urgent. When you see lots of aggressive selling (trades at
              bid), sellers are urgent.
            </p>
          </div>
        </ExampleBox>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 3: READING THE BOOK */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="Watch for Changes">
            It's not just about what the book looks like - it's about how it changes. Orders
            appearing and disappearing tell a story.
          </TipBlock>
        }
      >
        <SectionHeader
          title="Reading the Order Book"
          subtitle="What to look for"
        />

        <p className="text-muted-foreground leading-relaxed">
          The order book is a live display of supply and demand. Learning to read it gives you
          insight into market sentiment and potential price direction.
        </p>

        <div className="mt-6 grid gap-4">
          <div className="bg-card rounded-lg border p-4">
            <div className="mb-3 flex items-center gap-2">
              <div className="flex h-8 w-8 items-center justify-center rounded-full bg-emerald-500/20">
                <TrendingUp className="h-4 w-4 text-emerald-400" />
              </div>
              <h4 className="font-semibold">Bullish Signals</h4>
            </div>
            <ul className="text-muted-foreground space-y-2 text-sm">
              <li className="flex items-start gap-2">
                <span className="text-emerald-400">•</span>
                <span>
                  <strong>Thick bids, thin asks:</strong> Lots of buyers, few sellers at current
                  prices
                </span>
              </li>
              <li className="flex items-start gap-2">
                <span className="text-emerald-400">•</span>
                <span>
                  <strong>Bids moving up:</strong> Buyers are raising their prices, willing to pay
                  more
                </span>
              </li>
              <li className="flex items-start gap-2">
                <span className="text-emerald-400">•</span>
                <span>
                  <strong>Aggressive buying:</strong> Trades executing at the ask, not the bid
                </span>
              </li>
              <li className="flex items-start gap-2">
                <span className="text-emerald-400">•</span>
                <span>
                  <strong>Asks getting lifted:</strong> Sell orders being consumed quickly
                </span>
              </li>
            </ul>
          </div>

          <div className="bg-card rounded-lg border p-4">
            <div className="mb-3 flex items-center gap-2">
              <div className="flex h-8 w-8 items-center justify-center rounded-full bg-rose-500/20">
                <TrendingDown className="h-4 w-4 text-rose-400" />
              </div>
              <h4 className="font-semibold">Bearish Signals</h4>
            </div>
            <ul className="text-muted-foreground space-y-2 text-sm">
              <li className="flex items-start gap-2">
                <span className="text-rose-400">•</span>
                <span>
                  <strong>Thick asks, thin bids:</strong> Lots of sellers, few buyers at current
                  prices
                </span>
              </li>
              <li className="flex items-start gap-2">
                <span className="text-rose-400">•</span>
                <span>
                  <strong>Asks moving down:</strong> Sellers lowering prices, willing to take less
                </span>
              </li>
              <li className="flex items-start gap-2">
                <span className="text-rose-400">•</span>
                <span>
                  <strong>Aggressive selling:</strong> Trades executing at the bid, not the ask
                </span>
              </li>
              <li className="flex items-start gap-2">
                <span className="text-rose-400">•</span>
                <span>
                  <strong>Bids getting hit:</strong> Buy orders being consumed quickly
                </span>
              </li>
            </ul>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 3.5: UNDERSTANDING THE ORDER BOOK UI */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="Interactive Learning">
            Click on any element in the order book below to learn what it means. This matches
            exactly what you'll see in the trading interface.
          </TipBlock>
        }
      >
        <SectionHeader
          title="Understanding Our Order Book"
          subtitle="Reading the actual interface"
        />

        <p className="text-muted-foreground mb-6 leading-relaxed">
          The concepts above are important, but let's see how they appear in the actual trading
          interface. Our order book shows YES bids on the left (green) and NO asks on the right
          (red), with the spread between them.
        </p>

        <AnnotatedOrderBook />

        <div className="bg-muted/50 mt-6 rounded-lg border p-4">
          <h4 className="mb-3 font-semibold">Quick Reference: Reading the Layout</h4>
          <div className="grid gap-4 text-sm sm:grid-cols-2">
            <div>
              <div className="mb-2 font-medium text-emerald-400">Left Side (Bids)</div>
              <ul className="text-muted-foreground space-y-1">
                <li>
                  <span className="font-mono">Price</span> → What buyers will pay for YES
                </li>
                <li>
                  <span className="font-mono">Size</span> → Contracts at that price
                </li>
                <li>
                  <span className="font-mono">Total</span> → Cumulative depth
                </li>
              </ul>
            </div>
            <div>
              <div className="mb-2 font-medium text-rose-400">Right Side (Asks)</div>
              <ul className="text-muted-foreground space-y-1">
                <li>
                  <span className="font-mono">Total</span> → Cumulative depth
                </li>
                <li>
                  <span className="font-mono">Size</span> → Contracts at that price
                </li>
                <li>
                  <span className="font-mono">Price</span> → What sellers want for YES
                </li>
              </ul>
            </div>
          </div>
          <p className="text-muted-foreground mt-4 border-t pt-3 text-xs">
            <strong>Key insight:</strong> The columns are mirrored so the prices face outward (green
            on far left, red on far right) and depth accumulates toward the center. This makes it
            easy to see the spread at a glance.
          </p>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 4: DEPTH AND PRESSURE */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <InsightBlock title="Depth ≠ Direction">
            Deep bids don't mean price will go up. They mean it will take more selling to push it
            down. Think of depth as resistance, not direction.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="Market Depth & Pressure"
          subtitle="Understanding supply and demand"
        />

        <p className="text-muted-foreground leading-relaxed">
          <GlossaryTooltip term={terms.depth}>Market depth</GlossaryTooltip> shows how much volume
          is available at each price level.{' '}
          <GlossaryTooltip term={terms.bookPressure}>Book pressure</GlossaryTooltip> is the
          imbalance between bids and asks.
        </p>

        <ExampleBox title="Reading Depth">
          <div className="space-y-4">
            <div className="grid gap-4 sm:grid-cols-2">
              <div className="bg-muted/50 rounded-lg p-4">
                <div className="mb-3 text-sm font-medium text-emerald-400">
                  Buy Pressure (Bullish)
                </div>
                <div className="space-y-2 font-mono text-xs">
                  <div className="flex items-center gap-2">
                    <div className="h-3 w-32 rounded bg-rose-500/30" />
                    <span>58¢: 200</span>
                  </div>
                  <div className="flex items-center gap-2">
                    <div className="h-3 w-16 rounded bg-rose-500/30" />
                    <span>59¢: 100</span>
                  </div>
                  <div className="my-2 border-t" />
                  <div className="flex items-center gap-2">
                    <div className="h-3 w-full rounded bg-emerald-500/30" />
                    <span>55¢: 800</span>
                  </div>
                  <div className="flex items-center gap-2">
                    <div className="h-3 w-3/4 rounded bg-emerald-500/30" />
                    <span>54¢: 600</span>
                  </div>
                </div>
                <p className="text-muted-foreground mt-3 text-xs">
                  Bids much larger than asks → harder to push down
                </p>
              </div>

              <div className="bg-muted/50 rounded-lg p-4">
                <div className="mb-3 text-sm font-medium text-rose-400">
                  Sell Pressure (Bearish)
                </div>
                <div className="space-y-2 font-mono text-xs">
                  <div className="flex items-center gap-2">
                    <div className="h-3 w-full rounded bg-rose-500/30" />
                    <span>58¢: 800</span>
                  </div>
                  <div className="flex items-center gap-2">
                    <div className="h-3 w-3/4 rounded bg-rose-500/30" />
                    <span>59¢: 600</span>
                  </div>
                  <div className="my-2 border-t" />
                  <div className="flex items-center gap-2">
                    <div className="h-3 w-32 rounded bg-emerald-500/30" />
                    <span>55¢: 200</span>
                  </div>
                  <div className="flex items-center gap-2">
                    <div className="h-3 w-16 rounded bg-emerald-500/30" />
                    <span>54¢: 100</span>
                  </div>
                </div>
                <p className="text-muted-foreground mt-3 text-xs">
                  Asks much larger than bids → harder to push up
                </p>
              </div>
            </div>
          </div>
        </ExampleBox>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 5: COMMON PATTERNS */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <WarningBlock title="Context Matters">
            These patterns work in context. A sweep right before an event has different meaning than
            a sweep during quiet hours.
          </WarningBlock>
        }
      >
        <SectionHeader
          title="Common Order Flow Patterns"
          subtitle="Recognizing what the market is telling you"
        />

        <div className="space-y-4">
          <div className="bg-card rounded-lg border p-4">
            <div className="mb-3 flex items-center gap-2">
              <Zap className="h-5 w-5 text-amber-400" />
              <h4 className="font-semibold">Sweeps</h4>
            </div>
            <p className="text-muted-foreground mb-3 text-sm">
              A <GlossaryTooltip term={terms.sweeps}>sweep</GlossaryTooltip> is a large order that
              clears multiple price levels quickly. It signals strong conviction - someone wanted
              size immediately and was willing to pay through multiple prices to get it.
            </p>
            <div className="bg-muted/50 rounded p-3 text-sm">
              <strong>Example:</strong> Asks at 58¢ (400), 59¢ (300), 60¢ (200) all get filled in
              seconds. Someone bought 900 contracts and moved price from 58¢ to 60¢. Strong bullish
              signal.
            </div>
          </div>

          <div className="bg-card rounded-lg border p-4">
            <div className="mb-3 flex items-center gap-2">
              <Clock className="h-5 w-5 text-violet-400" />
              <h4 className="font-semibold">Absorption</h4>
            </div>
            <p className="text-muted-foreground mb-3 text-sm">
              When aggressive orders hit a price level but the level keeps refilling. Someone is
              absorbing the flow - a large player is accumulating or distributing without moving
              price.
            </p>
            <div className="bg-muted/50 rounded p-3 text-sm">
              <strong>Example:</strong> 500 contracts at 55¢ bid. Sellers keep hitting it, but it
              stays at 500. Someone is refilling - they want to buy a lot at this price quietly.
            </div>
          </div>

          <div className="bg-card rounded-lg border p-4">
            <div className="mb-3 flex items-center gap-2">
              <Eye className="h-5 w-5 text-emerald-400" />
              <h4 className="font-semibold">Iceberg Orders</h4>
            </div>
            <p className="text-muted-foreground mb-3 text-sm">
              Large orders split into smaller visible pieces. Only a portion shows in the book; as
              it fills, more appears. Used to hide true size and avoid moving the market.
            </p>
            <div className="bg-muted/50 rounded p-3 text-sm">
              <strong>Example:</strong> 100 contracts show at 55¢. Gets filled. Another 100 appears
              at 55¢. Gets filled. This repeats 10 times. Real order was 1,000 contracts.
            </div>
          </div>

          <div className="bg-card rounded-lg border p-4">
            <div className="mb-3 flex items-center gap-2">
              <Users className="h-5 w-5 text-rose-400" />
              <h4 className="font-semibold">Pulling/Stacking</h4>
            </div>
            <p className="text-muted-foreground mb-3 text-sm">
              Orders appearing and disappearing rapidly. Can signal indecision, or be deliberate
              manipulation (<GlossaryTooltip term={terms.spoofing}>spoofing</GlossaryTooltip>) -
              placing orders to create false impression, then canceling.
            </p>
            <div className="bg-muted/50 rounded p-3 text-sm">
              <strong>Example:</strong> Large 2,000 contract bid appears at 55¢. Price rises as
              others buy. The 2,000 suddenly disappears - it was never meant to fill.
            </div>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 6: READING THE TAPE */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="Time & Sales">
            Most platforms have a "time & sales" or "tape" window showing recent trades. This is
            where you see actual executions, not just intentions.
          </TipBlock>
        }
      >
        <SectionHeader
          title="Reading the Tape"
          subtitle="Watching actual trades"
        />

        <p className="text-muted-foreground leading-relaxed">
          The <GlossaryTooltip term={terms.tape}>tape</GlossaryTooltip> shows actual executed trades
          - not what people are willing to do, but what they actually did. This is ground truth,
          harder to fake than order book displays.
        </p>

        <ExampleBox title="What to Watch on the Tape">
          <div className="space-y-4">
            <div className="bg-muted/50 rounded-lg p-4 font-mono text-xs">
              <div className="text-muted-foreground mb-2">Recent Trades:</div>
              <div className="space-y-1">
                <div className="flex justify-between">
                  <span className="text-emerald-400">BUY</span>
                  <span>58¢</span>
                  <span>200</span>
                  <span className="text-muted-foreground">14:32:45</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-emerald-400">BUY</span>
                  <span>58¢</span>
                  <span>150</span>
                  <span className="text-muted-foreground">14:32:47</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-emerald-400">BUY</span>
                  <span>59¢</span>
                  <span>300</span>
                  <span className="text-muted-foreground">14:32:48</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-rose-400">SELL</span>
                  <span>58¢</span>
                  <span>50</span>
                  <span className="text-muted-foreground">14:32:52</span>
                </div>
              </div>
            </div>

            <div className="grid gap-3 sm:grid-cols-2">
              <div className="bg-muted/30 rounded border p-3">
                <div className="mb-2 text-sm font-medium">Trade Size</div>
                <p className="text-muted-foreground text-xs">
                  Large trades suggest institutional or professional activity. Many small trades
                  suggest retail participation.
                </p>
              </div>
              <div className="bg-muted/30 rounded border p-3">
                <div className="mb-2 text-sm font-medium">Trade Side</div>
                <p className="text-muted-foreground text-xs">
                  Trades at ask = aggressive buys. Trades at bid = aggressive sells. The balance
                  shows which side is more urgent.
                </p>
              </div>
            </div>
          </div>
        </ExampleBox>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 7: LIMITATIONS */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <WarningBlock title="Be Skeptical">
            The order book can be manipulated. Large orders can be fake. Don't blindly trust what
            you see.
          </WarningBlock>
        }
      >
        <SectionHeader
          title="Limitations & Traps"
          subtitle="What order flow can't tell you"
        />

        <p className="text-muted-foreground mb-6 leading-relaxed">
          Order flow analysis is powerful but imperfect. Understanding its limitations prevents
          costly mistakes.
        </p>

        <div className="grid gap-4 sm:grid-cols-2">
          <ConceptCard
            title="Spoofing"
            description="Fake orders placed to deceive, then canceled. The book can lie."
            icon={<Eye className="h-4 w-4" />}
          />
          <ConceptCard
            title="Hidden Orders"
            description="Iceberg and dark pool orders don't show full size. True liquidity is hidden."
            icon={<Activity className="h-4 w-4" />}
          />
          <ConceptCard
            title="Context Required"
            description="A sweep means different things before/during/after an event. Timing matters."
            icon={<Clock className="h-4 w-4" />}
          />
          <ConceptCard
            title="Self-Fulfilling"
            description="If everyone reads flow the same way, signals become crowded trades."
            icon={<Users className="h-4 w-4" />}
          />
        </div>

        <div className="mt-6 rounded-lg border border-amber-500/30 bg-amber-500/10 p-4">
          <h4 className="mb-2 font-semibold text-amber-400">Key Caution</h4>
          <p className="text-muted-foreground text-sm">
            Order flow is one input, not the only input. Combine it with your analysis of the
            underlying event. A bullish order flow pattern on a market where you fundamentally
            disagree might mean you're seeing manipulation, not signal.
          </p>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 8: KEY TAKEAWAYS */}
      {/* =================================================================== */}
      <LessonRow>
        <SectionHeader
          title="Key Takeaways"
          subtitle="Remember these principles"
        />

        <div className="space-y-4">
          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="bg-primary/20 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
              <span className="text-primary font-bold">1</span>
            </div>
            <div>
              <h4 className="font-medium">Order flow shows intent, not just history</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                Price tells you what happened. Flow tells you what's happening now and hints at
                what's next.
              </p>
            </div>
          </div>

          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="bg-primary/20 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
              <span className="text-primary font-bold">2</span>
            </div>
            <div>
              <h4 className="font-medium">Aggressive orders move price</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                Watch for trades at the ask (aggressive buys) vs at the bid (aggressive sells). The
                aggressor is the one with urgency.
              </p>
            </div>
          </div>

          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="bg-primary/20 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
              <span className="text-primary font-bold">3</span>
            </div>
            <div>
              <h4 className="font-medium">Depth shows resistance, not direction</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                Large bids don't mean price goes up - they mean it takes more selling to push down.
                Think of depth as walls, not magnets.
              </p>
            </div>
          </div>

          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="bg-primary/20 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
              <span className="text-primary font-bold">4</span>
            </div>
            <div>
              <h4 className="font-medium">The book can lie</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                Spoofing, iceberg orders, and manipulation exist. Don't blindly trust large orders.
                The tape (actual trades) is harder to fake.
              </p>
            </div>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 9: NEXT STEPS */}
      {/* =================================================================== */}
      <LessonRow>
        <SectionHeader
          title="What's Next?"
          subtitle="Continue building your skills"
        />

        <p className="text-muted-foreground mb-6 leading-relaxed">
          Now that you understand order flow, the next step is understanding the costs of trading on
          Kalshi - fees, slippage, and how they affect your profitability.
        </p>

        {nextGuide && (
          <NextStepBlock
            href={`/${nextGuide.slug}`}
            title={nextGuide.title}
            description="Understand Kalshi's fee structure and how trading costs affect your bottom line."
            buttonText="Continue to Next Lesson"
          />
        )}
      </LessonRow>
    </LessonLayout>
  );
}
