/**
 * Lesson: What is Kalshi?
 *
 * Introduction to prediction markets and how Kalshi works.
 * This is the foundational lesson - assumes ZERO prior knowledge.
 */

import { DollarSign, TrendingUp, Users, Scale, History } from 'lucide-react';
import { getGuide, getNextGuide } from '@/data/guides';
import { LessonLayout, LessonRow, LessonHeader, SectionHeader } from '../LessonLayout';
import {
  ObjectiveBlock,
  ConstraintBlock,
  TipBlock,
  InsightBlock,
  WarningBlock,
  ConceptCard,
  ConceptBlock,
  NextStepBlock,
  Term,
  ExampleBox,
} from '../blocks';
import { GlossaryTooltip } from '@/components/atoms/GlossaryTooltip';
import { ProbabilityPriceWidget } from '../widgets/ProbabilityPriceWidget';

// Custom glossary terms for this lesson
const terms = {
  predictionMarket: {
    term: 'Prediction Market',
    definition:
      'A market where you trade contracts based on whether future events will happen. Prices reflect collective probability estimates.',
    category: 'Markets',
  },
  exchange: {
    term: 'Exchange',
    definition:
      'A marketplace where buyers and sellers trade directly with each other. You trade against other users, not against a house.',
    category: 'Markets',
  },
  contract: {
    term: 'Contract',
    definition: "The basic unit you trade. Pays $1 if the outcome happens, $0 if it doesn't.",
    category: 'Contracts',
  },
  orderBook: {
    term: 'Order Book',
    definition:
      'A list of all buy and sell orders, organized by price. Shows what people are willing to pay and accept.',
    category: 'Trading',
  },
  binary: {
    term: 'Binary',
    definition: 'Having only two possible outcomes. YES or NO. True or false. 1 or 0.',
    category: 'Concepts',
  },
  cftc: {
    term: 'CFTC',
    definition:
      'Commodity Futures Trading Commission - the US government agency that regulates derivatives and prediction markets.',
    category: 'Regulation',
  },
};

export function WhatIsKalshiLesson() {
  const guide = getGuide('what-is-kalshi')!;
  const nextGuide = getNextGuide('what-is-kalshi');

  return (
    <LessonLayout>
      {/* Header */}
      <LessonRow>
        <LessonHeader
          title={guide.title}
          description={guide.description}
          duration="15 min"
          category="Fundamentals"
        />
      </LessonRow>

      {/* Objective */}
      <LessonRow
        aside={
          <ConstraintBlock
            title="What We'll Cover"
            items={[
              'What prediction markets are',
              'How Kalshi differs from sportsbooks',
              'How YES/NO contracts work',
              'Why prices equal probabilities',
            ]}
          />
        }
      >
        <ObjectiveBlock>
          By the end of this lesson, you'll understand what Kalshi is, how prediction markets work,
          and why trading on an exchange is fundamentally different from betting at a sportsbook.
        </ObjectiveBlock>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 1: THE BIG PICTURE */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <InsightBlock title="Why This Matters">
            Understanding that Kalshi is an exchange - not a sportsbook - changes how you think
            about every trade you make.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="The Big Picture"
          subtitle="What exactly is Kalshi?"
        />

        <p className="text-foreground leading-relaxed">
          <strong>
            Kalshi is an exchange where you trade contracts on the outcomes of real-world events.
          </strong>
        </p>

        <p className="text-muted-foreground mt-4 leading-relaxed">
          That sentence contains three concepts that might be unfamiliar: <em>exchange</em>,{' '}
          <em>contracts</em>, and <em>trading on outcomes</em>. Let's break each one down.
        </p>

        <div className="mt-6 space-y-4">
          <Term term="Exchange">
            A marketplace where buyers and sellers trade <em>directly with each other</em>. The
            exchange itself doesn't take sides - it just matches people who want to buy with people
            who want to sell. Think of it like eBay for predictions: the platform connects traders
            but doesn't own the items being sold.
          </Term>

          <Term term="Contracts">
            The "items" being traded. Each contract represents a bet on a specific outcome. If
            you're right, the contract pays $1.00. If you're wrong, it pays $0.00. Nothing in
            between - it's binary, like a light switch.
          </Term>

          <Term term="Trading on outcomes">
            Instead of buying stocks or commodities, you're buying and selling contracts based on
            whether something will happen. "Will the Lakers win tonight?" "Will it snow in NYC this
            week?" "Will inflation exceed 3%?" These are all tradeable events on Kalshi.
          </Term>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 2: PREDICTION MARKETS - HISTORY AND CONTEXT */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="Historical Context">
            Prediction markets have accurately forecasted elections, product launches, and economic
            indicators - often better than polls or experts.
          </TipBlock>
        }
      >
        <SectionHeader
          title="What Are Prediction Markets?"
          subtitle="A brief history and why they exist"
        />

        <div className="mb-6 flex items-start gap-3">
          <div className="bg-primary/10 flex h-10 w-10 flex-shrink-0 items-center justify-center rounded-full">
            <History className="text-primary h-5 w-5" />
          </div>
          <div>
            <h4 className="font-semibold">A Brief History</h4>
            <p className="text-muted-foreground mt-1 text-sm">
              The idea isn't new. Informal betting markets have predicted elections since the 1800s.
              In 1988, the University of Iowa created the Iowa Electronic Markets (IEM) to forecast
              presidential elections - and it consistently outperformed polls.
            </p>
          </div>
        </div>

        <p className="text-muted-foreground leading-relaxed">
          A <GlossaryTooltip term={terms.predictionMarket}>prediction market</GlossaryTooltip> is
          simply a market where the "price" of a contract reflects what people{' '}
          <em>collectively believe</em>
          is the probability of something happening.
        </p>

        <ExampleBox title="Why Prediction Markets Work">
          <p className="mb-3">
            Imagine 1,000 people with different information about whether the Lakers will win
            tonight. Some have injury reports, some follow betting trends, some have insider
            knowledge.
          </p>
          <p className="mb-3">
            If each person trades based on their information, the market price becomes a kind of
            "weighted average" of everyone's knowledge. This is called the{' '}
            <strong>wisdom of crowds</strong>.
          </p>
          <p className="text-muted-foreground text-sm">
            Research shows that prediction market prices are often more accurate than expert
            forecasts, precisely because they aggregate information from many sources.
          </p>
        </ExampleBox>

        <div className="mt-6">
          <h4 className="mb-3 font-semibold">Why Kalshi Matters</h4>
          <p className="text-muted-foreground leading-relaxed">
            Kalshi is the first prediction market{' '}
            <strong>
              regulated by the <GlossaryTooltip term={terms.cftc}>CFTC</GlossaryTooltip>
            </strong>
            (Commodity Futures Trading Commission) in the United States. This makes it legal,
            regulated, and available to retail traders. Before Kalshi (founded 2018, launched 2021),
            prediction markets were either academic experiments or operated in regulatory gray
            areas.
          </p>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 3: HOW CONTRACTS WORK */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <WarningBlock title="The $1 Rule">
            Every contract settles to exactly $1.00 (YES wins) or $0.00 (NO wins). There's no middle
            ground. This binary nature is what makes the math simple.
          </WarningBlock>
        }
      >
        <SectionHeader
          title="How Contracts Work"
          subtitle="The building blocks of prediction markets"
        />

        <p className="text-muted-foreground leading-relaxed">
          Every market on Kalshi asks a yes-or-no question. Your job is to decide which answer you
          believe in - and how much you're willing to pay for it.
        </p>

        <div className="bg-card mt-6 rounded-lg border p-5">
          <h4 className="mb-4 font-semibold">The Basic Structure</h4>
          <div className="grid gap-4 sm:grid-cols-2">
            <div className="rounded-lg border border-emerald-500/30 bg-emerald-500/10 p-4">
              <div className="mb-2 font-semibold text-emerald-400">YES Contract</div>
              <ul className="text-muted-foreground space-y-2 text-sm">
                <li>
                  • Pays <strong>$1.00</strong> if the event happens
                </li>
                <li>
                  • Pays <strong>$0.00</strong> if the event doesn't happen
                </li>
                <li>• You buy YES if you think it WILL happen</li>
              </ul>
            </div>
            <div className="rounded-lg border border-rose-500/30 bg-rose-500/10 p-4">
              <div className="mb-2 font-semibold text-rose-400">NO Contract</div>
              <ul className="text-muted-foreground space-y-2 text-sm">
                <li>
                  • Pays <strong>$1.00</strong> if the event doesn't happen
                </li>
                <li>
                  • Pays <strong>$0.00</strong> if the event does happen
                </li>
                <li>• You buy NO if you think it WON'T happen</li>
              </ul>
            </div>
          </div>
        </div>

        <ExampleBox title="Walk Through: Lakers vs Celtics">
          <div className="space-y-4">
            <p>
              <strong>The question:</strong> "Will the Lakers win tonight?"
            </p>
            <p>
              <strong>Current price:</strong> YES is trading at <strong>60¢</strong>
            </p>
            <p className="text-muted-foreground text-sm">
              This means the market collectively believes there's about a 60% chance the Lakers win.
            </p>

            <div className="mt-4 border-t pt-4">
              <p className="mb-2 font-medium">If you buy YES at 60¢:</p>
              <ul className="text-muted-foreground space-y-1 text-sm">
                <li>
                  • Lakers win → You receive $1.00 →{' '}
                  <span className="text-emerald-400">Profit: 40¢</span>
                </li>
                <li>
                  • Lakers lose → You receive $0.00 →{' '}
                  <span className="text-rose-400">Loss: 60¢</span>
                </li>
              </ul>
            </div>

            <div className="border-t pt-4">
              <p className="mb-2 font-medium">If you buy NO at 40¢:</p>
              <ul className="text-muted-foreground space-y-1 text-sm">
                <li>
                  • Lakers lose → You receive $1.00 →{' '}
                  <span className="text-emerald-400">Profit: 60¢</span>
                </li>
                <li>
                  • Lakers win → You receive $0.00 →{' '}
                  <span className="text-rose-400">Loss: 40¢</span>
                </li>
              </ul>
            </div>

            <p className="text-muted-foreground mt-4 text-xs">
              Notice: YES at 60¢ + NO at 40¢ = $1.00. This is always true because one side must win.
            </p>
          </div>
        </ExampleBox>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 4: PRICE = PROBABILITY */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <InsightBlock title="The Key Insight">
            If you think the true probability is higher than the market price, buy YES. If you think
            it's lower, buy NO. This is the foundation of all trading decisions.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="Price = Probability"
          subtitle="The most important concept in prediction markets"
        />

        <p className="text-muted-foreground leading-relaxed">
          Here's the mental model that makes everything click:
        </p>

        <div className="from-primary/10 via-primary/5 border-primary/30 my-6 rounded-lg border bg-gradient-to-br to-transparent p-5">
          <p className="text-center text-lg font-semibold">
            A contract trading at <span className="text-primary">65¢</span> means the market
            believes there's about a <span className="text-primary">65% chance</span> it happens.
          </p>
        </div>

        <p className="text-muted-foreground leading-relaxed">
          Why does this work? Because of how the payouts are structured:
        </p>

        <ul className="mt-4 space-y-3">
          <li className="flex items-start gap-3">
            <span className="bg-primary/20 text-primary flex h-6 w-6 flex-shrink-0 items-center justify-center rounded-full text-xs font-bold">
              1
            </span>
            <span className="text-muted-foreground">
              If something has a 65% chance of happening, a fair price for a $1 payout is 65¢. Over
              many bets, you'd expect to win 65% × $1 = 65¢ on average.
            </span>
          </li>
          <li className="flex items-start gap-3">
            <span className="bg-primary/20 text-primary flex h-6 w-6 flex-shrink-0 items-center justify-center rounded-full text-xs font-bold">
              2
            </span>
            <span className="text-muted-foreground">
              If you pay MORE than 65¢ for a 65% event, you're overpaying. Long-term, you'll lose
              money.
            </span>
          </li>
          <li className="flex items-start gap-3">
            <span className="bg-primary/20 text-primary flex h-6 w-6 flex-shrink-0 items-center justify-center rounded-full text-xs font-bold">
              3
            </span>
            <span className="text-muted-foreground">
              If you pay LESS than 65¢ for a 65% event, you're getting a bargain. Long-term, you'll
              profit.
            </span>
          </li>
        </ul>

        <ProbabilityPriceWidget />
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 5: EXCHANGE VS SPORTSBOOK */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="Why This Distinction Matters">
            On an exchange, skilled traders are welcome. On sportsbooks, winning too much gets you
            limited or banned. The business models are fundamentally different.
          </TipBlock>
        }
      >
        <SectionHeader
          title="Exchange vs. Sportsbook"
          subtitle="Why Kalshi is different from DraftKings"
        />

        <p className="text-muted-foreground mb-6 leading-relaxed">
          Most people are familiar with sportsbooks (DraftKings, FanDuel, BetMGM). Kalshi works
          completely differently. Understanding the difference is crucial.
        </p>

        <div className="overflow-x-auto">
          <table className="w-full text-sm">
            <thead>
              <tr className="border-b">
                <th className="w-1/4 py-3 text-left font-medium"></th>
                <th className="py-3 text-left font-medium">Kalshi (Exchange)</th>
                <th className="py-3 text-left font-medium">Sportsbook</th>
              </tr>
            </thead>
            <tbody className="divide-y">
              <tr>
                <td className="text-muted-foreground py-3 font-medium">Who do you bet against?</td>
                <td className="py-3">Other users like you</td>
                <td className="py-3">The sportsbook itself (the "house")</td>
              </tr>
              <tr>
                <td className="text-muted-foreground py-3 font-medium">Who sets the prices?</td>
                <td className="py-3">Supply and demand from all traders</td>
                <td className="py-3">The sportsbook's traders and algorithms</td>
              </tr>
              <tr>
                <td className="text-muted-foreground py-3 font-medium">Can you exit early?</td>
                <td className="py-3">Yes - sell your position anytime</td>
                <td className="py-3">Usually no (some allow "cash out" at bad prices)</td>
              </tr>
              <tr>
                <td className="text-muted-foreground py-3 font-medium">
                  Can you set your own price?
                </td>
                <td className="py-3">Yes - use limit orders</td>
                <td className="py-3">No - take their odds or don't bet</td>
              </tr>
              <tr>
                <td className="text-muted-foreground py-3 font-medium">How do they make money?</td>
                <td className="py-3">Small fee per trade (both sides)</td>
                <td className="py-3">The "juice" (built-in edge on every bet)</td>
              </tr>
              <tr>
                <td className="text-muted-foreground py-3 font-medium">If you win consistently?</td>
                <td className="py-3">Welcome - you're providing liquidity</td>
                <td className="py-3">You get limited or banned</td>
              </tr>
            </tbody>
          </table>
        </div>

        <div className="mt-6 grid gap-4 sm:grid-cols-2">
          <ConceptBlock title="The Sportsbook Model">
            <p>
              A sportsbook is like a casino. They set the rules, they set the odds, and they always
              have a mathematical edge. When you bet, you're betting against <em>them</em>. If you
              win too much, they see you as a threat to their profits - so they limit your account.
            </p>
          </ConceptBlock>
          <ConceptBlock title="The Exchange Model">
            <p>
              An exchange is like a stock market. Kalshi doesn't care who wins - they make money on
              every trade through small fees. Your counterparty is another trader, not the house.
              Skilled traders are welcome because they add{' '}
              <GlossaryTooltip term={terms.orderBook}>liquidity</GlossaryTooltip>.
            </p>
          </ConceptBlock>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 6: THE ORDER BOOK */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <WarningBlock title="Not Like Sportsbooks">
            On a sportsbook, you see one price and take it. On an exchange, you see a range of
            prices and can choose where to trade - or set your own price.
          </WarningBlock>
        }
      >
        <SectionHeader
          title="The Order Book"
          subtitle="How prices are determined on an exchange"
        />

        <p className="text-muted-foreground leading-relaxed">
          Unlike a sportsbook where you see one price, an exchange has an{' '}
          <GlossaryTooltip term={terms.orderBook}>order book</GlossaryTooltip> - a list of all the
          buy and sell orders from every trader.
        </p>

        <div className="bg-card mt-6 rounded-lg border p-5">
          <h4 className="mb-4 font-semibold">Anatomy of an Order Book</h4>

          <div className="grid gap-4 sm:grid-cols-2">
            <div>
              <div className="mb-2 text-sm font-medium text-emerald-400">Bids (Buy Orders)</div>
              <p className="text-muted-foreground mb-3 text-sm">
                What buyers are willing to pay. The highest bid is the "best bid."
              </p>
              <div className="space-y-1 font-mono text-sm">
                <div className="flex justify-between rounded bg-emerald-500/10 p-2">
                  <span>55¢</span>
                  <span className="text-muted-foreground">500 contracts</span>
                </div>
                <div className="flex justify-between rounded bg-emerald-500/5 p-2">
                  <span>54¢</span>
                  <span className="text-muted-foreground">300 contracts</span>
                </div>
                <div className="flex justify-between rounded bg-emerald-500/5 p-2">
                  <span>53¢</span>
                  <span className="text-muted-foreground">800 contracts</span>
                </div>
              </div>
            </div>
            <div>
              <div className="mb-2 text-sm font-medium text-rose-400">Asks (Sell Orders)</div>
              <p className="text-muted-foreground mb-3 text-sm">
                What sellers want to receive. The lowest ask is the "best ask."
              </p>
              <div className="space-y-1 font-mono text-sm">
                <div className="flex justify-between rounded bg-rose-500/10 p-2">
                  <span>58¢</span>
                  <span className="text-muted-foreground">400 contracts</span>
                </div>
                <div className="flex justify-between rounded bg-rose-500/5 p-2">
                  <span>59¢</span>
                  <span className="text-muted-foreground">250 contracts</span>
                </div>
                <div className="flex justify-between rounded bg-rose-500/5 p-2">
                  <span>60¢</span>
                  <span className="text-muted-foreground">600 contracts</span>
                </div>
              </div>
            </div>
          </div>

          <div className="bg-muted/50 mt-4 rounded p-3">
            <p className="text-sm">
              <strong>The Spread:</strong> The gap between best bid (55¢) and best ask (58¢) is
              called the "spread" - here it's 3¢. Tighter spreads mean more liquidity.
            </p>
          </div>
        </div>

        <div className="mt-6 space-y-4">
          <Term term="Market Order">
            Buy or sell immediately at whatever the best price is. You're guaranteed to trade, but
            not guaranteed the price. Use when speed matters more than price.
          </Term>
          <Term term="Limit Order">
            Set your own price and wait. "I'll buy at 54¢" sits in the order book until someone is
            willing to sell at 54¢. Use when price matters more than speed.
          </Term>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 7: KEY CONCEPTS SUMMARY */}
      {/* =================================================================== */}
      <LessonRow>
        <SectionHeader
          title="Key Concepts to Remember"
          subtitle="Your foundation for everything that follows"
        />

        <div className="grid gap-4 sm:grid-cols-2">
          <ConceptCard
            title="Binary Contracts"
            description="Every contract pays $1 or $0. No middle ground. One side wins, one side loses."
            icon={<Scale className="h-4 w-4" />}
          />
          <ConceptCard
            title="Price = Probability"
            description="A 65¢ price means roughly 65% probability. Use this to find mispriced markets."
            icon={<TrendingUp className="h-4 w-4" />}
          />
          <ConceptCard
            title="Trade Against Others"
            description="You're not betting against a house. Other traders are your counterparties."
            icon={<Users className="h-4 w-4" />}
          />
          <ConceptCard
            title="Real Money"
            description="Real USD, real payouts. This is a regulated financial exchange, not a game."
            icon={<DollarSign className="h-4 w-4" />}
          />
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 8: NEXT STEPS */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="Review the Glossary">
            If any terms are still unclear, check the{' '}
            <a
              href="/glossary"
              className="text-primary hover:underline"
            >
              Glossary
            </a>{' '}
            for detailed definitions and etymology.
          </TipBlock>
        }
      >
        <SectionHeader
          title="What's Next?"
          subtitle="Building on this foundation"
        />

        <p className="text-muted-foreground mb-6 leading-relaxed">
          Now that you understand what Kalshi is and how prediction markets work, the next step is
          learning about the specific types of bets available on sports markets: moneylines, point
          spreads, and totals.
        </p>

        {nextGuide && (
          <NextStepBlock
            href={`/${nextGuide.slug}`}
            title="Continue to Sports Betting 101"
            description={`Learn the three fundamental bet types - moneylines, spreads, and totals - and how they work on Kalshi.`}
            buttonText="Next Lesson"
          />
        )}
      </LessonRow>
    </LessonLayout>
  );
}
