/**
 * Dual Market Visualizer
 *
 * Shows how two markets relate for a single game.
 */

import { useState } from 'react';
import { cn } from '@/lib/utils';
import { ArrowRight, Check, X } from 'lucide-react';

type Outcome = 'lakers-win' | 'celtics-win' | null;

export function DualMarketVisualizer() {
  const [selectedOutcome, setSelectedOutcome] = useState<Outcome>(null);

  const getPositionState = (market: 'lakers' | 'celtics', side: 'yes' | 'no') => {
    if (!selectedOutcome) return 'neutral';

    // Lakers win scenario
    if (selectedOutcome === 'lakers-win') {
      if ((market === 'lakers' && side === 'yes') || (market === 'celtics' && side === 'no')) {
        return 'winner';
      }
      return 'loser';
    }

    // Celtics win scenario
    if (selectedOutcome === 'celtics-win') {
      if ((market === 'celtics' && side === 'yes') || (market === 'lakers' && side === 'no')) {
        return 'winner';
      }
      return 'loser';
    }

    return 'neutral';
  };

  const positionStyles = {
    neutral: 'bg-card border-border',
    winner: 'bg-emerald-500/20 border-emerald-500',
    loser: 'bg-rose-500/10 border-rose-500/50 opacity-60',
  };

  const PositionBox = ({
    market,
    side,
    price,
  }: {
    market: 'lakers' | 'celtics';
    side: 'yes' | 'no';
    price: number;
  }) => {
    const state = getPositionState(market, side);
    const label = `${market === 'lakers' ? 'Lakers' : 'Celtics'} ${side.toUpperCase()}`;

    return (
      <div
        className={cn('rounded-lg border p-3 text-center transition-all', positionStyles[state])}
      >
        <div className="text-sm font-medium">{label}</div>
        <div className="text-lg font-bold tabular-nums">${price.toFixed(2)}</div>
        {state === 'winner' && (
          <div className="mt-1 flex items-center justify-center gap-1 text-xs text-emerald-400">
            <Check className="h-3 w-3" />
            Pays $1.00
          </div>
        )}
        {state === 'loser' && (
          <div className="mt-1 flex items-center justify-center gap-1 text-xs text-rose-400">
            <X className="h-3 w-3" />
            Worth $0.00
          </div>
        )}
      </div>
    );
  };

  return (
    <div className="bg-card mt-4 rounded-lg border p-4">
      <div className="text-muted-foreground mb-4 text-xs font-semibold uppercase tracking-wider">
        Interactive: See how outcomes affect each position
      </div>

      {/* Outcome Selector */}
      <div className="mb-6 flex gap-2">
        <button
          onClick={() => setSelectedOutcome(selectedOutcome === 'lakers-win' ? null : 'lakers-win')}
          className={cn(
            'flex-1 rounded-lg border p-3 text-sm font-medium transition-colors',
            selectedOutcome === 'lakers-win'
              ? 'bg-primary/20 border-primary text-primary'
              : 'hover:bg-muted'
          )}
        >
          Lakers Win
        </button>
        <button
          onClick={() =>
            setSelectedOutcome(selectedOutcome === 'celtics-win' ? null : 'celtics-win')
          }
          className={cn(
            'flex-1 rounded-lg border p-3 text-sm font-medium transition-colors',
            selectedOutcome === 'celtics-win'
              ? 'bg-primary/20 border-primary text-primary'
              : 'hover:bg-muted'
          )}
        >
          Celtics Win
        </button>
      </div>

      {/* Markets Grid */}
      <div className="grid gap-6 md:grid-cols-2">
        {/* Lakers Market */}
        <div>
          <h4 className="mb-3 text-center text-sm font-semibold">
            Market A: &quot;Will Lakers win?&quot;
          </h4>
          <div className="grid grid-cols-2 gap-3">
            <PositionBox
              market="lakers"
              side="yes"
              price={0.65}
            />
            <PositionBox
              market="lakers"
              side="no"
              price={0.35}
            />
          </div>
        </div>

        {/* Celtics Market */}
        <div>
          <h4 className="mb-3 text-center text-sm font-semibold">
            Market B: &quot;Will Celtics win?&quot;
          </h4>
          <div className="grid grid-cols-2 gap-3">
            <PositionBox
              market="celtics"
              side="yes"
              price={0.35}
            />
            <PositionBox
              market="celtics"
              side="no"
              price={0.65}
            />
          </div>
        </div>
      </div>

      {/* Equivalence Indicator */}
      <div className="mt-6 border-t pt-4">
        <div className="text-muted-foreground mb-3 text-center text-xs">Equivalent positions:</div>
        <div className="flex items-center justify-center gap-2 text-sm">
          <span
            className={cn(
              'rounded px-2 py-1',
              selectedOutcome === 'lakers-win' ? 'bg-emerald-500/20 text-emerald-400' : 'bg-muted'
            )}
          >
            Lakers YES
          </span>
          <ArrowRight className="text-muted-foreground h-4 w-4" />
          <span
            className={cn(
              'rounded px-2 py-1',
              selectedOutcome === 'lakers-win' ? 'bg-emerald-500/20 text-emerald-400' : 'bg-muted'
            )}
          >
            Celtics NO
          </span>
          <span className="text-muted-foreground mx-2">|</span>
          <span
            className={cn(
              'rounded px-2 py-1',
              selectedOutcome === 'celtics-win' ? 'bg-emerald-500/20 text-emerald-400' : 'bg-muted'
            )}
          >
            Lakers NO
          </span>
          <ArrowRight className="text-muted-foreground h-4 w-4" />
          <span
            className={cn(
              'rounded px-2 py-1',
              selectedOutcome === 'celtics-win' ? 'bg-emerald-500/20 text-emerald-400' : 'bg-muted'
            )}
          >
            Celtics YES
          </span>
        </div>
      </div>

      <div className="text-muted-foreground mt-4 text-center text-xs">
        Click an outcome to see which positions win
      </div>
    </div>
  );
}
