/**
 * Market Maker Dashboard Preview
 *
 * Shows a preview of what market makers look at when analyzing markets.
 * Based on the client's NBA Value Dashboard design.
 */

import { cn } from '@/lib/utils';

interface GameRow {
  date: string;
  away: string;
  home: string;
  awayKalshi: number;
  homeKalshi: number;
  awayPoly: number;
  homePoly: number;
  awayKLiq: number;
  homeKLiq: number;
}

const sampleGames: GameRow[] = [
  {
    date: '2026-01-20',
    away: 'Atlanta Hawks',
    home: 'Portland Trail Blazers',
    awayKalshi: 0.5942,
    homeKalshi: 0.4042,
    awayPoly: 0.6,
    homePoly: 0.39,
    awayKLiq: 50100,
    homeKLiq: 202,
  },
  {
    date: '2026-01-20',
    away: 'Boston Celtics',
    home: 'Miami Heat',
    awayKalshi: 0.5643,
    homeKalshi: 0.4243,
    awayPoly: 0.56,
    homePoly: 0.44,
    awayKLiq: 42500,
    homeKLiq: 24100,
  },
  {
    date: '2026-01-20',
    away: 'Charlotte Hornets',
    home: 'Los Angeles Lakers',
    awayKalshi: 0.364,
    homeKalshi: 0.6241,
    awayPoly: 0.37,
    homePoly: 0.62,
    awayKLiq: 36300,
    homeKLiq: 309,
  },
];

function LiquidityBar({ value, max }: { value: number; max: number }) {
  const pct = Math.min((value / max) * 100, 100);

  // Color based on liquidity level
  const getColor = () => {
    if (pct > 60) return 'bg-emerald-500';
    if (pct > 30) return 'bg-yellow-500';
    return 'bg-rose-500';
  };

  return (
    <div className="flex items-center gap-2">
      <div className="bg-muted h-2 w-16 overflow-hidden rounded-full">
        <div
          className={cn('h-full rounded-full', getColor())}
          style={{ width: `${pct}%` }}
        />
      </div>
      <span className="text-xs tabular-nums">${(value / 1000).toFixed(1)}K</span>
    </div>
  );
}

export function MarketMakerDashboardPreview() {
  const maxLiquidity = Math.max(...sampleGames.flatMap((g) => [g.awayKLiq, g.homeKLiq]));

  return (
    <div className="bg-card overflow-hidden rounded-lg border">
      {/* Header */}
      <div className="bg-muted/50 flex items-center justify-between border-b px-4 py-2">
        <span className="text-sm font-semibold">NBA Value Dashboard</span>
        <div className="flex items-center gap-2 text-xs">
          <span className="rounded bg-emerald-500/20 px-2 py-0.5 text-emerald-400">MAKER</span>
          <span className="text-muted-foreground">Feed: live | Last update: 08:15:13</span>
        </div>
      </div>

      {/* Table */}
      <div className="overflow-x-auto">
        <table className="w-full text-xs">
          <thead>
            <tr className="bg-muted/30 border-b">
              <th className="text-muted-foreground px-3 py-2 text-left font-medium">GAME DATE</th>
              <th className="text-muted-foreground px-3 py-2 text-left font-medium">AWAY TEAM</th>
              <th className="text-muted-foreground px-3 py-2 text-left font-medium">HOME TEAM</th>
              <th className="text-muted-foreground px-3 py-2 text-center font-medium">AWAY K</th>
              <th className="text-muted-foreground px-3 py-2 text-center font-medium">HOME K</th>
              <th className="text-muted-foreground px-3 py-2 text-center font-medium">AWAY POLY</th>
              <th className="text-muted-foreground px-3 py-2 text-center font-medium">HOME POLY</th>
              <th className="text-muted-foreground px-3 py-2 text-left font-medium">AWAY K LIQ</th>
              <th className="text-muted-foreground px-3 py-2 text-left font-medium">HOME K LIQ</th>
            </tr>
          </thead>
          <tbody className="divide-y">
            {sampleGames.map((game, i) => (
              <tr
                key={i}
                className="hover:bg-muted/30"
              >
                <td className="text-muted-foreground px-3 py-2">{game.date}</td>
                <td className="px-3 py-2 font-medium">{game.away}</td>
                <td className="px-3 py-2 font-medium">{game.home}</td>
                <td className="px-3 py-2 text-center tabular-nums">{game.awayKalshi.toFixed(4)}</td>
                <td className="px-3 py-2 text-center tabular-nums">{game.homeKalshi.toFixed(4)}</td>
                <td className="text-muted-foreground px-3 py-2 text-center tabular-nums">
                  {game.awayPoly.toFixed(4)}
                </td>
                <td className="text-muted-foreground px-3 py-2 text-center tabular-nums">
                  {game.homePoly.toFixed(4)}
                </td>
                <td className="px-3 py-2">
                  <LiquidityBar
                    value={game.awayKLiq}
                    max={maxLiquidity}
                  />
                </td>
                <td className="px-3 py-2">
                  <LiquidityBar
                    value={game.homeKLiq}
                    max={maxLiquidity}
                  />
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>

      {/* Legend */}
      <div className="bg-muted/30 border-t px-4 py-3">
        <div className="text-muted-foreground flex items-center gap-6 text-xs">
          <div className="flex items-center gap-2">
            <strong>K</strong> = Kalshi price
          </div>
          <div className="flex items-center gap-2">
            <strong>POLY</strong> = Polymarket price
          </div>
          <div className="flex items-center gap-2">
            <strong>LIQ</strong> = Liquidity depth
          </div>
          <div className="flex items-center gap-2">
            <div className="h-2 w-3 rounded bg-emerald-500" />
            High liquidity
          </div>
          <div className="flex items-center gap-2">
            <div className="h-2 w-3 rounded bg-rose-500" />
            Low liquidity
          </div>
        </div>
      </div>

      {/* Insight callout */}
      <div className="border-t bg-violet-500/5 px-4 py-3">
        <div className="text-xs">
          <strong className="text-violet-400">What to look for:</strong>{' '}
          <span className="text-muted-foreground">
            Market makers compare Kalshi prices to other exchanges (like Polymarket) to find
            mispricing. They also check liquidity — low liquidity on one side may indicate
            opportunity to provide quotes.
          </span>
        </div>
      </div>
    </div>
  );
}
