/**
 * Order Book Visualizer
 *
 * Shows how order books work with bids and asks.
 */

import { useState } from 'react';
import { cn } from '@/lib/utils';
import { Users } from 'lucide-react';

type Market = 'lakers' | 'celtics';
type Side = 'yes' | 'no';

interface OrderLevel {
  price: number;
  size: number;
  isUserOrder?: boolean;
}

interface OrderBookProps {
  market: Market;
  side: Side;
  bids: OrderLevel[];
  asks: OrderLevel[];
  onPlaceOrder?: (price: number) => void;
}

function MiniOrderBook({ market, side, bids, asks, onPlaceOrder }: OrderBookProps) {
  const label = `${market === 'lakers' ? 'Lakers' : 'Celtics'} ${side.toUpperCase()}`;
  const maxSize = Math.max(...bids.map((b) => b.size), ...asks.map((a) => a.size));

  return (
    <div className="bg-card rounded-lg border p-3">
      <div className="text-muted-foreground mb-2 text-xs font-semibold">{label}</div>

      {/* Asks (sells) */}
      <div className="mb-2 space-y-1">
        {asks
          .slice()
          .reverse()
          .map((ask, i) => (
            <button
              key={`ask-${i}`}
              onClick={() => onPlaceOrder?.(ask.price)}
              className="hover:bg-muted/50 flex w-full items-center gap-2 rounded px-1 py-0.5 text-xs"
            >
              <div
                className="h-3 rounded-sm bg-rose-500/30"
                style={{ width: `${(ask.size / maxSize) * 100}%` }}
              />
              <span className="w-10 tabular-nums text-rose-400">${ask.price.toFixed(2)}</span>
              <span className="text-muted-foreground tabular-nums">{ask.size}</span>
            </button>
          ))}
      </div>

      {/* Spread */}
      <div className="text-muted-foreground border-y border-dashed py-1 text-center text-xs">
        Spread: ${(asks[0].price - bids[0].price).toFixed(2)}
      </div>

      {/* Bids (buys) */}
      <div className="mt-2 space-y-1">
        {bids.map((bid, i) => (
          <button
            key={`bid-${i}`}
            onClick={() => onPlaceOrder?.(bid.price)}
            className={cn(
              'hover:bg-muted/50 flex w-full items-center gap-2 rounded px-1 py-0.5 text-xs',
              bid.isUserOrder && 'bg-primary/10 ring-primary/50 ring-1'
            )}
          >
            <div
              className={cn(
                'h-3 rounded-sm',
                bid.isUserOrder ? 'bg-primary/50' : 'bg-emerald-500/30'
              )}
              style={{ width: `${(bid.size / maxSize) * 100}%` }}
            />
            <span
              className={cn(
                'w-10 tabular-nums',
                bid.isUserOrder ? 'text-primary' : 'text-emerald-400'
              )}
            >
              ${bid.price.toFixed(2)}
            </span>
            <span className="text-muted-foreground tabular-nums">
              {bid.size}
              {bid.isUserOrder && <span className="text-primary ml-1">(you)</span>}
            </span>
          </button>
        ))}
      </div>
    </div>
  );
}

export function OrderBookVisualizer() {
  const [userOrderMarket, setUserOrderMarket] = useState<{ market: Market; side: Side } | null>(
    null
  );

  // Sample order book data
  const lakersYesBids: OrderLevel[] = [
    { price: 0.64, size: 500 },
    { price: 0.63, size: 300 },
    { price: 0.62, size: 800 },
  ];
  const lakersYesAsks: OrderLevel[] = [
    { price: 0.66, size: 400 },
    { price: 0.67, size: 600 },
    { price: 0.68, size: 200 },
  ];

  const celticsNoBids: OrderLevel[] = [
    {
      price: 0.64,
      size: 200,
      isUserOrder: userOrderMarket?.market === 'celtics' && userOrderMarket?.side === 'no',
    },
    { price: 0.63, size: 150 },
    { price: 0.62, size: 400 },
  ];
  const celticsNoAsks: OrderLevel[] = [
    { price: 0.66, size: 300 },
    { price: 0.67, size: 250 },
    { price: 0.68, size: 150 },
  ];

  const handlePlaceOrder = (market: Market, side: Side) => {
    setUserOrderMarket({ market, side });
  };

  return (
    <div className="bg-muted/30 rounded-lg border p-4">
      <div className="text-muted-foreground mb-4 text-xs font-semibold uppercase tracking-wider">
        Order Books for &quot;Lakers Win&quot; Outcome
      </div>

      <div className="grid gap-4 md:grid-cols-2">
        <div>
          <div className="mb-2 flex items-center gap-2 text-sm font-medium">
            Lakers YES
            <span className="text-muted-foreground text-xs font-normal">(Lakers win market)</span>
          </div>
          <MiniOrderBook
            market="lakers"
            side="yes"
            bids={lakersYesBids}
            asks={lakersYesAsks}
            onPlaceOrder={() => handlePlaceOrder('lakers', 'yes')}
          />
        </div>

        <div>
          <div className="mb-2 flex items-center gap-2 text-sm font-medium">
            Celtics NO
            <span className="text-muted-foreground text-xs font-normal">(Celtics win market)</span>
          </div>
          <MiniOrderBook
            market="celtics"
            side="no"
            bids={celticsNoBids}
            asks={celticsNoAsks}
            onPlaceOrder={() => handlePlaceOrder('celtics', 'no')}
          />
        </div>
      </div>

      {/* Key insight */}
      <div className="mt-4 rounded-lg border border-violet-500/30 bg-violet-500/10 p-3">
        <div className="flex items-start gap-2">
          <Users className="mt-0.5 h-4 w-4 flex-shrink-0 text-violet-400" />
          <div className="text-muted-foreground text-sm">
            <strong className="text-violet-300">Same outcome, different books:</strong> Both
            positions pay $1 if Lakers win, but notice the different order sizes and spread. Market
            makers choose where to post orders based on queue position and liquidity.
          </div>
        </div>
      </div>

      {userOrderMarket && (
        <div className="text-muted-foreground mt-3 text-center text-xs">
          Your order is highlighted in the{' '}
          {userOrderMarket.market === 'celtics' ? 'Celtics NO' : 'Lakers YES'} book
        </div>
      )}
    </div>
  );
}
