/**
 * Probability Price Widget
 *
 * Interactive widget demonstrating the relationship between
 * contract prices and implied probability.
 */

import { useState } from 'react';
import { cn } from '@/lib/utils';
import { TrendingUp, TrendingDown, Minus } from 'lucide-react';

export function ProbabilityPriceWidget() {
  const [price, setPrice] = useState(65);
  const [beliefProbability, setBeliefProbability] = useState(70);

  const impliedProbability = price;
  const profit = 100 - price; // cents profit if you win
  const loss = price; // cents loss if you lose
  const edge = beliefProbability - impliedProbability;

  // Calculate expected value based on your belief
  // EV = (probability * win amount) - ((1 - probability) * loss amount)
  const ev = (beliefProbability / 100) * profit - ((100 - beliefProbability) / 100) * loss;

  const getEdgeStatus = () => {
    if (edge > 3) return { text: '+EV (Good Bet)', color: 'text-emerald-400', icon: TrendingUp };
    if (edge < -3) return { text: '-EV (Bad Bet)', color: 'text-rose-400', icon: TrendingDown };
    return { text: 'Neutral', color: 'text-muted-foreground', icon: Minus };
  };

  const edgeStatus = getEdgeStatus();
  const EdgeIcon = edgeStatus.icon;

  return (
    <div className="bg-muted/30 mt-6 rounded-lg border p-4">
      <div className="text-muted-foreground mb-4 text-xs font-semibold uppercase tracking-wider">
        Interactive: Price vs Probability
      </div>

      <div className="grid gap-6 lg:grid-cols-2">
        {/* Controls */}
        <div className="space-y-6">
          {/* Market Price Slider */}
          <div>
            <div className="mb-2 flex items-center justify-between">
              <label className="text-sm font-medium">Market Price</label>
              <span className="text-primary text-lg font-bold">{price}¢</span>
            </div>
            <input
              type="range"
              min="5"
              max="95"
              value={price}
              onChange={(e) => setPrice(Number(e.target.value))}
              className="bg-muted accent-primary h-2 w-full cursor-pointer appearance-none rounded-lg"
            />
            <div className="text-muted-foreground mt-1 flex justify-between text-xs">
              <span>5¢</span>
              <span>Implied probability: {impliedProbability}%</span>
              <span>95¢</span>
            </div>
          </div>

          {/* Your Belief Slider */}
          <div>
            <div className="mb-2 flex items-center justify-between">
              <label className="text-sm font-medium">Your Belief (True Probability)</label>
              <span className="text-lg font-bold text-violet-400">{beliefProbability}%</span>
            </div>
            <input
              type="range"
              min="5"
              max="95"
              value={beliefProbability}
              onChange={(e) => setBeliefProbability(Number(e.target.value))}
              className="bg-muted h-2 w-full cursor-pointer appearance-none rounded-lg accent-violet-500"
            />
            <div className="text-muted-foreground mt-1 flex justify-between text-xs">
              <span>5%</span>
              <span>What you think the real odds are</span>
              <span>95%</span>
            </div>
          </div>

          {/* Edge Display */}
          <div
            className={cn(
              'rounded-lg border p-4',
              edge > 3 && 'border-emerald-500/30 bg-emerald-500/10',
              edge < -3 && 'border-rose-500/30 bg-rose-500/10',
              Math.abs(edge) <= 3 && 'bg-muted border-border'
            )}
          >
            <div className="mb-2 flex items-center gap-2">
              <EdgeIcon className={cn('h-5 w-5', edgeStatus.color)} />
              <span className={cn('font-semibold', edgeStatus.color)}>{edgeStatus.text}</span>
            </div>
            <p className="text-muted-foreground text-sm">
              {edge > 3 &&
                `You believe the event is ${edge}% more likely than the market price implies. Buying YES at this price has positive expected value.`}
              {edge < -3 &&
                `The market thinks it's ${Math.abs(edge)}% more likely than you do. Buying YES at this price has negative expected value.`}
              {Math.abs(edge) <= 3 &&
                'Your belief roughly matches the market price. Neither side has a clear edge.'}
            </p>
          </div>
        </div>

        {/* Results Display */}
        <div className="space-y-4">
          {/* Scenario Box */}
          <div className="bg-card rounded-lg border p-4">
            <h4 className="mb-4 font-semibold">If You Buy YES at {price}¢</h4>

            <div className="space-y-3">
              <div className="flex items-center justify-between rounded bg-emerald-500/10 p-3">
                <span className="text-sm">Event happens (you win)</span>
                <span className="font-semibold text-emerald-400">+{profit}¢</span>
              </div>
              <div className="flex items-center justify-between rounded bg-rose-500/10 p-3">
                <span className="text-sm">Event doesn't happen (you lose)</span>
                <span className="font-semibold text-rose-400">-{loss}¢</span>
              </div>
            </div>

            <div className="mt-4 border-t pt-4">
              <div className="flex items-center justify-between">
                <span className="text-muted-foreground text-sm">Expected Value (per contract)</span>
                <span
                  className={cn(
                    'text-lg font-bold',
                    ev > 0 ? 'text-emerald-400' : ev < 0 ? 'text-rose-400' : 'text-muted-foreground'
                  )}
                >
                  {ev > 0 ? '+' : ''}
                  {ev.toFixed(1)}¢
                </span>
              </div>
              <p className="text-muted-foreground mt-2 text-xs">
                Based on your belief of {beliefProbability}% true probability
              </p>
            </div>
          </div>

          {/* Key Insight */}
          <div className="bg-primary/10 border-primary/30 rounded-lg border p-3">
            <p className="text-muted-foreground text-sm">
              <strong className="text-primary">The Trading Edge:</strong> Profitable trading means
              finding situations where your assessment of probability differs from the market's.
              When you believe the true probability is higher than the price, buy YES. When lower,
              buy NO.
            </p>
          </div>

          {/* Math Explanation */}
          <div className="text-muted-foreground bg-muted/50 rounded p-3 text-xs">
            <p className="mb-1 font-medium">Expected Value Formula:</p>
            <code className="block text-[10px]">EV = (P × Win) - ((1-P) × Loss)</code>
            <code className="mt-1 block text-[10px]">
              EV = ({beliefProbability}% × {profit}¢) - ({100 - beliefProbability}% × {loss}¢)
            </code>
            <code className="mt-1 block text-[10px]">
              EV = {((beliefProbability / 100) * profit).toFixed(1)}¢ -{' '}
              {(((100 - beliefProbability) / 100) * loss).toFixed(1)}¢ = {ev.toFixed(1)}¢
            </code>
          </div>
        </div>
      </div>
    </div>
  );
}
