/**
 * HTTP Relay
 *
 * Forwards HTTP requests from client to Kalshi API with byte-faithful precision.
 */
import axios from 'axios';
import { ValidationError, NetworkError, TimeoutError, } from '@galactus/shared';
/**
 * Validates an HTTP relay request.
 *
 * @param request - Request to validate
 * @throws ValidationError if request is invalid
 */
export function validateHttpRequest(request) {
    if (!request.id || typeof request.id !== 'string') {
        throw new ValidationError('Missing or invalid request id');
    }
    if (!request.method || !['GET', 'POST', 'PUT', 'DELETE', 'PATCH'].includes(request.method)) {
        throw new ValidationError('Missing or invalid method');
    }
    if (!request.url || typeof request.url !== 'string') {
        throw new ValidationError('Missing or invalid url');
    }
    if (!request.headers || typeof request.headers !== 'object') {
        throw new ValidationError('Missing or invalid headers');
    }
    // Validate URL is a valid HTTP/HTTPS URL
    try {
        const url = new URL(request.url);
        if (!['http:', 'https:'].includes(url.protocol)) {
            throw new ValidationError('URL must use http or https protocol');
        }
    }
    catch (_error) {
        throw new ValidationError('Invalid URL format');
    }
    // Validate body is string if present
    if (request.body !== undefined && typeof request.body !== 'string') {
        throw new ValidationError('Body must be a string');
    }
}
/**
 * Forwards an HTTP request from client to Kalshi API.
 *
 * Forwards the request byte-faithfully:
 * - All headers forwarded exactly as received
 * - Body forwarded exactly as received (no parsing/re-stringifying)
 * - Query parameters preserved exactly
 *
 * @param request - HttpRelayRequest from client
 * @param config - Server configuration
 * @param logger - Logger instance
 * @returns HttpRelayResponse with Kalshi API response
 * @throws RelayError on forwarding failures
 */
export async function forwardHttpRequest(request, config, logger) {
    // Validate request
    validateHttpRequest(request);
    // Production URL validation: restrict to Kalshi API
    if (config.nodeEnv === 'production') {
        const url = new URL(request.url);
        const kalshiHost = new URL(config.kalshiBaseUrl).hostname;
        if (url.hostname !== kalshiHost) {
            throw new ValidationError(`URL must be from ${kalshiHost} in production`);
        }
    }
    logger.debug('Forwarding HTTP request', {
        id: request.id,
        method: request.method,
        url: request.url,
        hasBody: !!request.body,
    });
    try {
        // Prepare axios request config
        const axiosConfig = {
            method: request.method,
            url: request.url,
            headers: {
                ...request.headers,
                // Remove hop-by-hop headers that shouldn't be forwarded
                connection: undefined,
                'keep-alive': undefined,
                'transfer-encoding': undefined,
            },
            data: request.body,
            timeout: config.httpTimeoutMs,
            // Important: Don't transform response - we want raw bytes
            transformResponse: [(data) => data],
            // Validate status - we'll handle all status codes
            validateStatus: () => true,
        };
        // Make request to Kalshi API
        const response = await axios(axiosConfig);
        logger.debug('Received HTTP response', {
            id: request.id,
            status: response.status,
        });
        // Build response envelope
        const relayResponse = {
            id: request.id,
            status: response.status,
            headers: response.headers,
            body: response.data || '',
        };
        return relayResponse;
    }
    catch (error) {
        if (axios.isAxiosError(error)) {
            if (error.code === 'ECONNABORTED' || error.code === 'ETIMEDOUT') {
                throw new TimeoutError('Request timeout', request.id);
            }
            if (error.code === 'ECONNREFUSED' || error.code === 'ENOTFOUND') {
                throw new NetworkError('Network error', request.id);
            }
            // If we got a response, forward it
            if (error.response) {
                return {
                    id: request.id,
                    status: error.response.status,
                    headers: error.response.headers,
                    body: error.response.data || '',
                };
            }
        }
        logger.error('HTTP relay error', {
            id: request.id,
            error: error instanceof Error ? error.message : String(error),
        });
        throw new NetworkError(error instanceof Error ? error.message : 'Unknown error', request.id);
    }
}
//# sourceMappingURL=httpRelay.js.map