/**
 * Request Validation Middleware
 *
 * Validates HTTP relay requests before processing.
 */
import { ValidationError } from '@galactus/shared';
/**
 * Express middleware to validate HTTP relay requests.
 *
 * Validates:
 * - Content-Type is application/json
 * - Request body structure
 * - Required fields
 * - Request size limits
 *
 * @param config - Server configuration
 * @returns Express middleware
 */
export function createRequestValidator(config) {
    return (req, res, next) => {
        // Check Content-Type
        const contentType = req.get('Content-Type');
        if (!contentType || !contentType.includes('application/json')) {
            return res.status(400).json({
                error: 'Relay error',
                message: 'Content-Type must be application/json',
                code: 'VALIDATION_ERROR',
            });
        }
        // Check request body exists
        if (!req.body || typeof req.body !== 'object') {
            return res.status(400).json({
                error: 'Relay error',
                message: 'Request body is required',
                code: 'VALIDATION_ERROR',
            });
        }
        // Check request size (approximate)
        const bodySize = JSON.stringify(req.body).length;
        if (bodySize > config.requestSizeLimit) {
            return res.status(413).json({
                error: 'Relay error',
                message: 'Request body too large',
                code: 'VALIDATION_ERROR',
            });
        }
        // Validate request structure
        const request = req.body;
        try {
            if (!request.id || typeof request.id !== 'string') {
                throw new ValidationError('Missing or invalid request id');
            }
            if (!request.method || !['GET', 'POST', 'PUT', 'DELETE', 'PATCH'].includes(request.method)) {
                throw new ValidationError('Missing or invalid method');
            }
            if (!request.url || typeof request.url !== 'string') {
                throw new ValidationError('Missing or invalid url');
            }
            if (!request.headers || typeof request.headers !== 'object') {
                throw new ValidationError('Missing or invalid headers');
            }
            if (request.body !== undefined && typeof request.body !== 'string') {
                throw new ValidationError('Body must be a string');
            }
            // All validation passed
            next();
        }
        catch (error) {
            if (error instanceof ValidationError) {
                return res.status(400).json({
                    error: 'Relay error',
                    message: error.message,
                    code: 'VALIDATION_ERROR',
                    requestId: request.id,
                });
            }
            throw error;
        }
    };
}
//# sourceMappingURL=requestValidator.js.map