/**
 * MarketCache — In-memory cache for market data with event emission.
 *
 * Stores normalized market prices and orderbooks, emitting events when data changes
 * so downstream consumers (StreamBroadcaster) can push updates to clients.
 *
 * Supports TTL-based pruning: entries older than `cacheTtlMs` are automatically
 * removed on a periodic sweep (every 60 seconds).
 */
import { EventEmitter } from 'events';
import type { CachedMarket, CachedOrderbook } from '@galactus/shared';
export interface MarketCacheEvents {
    'market:update': (market: CachedMarket) => void;
    'orderbook:update': (orderbook: CachedOrderbook) => void;
}
export declare class MarketCache extends EventEmitter {
    private markets;
    private orderbooks;
    private cacheTtlMs;
    private pruneTimer;
    constructor(cacheTtlMs: number);
    /** Update or insert a market entry. Emits 'market:update' on change. */
    updateMarket(market: CachedMarket): void;
    /** Update or insert an orderbook entry. Emits 'orderbook:update' on change. */
    updateOrderbook(orderbook: CachedOrderbook): void;
    /** Get a full snapshot of all cached markets. */
    getMarketSnapshot(): CachedMarket[];
    /** Get a full snapshot of all cached orderbooks. */
    getOrderbookSnapshot(): CachedOrderbook[];
    /** Get a single market by ticker. */
    getMarket(ticker: string): CachedMarket | undefined;
    /** Get a single orderbook by ticker. */
    getOrderbook(ticker: string): CachedOrderbook | undefined;
    /** Cache stats for health endpoint. */
    getStats(): {
        marketCount: number;
        orderbookCount: number;
    };
    /** Clear all cached data and stop pruning. */
    clear(): void;
    /** Remove entries whose updatedAt is older than cacheTtlMs. */
    private prune;
}
//# sourceMappingURL=MarketCache.d.ts.map