/**
 * SmartRelay — Orchestrator that wires MarketCache + KalshiFetcher + PolymarketFetcher + StreamBroadcaster.
 *
 * Provides start/stop lifecycle, health stats, and API key management.
 */
import { MarketCache } from './MarketCache.js';
import { KalshiFetcher } from './KalshiFetcher.js';
import { PolymarketFetcher } from './PolymarketFetcher.js';
import { StreamBroadcaster } from './StreamBroadcaster.js';
import { ApiKeyStore } from './ApiKeyStore.js';
export class SmartRelay {
    cache;
    kalshi;
    polymarket;
    broadcaster;
    keyStore;
    logger;
    constructor(config, logger) {
        this.logger = logger;
        this.cache = new MarketCache(config.cacheTtlMs);
        this.keyStore = new ApiKeyStore(logger);
        this.keyStore.loadFromEnv();
        this.kalshi = new KalshiFetcher(config, logger, this.cache, this.keyStore);
        this.polymarket = new PolymarketFetcher(config, logger, this.cache);
        this.broadcaster = new StreamBroadcaster(logger, this.cache, {
            getKalshiConnected: () => this.kalshi.isConnected,
            getPolymarketConnected: () => this.polymarket.isConnected,
        });
    }
    async start() {
        this.logger.info('SmartRelay: starting...', {
            apiKeys: this.keyStore.getStats(),
        });
        this.broadcaster.start();
        // Start fetchers concurrently
        await Promise.all([
            this.kalshi
                .start()
                .catch((e) => this.logger.error('Kalshi fetcher failed to start', { error: String(e) })),
            this.polymarket
                .start()
                .catch((e) => this.logger.error('Polymarket fetcher failed to start', { error: String(e) })),
        ]);
        this.logger.info('SmartRelay: started');
    }
    stop() {
        this.logger.info('SmartRelay: stopping...');
        this.kalshi.stop();
        this.polymarket.stop();
        this.broadcaster.stop();
        this.cache.clear();
        this.logger.info('SmartRelay: stopped');
    }
    /** Handle a new client WebSocket connection (from /stream/markets). */
    handleClientConnection(ws) {
        this.broadcaster.handleClientConnection(ws);
    }
    /** Get stats for /health and /stream/status endpoints. */
    getStats() {
        return this.broadcaster.getStatus();
    }
    /** Expose the key store for admin endpoints. */
    getKeyStore() {
        return this.keyStore;
    }
}
export { MarketCache } from './MarketCache.js';
export { KalshiFetcher } from './KalshiFetcher.js';
export { PolymarketFetcher } from './PolymarketFetcher.js';
export { StreamBroadcaster } from './StreamBroadcaster.js';
export { ApiKeyStore } from './ApiKeyStore.js';
//# sourceMappingURL=index.js.map