/**
 * WebSocket Relay
 *
 * Manages WebSocket connections and message forwarding between client and Kalshi API.
 * Supports multiplexing (multiple streams per client connection).
 */
import WebSocket from 'ws';
import { ServerConfig } from './config.js';
import { Logger } from './logger.js';
/**
 * WebSocket relay manager.
 *
 * Handles:
 * - Client connections to relay
 * - Upstream connections to Kalshi
 * - Message forwarding (bidirectional)
 * - Multiplexing (multiple streams per client)
 * - Connection lifecycle management
 * - Reconnection handling
 */
export declare class WebSocketRelay {
    private streams;
    private clientToStreams;
    private clientIdByWs;
    private nextClientId;
    private reconnectConfig;
    private logger;
    private nodeEnv;
    private kalshiAllowedHost;
    constructor(config: ServerConfig, logger: Logger);
    private getClientId;
    private makeInternalKey;
    private validateUpstreamUrl;
    private isRetryableCloseCode;
    private stopPing;
    private startPing;
    private removeStream;
    /**
     * Convert upstream WebSocket data to a string without JSON parsing.
     * Kalshi frames are expected to be UTF-8 text JSON.
     */
    private upstreamDataToString;
    /**
     * Handles a new client WebSocket connection.
     *
     * @param clientWs - Client WebSocket connection
     */
    handleClientConnection(clientWs: WebSocket): void;
    /**
     * Handles a message from a client.
     *
     * @param clientWs - Client WebSocket
     * @param operation - Operation from client
     */
    private handleClientMessage;
    /**
     * Handles a 'connect' operation from client.
     *
     * Opens upstream WebSocket connection to Kalshi.
     *
     * @param clientWs - Client WebSocket
     * @param operation - Connect operation
     */
    private handleConnect;
    /**
     * Handles a 'subscribe' operation from client.
     *
     * @param clientWs - Client WebSocket
     * @param operation - Subscribe operation
     */
    private handleSubscribe;
    /**
     * Handles a 'send' operation from client.
     *
     * @param clientWs - Client WebSocket
     * @param operation - Send operation
     */
    private handleSend;
    /**
     * Handles a 'close' operation from client.
     *
     * @param clientWs - Client WebSocket
     * @param operation - Close operation
     */
    private handleClose;
    /**
     * Handles message from upstream WebSocket.
     *
     * @param streamId - Stream ID
     * @param data - Message data from Kalshi
     */
    private handleUpstreamMessage;
    /**
     * Handles upstream WebSocket error.
     *
     * @param streamId - Stream ID
     * @param error - Error from upstream
     */
    private handleUpstreamError;
    /**
     * Handles upstream WebSocket close.
     *
     * @param streamId - Stream ID
     */
    private handleUpstreamClose;
    /**
     * Attempts to reconnect upstream WebSocket.
     *
     * @param streamId - Stream ID
     */
    private attemptReconnect;
    /**
     * Sends a frame to a client.
     *
     * @param clientWs - Client WebSocket
     * @param frame - Frame to send
     */
    private sendFrame;
    /**
     * Sends an error frame to a client.
     *
     * @param clientWs - Client WebSocket
     * @param streamId - Stream ID
     * @param error - Error message
     */
    private sendErrorFrame;
    /**
     * Cleans up all connections for a client.
     *
     * @param clientWs - Client WebSocket
     */
    cleanupClientConnections(clientWs: WebSocket): void;
    /**
     * Cleans up all WebSocket connections.
     *
     * Called during graceful shutdown to close all upstream connections.
     */
    cleanupAllConnections(): void;
    /**
     * Gets the current number of active WebSocket connections.
     *
     * @returns Number of active streams
     */
    getConnectionCount(): number;
}
//# sourceMappingURL=wsRelay.d.ts.map