/**
 * Relay Server Main Entry Point
 *
 * Sets up Express HTTP server and WebSocket server.
 */

import 'dotenv/config';

import { createServer as createHttpServer, Server } from 'http';
import { createServer as createHttpsServer } from 'https';
import { readFileSync } from 'fs';
import path from 'path';
import { fileURLToPath } from 'url';
import express, { Express, Request, Response } from 'express';
import { WebSocketServer } from 'ws';
import { loadConfig, ServerConfig } from './config.js';
import { createLogger, Logger } from './logger.js';
import { forwardHttpRequest } from './httpRelay.js';
import { WebSocketRelay } from './wsRelay.js';
import { createCorsMiddleware } from './middleware/cors.js';
import { createErrorHandler } from './middleware/errorHandler.js';
import { createRequestValidator } from './middleware/requestValidator.js';
import { HttpRelayRequest } from '@galactus/shared';
import { SmartRelay } from './smart-relay/index.js';
import { createAdminAuth } from './middleware/adminAuth.js';
import { ResponseCache } from './responseCache.js';
import type { IncomingMessage } from 'http';

/**
 * Main server instance.
 */
class RelayServer {
  private app: Express;
  private httpServer: Server;
  private wsServer: WebSocketServer;
  private wsRelay: WebSocketRelay;
  private config: ServerConfig;
  private logger: Logger;
  private smartRelay: SmartRelay | null = null;
  private streamWsServer: WebSocketServer | null = null;
  private responseCache: ResponseCache | null = null;

  constructor() {
    // Load configuration
    this.config = loadConfig();
    this.logger = createLogger(this.config);

    // Create Express app
    this.app = express();

    // Apply middleware
    this.setupMiddleware();

    // Response cache for GET relay requests
    if (this.config.relayCacheEnabled) {
      this.responseCache = new ResponseCache();
      this.logger.info('Response cache enabled');
    }

    // Register routes
    this.setupRoutes();

    // Create HTTP server
    // Use HTTPS if SSL_CERT_PATH and SSL_KEY_PATH are set
    const certPath = process.env.SSL_CERT_PATH;
    const keyPath = process.env.SSL_KEY_PATH;
    if (certPath && keyPath) {
      this.httpServer = createHttpsServer(
        { cert: readFileSync(certPath), key: readFileSync(keyPath) },
        this.app
      );
    } else {
      this.httpServer = createHttpServer(this.app);
    }

    // Create WebSocket server
    this.wsServer = new WebSocketServer({
      server: this.httpServer,
      path: '/relay/ws',
    });

    // Create WebSocket relay
    this.wsRelay = new WebSocketRelay(this.config, this.logger);

    // Set up WebSocket connection handler
    this.wsServer.on('connection', (ws) => {
      this.wsRelay.handleClientConnection(ws);
    });

    // Smart Relay (optional, enabled via SMART_RELAY_ENABLED=true)
    if (this.config.smartRelayEnabled) {
      this.smartRelay = new SmartRelay(this.config, this.logger);

      this.streamWsServer = new WebSocketServer({
        server: this.httpServer,
        path: '/stream/markets',
      });

      this.streamWsServer.on('connection', (ws, req: IncomingMessage) => {
        // Validate stream token if configured
        if (this.config.streamToken) {
          const url = new URL(req.url || '/', `http://${req.headers.host || 'localhost'}`);
          const token = url.searchParams.get('token');
          if (token !== this.config.streamToken) {
            ws.close(4001, 'Invalid or missing stream token');
            return;
          }
        }
        this.smartRelay!.handleClientConnection(ws);
      });
    }

    // Set up graceful shutdown
    this.setupGracefulShutdown();
  }

  /**
   * Sets up Express middleware.
   */
  private setupMiddleware(): void {
    // Body parser (JSON)
    this.app.use(
      express.json({
        limit: `${this.config.requestSizeLimit}b`,
      })
    );

    // CORS
    this.app.use(createCorsMiddleware(this.config));

    // Request logging
    this.app.use((req: Request, res: Response, next) => {
      this.logger.debug('Incoming request', {
        method: req.method,
        path: req.path,
      });
      next();
    });
  }

  /**
   * Sets up Express routes.
   */
  private setupRoutes(): void {
    // Health check endpoint
    this.app.get('/health', (req: Request, res: Response) => {
      const wsConnectionCount = this.wsRelay.getConnectionCount();
      const smartRelayStats = this.smartRelay ? this.smartRelay.getStats() : null;
      res.json({
        status: 'ok',
        service: 'galactus-relay',
        uptime: process.uptime(),
        timestamp: new Date().toISOString(),
        connections: {
          websocket: wsConnectionCount,
        },
        ...(smartRelayStats
          ? {
              smartRelay: {
                enabled: true,
                kalshiConnected: smartRelayStats.kalshiConnected,
                polymarketConnected: smartRelayStats.polymarketConnected,
                marketCount: smartRelayStats.marketCount,
                orderbookCount: smartRelayStats.orderbookCount,
                uptimeMs: smartRelayStats.uptimeMs,
              },
            }
          : { smartRelay: { enabled: false } }),
        responseCache: this.responseCache
          ? { enabled: true, ...this.responseCache.getStats() }
          : { enabled: false },
      });
    });

    // Admin auth middleware (reusable)
    const adminAuth = createAdminAuth(this.config.adminSecret);

    // Smart relay status endpoint (admin-only)
    this.app.get('/stream/status', adminAuth, (req: Request, res: Response) => {
      if (!this.smartRelay) {
        res.status(404).json({ error: 'Smart relay not enabled' });
        return;
      }
      res.json(this.smartRelay.getStats());
    });

    // ── Admin API key management ────────────────────────────────────
    // GET /admin/keys — list all keys (redacted)
    this.app.get('/admin/keys', adminAuth, (req: Request, res: Response) => {
      if (!this.smartRelay) {
        res.status(404).json({ error: 'Smart relay not enabled' });
        return;
      }
      res.json(this.smartRelay.getKeyStore().listKeys());
    });

    // POST /admin/keys — add a new key
    // Body: { accessKey: string, privateKeyPem: string }
    //   OR: { accessKey: string, privateKeyPemBase64: string }
    this.app.post('/admin/keys', adminAuth, (req: Request, res: Response) => {
      if (!this.smartRelay) {
        res.status(404).json({ error: 'Smart relay not enabled' });
        return;
      }

      const { accessKey, privateKeyPem, privateKeyPemBase64 } = req.body as {
        accessKey?: string;
        privateKeyPem?: string;
        privateKeyPemBase64?: string;
      };

      if (!accessKey || typeof accessKey !== 'string') {
        res.status(400).json({ error: 'accessKey (string) is required' });
        return;
      }

      let pem: string;
      if (privateKeyPem && typeof privateKeyPem === 'string') {
        pem = privateKeyPem;
      } else if (privateKeyPemBase64 && typeof privateKeyPemBase64 === 'string') {
        try {
          pem = Buffer.from(privateKeyPemBase64, 'base64').toString('utf-8');
        } catch {
          res.status(400).json({ error: 'Invalid base64 in privateKeyPemBase64' });
          return;
        }
      } else {
        res.status(400).json({ error: 'privateKeyPem or privateKeyPemBase64 is required' });
        return;
      }

      const id = this.smartRelay.getKeyStore().addKey(accessKey.trim(), pem.trim());
      res.status(201).json({ id, message: 'Key added' });
    });

    // DELETE /admin/keys/:id — remove a key
    this.app.delete('/admin/keys/:id', adminAuth, (req: Request, res: Response) => {
      if (!this.smartRelay) {
        res.status(404).json({ error: 'Smart relay not enabled' });
        return;
      }

      const deleted = this.smartRelay.getKeyStore().removeKey(req.params.id);
      if (deleted) {
        res.json({ message: 'Key removed' });
      } else {
        res.status(404).json({ error: 'Key not found' });
      }
    });

    // POST /admin/keys/:id/enable — re-enable a disabled key
    this.app.post('/admin/keys/:id/enable', adminAuth, (req: Request, res: Response) => {
      if (!this.smartRelay) {
        res.status(404).json({ error: 'Smart relay not enabled' });
        return;
      }

      const ok = this.smartRelay.getKeyStore().enableKey(req.params.id);
      if (ok) {
        res.json({ message: 'Key re-enabled' });
      } else {
        res.status(404).json({ error: 'Key not found' });
      }
    });

    // HTTP relay endpoint
    this.app.post(
      '/relay/http',
      createRequestValidator(this.config),
      async (req: Request, res: Response, next) => {
        try {
          const request = req.body as HttpRelayRequest;

          // Check cache for GET requests
          if (request.method === 'GET' && this.responseCache) {
            const cacheKey = this.responseCache.buildKey(request.url);
            if (cacheKey) {
              const cached = this.responseCache.get(cacheKey);
              if (cached) {
                res.json({ ...cached, id: request.id });
                return;
              }
            }
          }

          const response = await forwardHttpRequest(request, this.config, this.logger);

          // Cache successful GET responses for cacheable URLs
          if (request.method === 'GET' && response.status === 200 && this.responseCache) {
            this.responseCache.tryStore(request.url, response);
          }

          res.json(response);
        } catch (error) {
          next(error);
        }
      }
    );

    // Error handler (must be last)
    this.app.use(createErrorHandler(this.logger));
  }

  /**
   * Sets up graceful shutdown handlers.
   */
  private setupGracefulShutdown(): void {
    const shutdown = async (signal: string) => {
      this.logger.info(`Received ${signal}, shutting down gracefully...`);

      // Stop smart relay
      if (this.smartRelay) {
        this.smartRelay.stop();
      }

      // Clean up response cache
      if (this.responseCache) {
        this.responseCache.destroy();
      }

      // Clean up all WebSocket connections
      this.wsRelay.cleanupAllConnections();

      // Stop accepting new connections
      this.httpServer.close(() => {
        this.logger.info('HTTP server closed');
      });

      // Close WebSocket servers
      this.wsServer.close(() => {
        this.logger.info('WebSocket server closed');
      });

      if (this.streamWsServer) {
        this.streamWsServer.close(() => {
          this.logger.info('Stream WebSocket server closed');
        });
      }

      // Give connections time to close
      setTimeout(() => {
        this.logger.info('Shutdown complete');
        process.exit(0);
      }, 5000);
    };

    process.on('SIGTERM', () => shutdown('SIGTERM'));
    process.on('SIGINT', () => shutdown('SIGINT'));
  }

  /**
   * Starts the server.
   */
  async start(): Promise<void> {
    return new Promise((resolve, reject) => {
      try {
        this.httpServer.listen(this.config.port, async () => {
          this.logger.info('Relay server started', {
            port: this.config.port,
            environment: this.config.nodeEnv,
            httpEndpoint: `http://localhost:${this.config.port}/relay/http`,
            wsEndpoint: `ws://localhost:${this.config.port}/relay/ws`,
            smartRelay: this.config.smartRelayEnabled,
            ...(this.config.smartRelayEnabled
              ? { streamEndpoint: `ws://localhost:${this.config.port}/stream/markets` }
              : {}),
          });

          // Start smart relay in background (non-blocking)
          if (this.smartRelay) {
            this.smartRelay.start().catch((err) => {
              this.logger.error('Smart relay failed to start', { error: String(err) });
            });
          }

          resolve();
        });

        this.httpServer.on('error', (error: Error) => {
          this.logger.error('Server error', {
            error: error.message,
          });
          reject(error);
        });
      } catch (error) {
        reject(error);
      }
    });
  }
}

/**
 * Main entry point.
 */
async function main(): Promise<void> {
  // Safety net: prevent unhandled rejections from crashing the process
  process.on('unhandledRejection', (reason) => {
    console.error('Unhandled rejection (process kept alive):', reason);
  });

  try {
    const server = new RelayServer();
    await server.start();
  } catch (error) {
    console.error('Failed to start server:', error);
    process.exit(1);
  }
}

// Start server if this file is run directly (ESM equivalent of require.main === module).
// Do not remove the NODE_APP_INSTANCE check; under PM2 argv can differ and main() would not run. See docs/devops.md.
const __filename = fileURLToPath(import.meta.url);
const scriptPath = process.argv[1];
const isEntryScript =
  scriptPath &&
  (path.resolve(scriptPath) === __filename ||
    path.resolve(process.cwd(), scriptPath) === __filename);
if (isEntryScript || process.env.NODE_APP_INSTANCE !== undefined) {
  main();
}

export { RelayServer };
