/**
 * Structured Logging
 *
 * Provides structured logging with sanitization of sensitive data.
 */

import { ServerConfig } from './config.js';

/**
 * Logger interface.
 */
export interface Logger {
  error(message: string, meta?: Record<string, unknown>): void;
  warn(message: string, meta?: Record<string, unknown>): void;
  info(message: string, meta?: Record<string, unknown>): void;
  debug(message: string, meta?: Record<string, unknown>): void;
}

/**
 * Sensitive fields that should be redacted from logs.
 */
const SENSITIVE_FIELDS = [
  'KALSHI-ACCESS-KEY',
  'KALSHI-ACCESS-SIGNATURE',
  'kalshi-access-key',
  'kalshi-access-signature',
  'authorization',
  'Authorization',
  'password',
  'token',
  'secret',
];

/**
 * Sanitizes sensitive data from log metadata.
 *
 * Removes or redacts credentials, signatures, and other sensitive fields.
 *
 * @param meta - Log metadata
 * @returns Sanitized metadata
 */
function sanitizeLogMeta(meta: Record<string, unknown>): Record<string, unknown> {
  const sanitized = { ...meta };

  for (const key in sanitized) {
    const lowerKey = key.toLowerCase();
    if (SENSITIVE_FIELDS.some((field) => lowerKey.includes(field.toLowerCase()))) {
      sanitized[key] = '[REDACTED]';
    }
  }

  return sanitized;
}

/**
 * Creates a logger instance.
 *
 * Uses console for now. Can be replaced with pino, winston, etc.
 *
 * @param config - Server configuration
 * @returns Logger instance
 */
export function createLogger(config: ServerConfig): Logger {
  const isDevelopment = config.nodeEnv === 'development';
  // Default to info to avoid noisy dev logs; override with LOG_LEVEL=debug when needed.
  const logLevel = process.env.LOG_LEVEL || 'info';

  const shouldLog = (level: string): boolean => {
    const levels = ['error', 'warn', 'info', 'debug'];
    const currentLevelIndex = levels.indexOf(logLevel);
    const messageLevelIndex = levels.indexOf(level);
    return messageLevelIndex <= currentLevelIndex;
  };

  const formatMessage = (
    level: string,
    message: string,
    meta?: Record<string, unknown>
  ): string => {
    if (isDevelopment) {
      // Pretty format for development
      const metaStr = meta ? ` ${JSON.stringify(sanitizeLogMeta(meta), null, 2)}` : '';
      return `[${new Date().toISOString()}] [${level.toUpperCase()}] ${message}${metaStr}`;
    } else {
      // JSON format for production
      return JSON.stringify({
        timestamp: new Date().toISOString(),
        level,
        message,
        ...(meta ? sanitizeLogMeta(meta) : {}),
      });
    }
  };

  return {
    error: (message: string, meta?: Record<string, unknown>) => {
      if (shouldLog('error')) {
        console.error(formatMessage('error', message, meta));
      }
    },
    warn: (message: string, meta?: Record<string, unknown>) => {
      if (shouldLog('warn')) {
        console.warn(formatMessage('warn', message, meta));
      }
    },
    info: (message: string, meta?: Record<string, unknown>) => {
      if (shouldLog('info')) {
        console.info(formatMessage('info', message, meta));
      }
    },
    debug: (message: string, meta?: Record<string, unknown>) => {
      if (shouldLog('debug')) {
        console.debug(formatMessage('debug', message, meta));
      }
    },
  };
}
