/**
 * Server-Specific Types
 *
 * TypeScript types specific to the relay server implementation.
 */

import { WebSocket } from 'ws';

/**
 * Connection state for WebSocket streams.
 */
export type ConnectionState = 'connecting' | 'connected' | 'disconnected' | 'reconnecting';

/**
 * Stream metadata for tracking WebSocket connections.
 */
export interface StreamMetadata {
  /** Stream ID */
  streamId: string;
  /**
   * Internal stream key, unique across all relay clients.
   * Used to avoid collisions when different clients reuse the same streamId.
   */
  internalKey: string;
  /** Internal client connection id (for diagnostics / namespacing). */
  clientId: string;
  /** Client WebSocket connection */
  clientWs: WebSocket;
  /** Upstream WebSocket connection to Kalshi (null if not connected) */
  upstreamWs: WebSocket | null;
  /** Current connection state */
  state: ConnectionState;
  /** Number of reconnection attempts */
  reconnectAttempts: number;
  /** Timestamp when stream was created */
  createdAt: Date;
  /** Timestamp of last activity */
  lastActivity: Date;
  /** Kalshi WebSocket URL */
  url?: string;
  /** Auth headers for reconnection */
  headers?: Record<string, string>;

  /**
   * Keepalive ping timer for upstream.
   * Kalshi expects periodic ping/pong to keep the connection alive.
   */
  pingInterval?: NodeJS.Timeout | null;
}

/**
 * Reconnection configuration.
 */
export interface ReconnectConfig {
  /** Initial delay in milliseconds */
  delayMs: number;
  /** Maximum number of reconnection attempts */
  maxAttempts: number;
  /** Backoff multiplier (e.g., 2 for exponential backoff) */
  backoffMultiplier: number;
}
