/**
 * WebSocket Relay
 *
 * Manages WebSocket connections and message forwarding between client and Kalshi API.
 * Supports multiplexing (multiple streams per client connection).
 */

import WebSocket from 'ws';
import {
  WsRelayOperation,
  WsRelayConnect,
  WsRelaySubscribe,
  WsRelaySend,
  WsRelayClose,
  WsRelayFrame,
} from '@galactus/shared';
import { ServerConfig } from './config.js';
import { Logger } from './logger.js';
import { StreamMetadata, ReconnectConfig } from './types.js';

/**
 * WebSocket relay manager.
 *
 * Handles:
 * - Client connections to relay
 * - Upstream connections to Kalshi
 * - Message forwarding (bidirectional)
 * - Multiplexing (multiple streams per client)
 * - Connection lifecycle management
 * - Reconnection handling
 */
export class WebSocketRelay {
  private streams: Map<string, StreamMetadata> = new Map();
  private clientToStreams: Map<WebSocket, Set<string>> = new Map();
  private clientIdByWs: WeakMap<WebSocket, string> = new WeakMap();
  private nextClientId = 1;
  private reconnectConfig: ReconnectConfig;
  private logger: Logger;
  private nodeEnv: ServerConfig['nodeEnv'];
  private kalshiAllowedHost: string;

  constructor(config: ServerConfig, logger: Logger) {
    this.logger = logger;
    this.nodeEnv = config.nodeEnv;
    this.kalshiAllowedHost = new URL(config.kalshiBaseUrl).hostname;
    this.reconnectConfig = {
      delayMs: config.wsReconnectDelayMs,
      maxAttempts: config.wsMaxReconnectAttempts,
      backoffMultiplier: 2,
    };
  }

  private getClientId(clientWs: WebSocket): string {
    const existing = this.clientIdByWs.get(clientWs);
    if (existing) return existing;
    const id = `c${this.nextClientId++}`;
    this.clientIdByWs.set(clientWs, id);
    return id;
  }

  private makeInternalKey(clientWs: WebSocket, streamId: string): string {
    return `${this.getClientId(clientWs)}:${streamId}`;
  }

  private validateUpstreamUrl(upstreamUrl: string): void {
    let parsed: URL;
    try {
      parsed = new URL(upstreamUrl);
    } catch {
      throw new Error('Invalid upstream WebSocket URL');
    }

    if (parsed.protocol !== 'ws:' && parsed.protocol !== 'wss:') {
      throw new Error('Upstream WebSocket URL must use ws: or wss:');
    }

    // In production, restrict upstream WS to Kalshi host
    if (this.nodeEnv === 'production' && parsed.hostname !== this.kalshiAllowedHost) {
      throw new Error(`Upstream WebSocket host must be ${this.kalshiAllowedHost} in production`);
    }
  }

  private isRetryableCloseCode(code: number): boolean {
    // Normal closure / going away: no retry
    if (code === 1000 || code === 1001) return false;
    // Protocol / unsupported data / policy violation: no retry (usually indicates bad client payload/auth)
    if (code === 1002 || code === 1003 || code === 1008) return false;
    // App-defined close codes (common for auth failures): no retry
    if (code >= 4000 && code < 5000) return false;
    // Otherwise: retry (includes 1006 abnormal, 1011 server error, etc.)
    return true;
  }

  private stopPing(metadata: StreamMetadata): void {
    if (metadata.pingInterval) {
      clearInterval(metadata.pingInterval);
      metadata.pingInterval = null;
    }
  }

  private startPing(metadata: StreamMetadata): void {
    this.stopPing(metadata);

    // Kalshi docs recommend ping/pong every ~10s
    metadata.pingInterval = setInterval(() => {
      const ws = metadata.upstreamWs;
      if (!ws) return;
      if (ws.readyState !== WebSocket.OPEN) return;
      try {
        ws.ping();
      } catch (error) {
        this.logger.debug('Upstream ping failed', {
          clientId: metadata.clientId,
          streamId: metadata.streamId,
          error: error instanceof Error ? error.message : String(error),
        });
      }
    }, 10_000);
  }

  private removeStream(internalKey: string): void {
    const metadata = this.streams.get(internalKey);
    if (!metadata) return;

    this.stopPing(metadata);

    const set = this.clientToStreams.get(metadata.clientWs);
    if (set) set.delete(internalKey);
    this.streams.delete(internalKey);
  }

  /**
   * Convert upstream WebSocket data to a string without JSON parsing.
   * Kalshi frames are expected to be UTF-8 text JSON.
   */
  private upstreamDataToString(data: WebSocket.Data): string {
    if (typeof data === 'string') return data;
    // ws (Node) commonly delivers Buffer for text frames
    if (Buffer.isBuffer(data)) return data.toString('utf8');
    // ws can deliver ArrayBuffer
    if (data instanceof ArrayBuffer) return Buffer.from(data).toString('utf8');
    // ws can deliver array of Buffer chunks
    if (Array.isArray(data)) return Buffer.concat(data).toString('utf8');
    // Fallback
    return String(data);
  }

  /**
   * Handles a new client WebSocket connection.
   *
   * @param clientWs - Client WebSocket connection
   */
  handleClientConnection(clientWs: WebSocket): void {
    const clientId = this.getClientId(clientWs);
    this.logger.info('New client WebSocket connection', { clientId });

    // Track streams for this client
    this.clientToStreams.set(clientWs, new Set());

    // Set up message handler
    clientWs.on('message', (data: WebSocket.Data) => {
      try {
        const message = JSON.parse(data.toString()) as WsRelayOperation;
        this.handleClientMessage(clientWs, message);
      } catch (error) {
        this.logger.error('Failed to parse client message', {
          error: error instanceof Error ? error.message : String(error),
        });
        this.sendErrorFrame(clientWs, 'unknown', 'Invalid message format');
      }
    });

    // Set up error handler
    clientWs.on('error', (error: Error) => {
      this.logger.error('Client WebSocket error', {
        error: error.message,
      });
    });

    // Set up close handler
    clientWs.on('close', () => {
      this.logger.info('Client WebSocket closed');
      this.cleanupClientConnections(clientWs);
    });
  }

  /**
   * Handles a message from a client.
   *
   * @param clientWs - Client WebSocket
   * @param operation - Operation from client
   */
  private handleClientMessage(clientWs: WebSocket, operation: WsRelayOperation): void {
    switch (operation.op) {
      case 'connect':
        this.handleConnect(clientWs, operation);
        break;
      case 'subscribe':
        this.handleSubscribe(clientWs, operation);
        break;
      case 'send':
        this.handleSend(clientWs, operation);
        break;
      case 'close':
        this.handleClose(clientWs, operation);
        break;
      default:
        this.sendErrorFrame(
          clientWs,
          'unknown',
          `Unknown operation: ${(operation as { op: string }).op}`
        );
    }
  }

  /**
   * Handles a 'connect' operation from client.
   *
   * Opens upstream WebSocket connection to Kalshi.
   *
   * @param clientWs - Client WebSocket
   * @param operation - Connect operation
   */
  private async handleConnect(clientWs: WebSocket, operation: WsRelayConnect): Promise<void> {
    const { id: streamId, url, headers } = operation;
    const clientId = this.getClientId(clientWs);
    const internalKey = this.makeInternalKey(clientWs, streamId);

    this.logger.debug('Handling connect operation', { clientId, streamId, url });

    // Validate operation
    if (!streamId || !url || !headers) {
      this.sendErrorFrame(clientWs, streamId, 'Missing required fields');
      return;
    }

    // Check if stream already exists
    if (this.streams.has(internalKey)) {
      this.sendErrorFrame(clientWs, streamId, 'Stream already exists');
      return;
    }

    try {
      this.validateUpstreamUrl(url);

      // Create upstream WebSocket connection
      const upstreamWs = new WebSocket(url, {
        headers: headers as Record<string, string>,
      });

      // Create stream metadata
      const metadata: StreamMetadata = {
        streamId,
        internalKey,
        clientId,
        clientWs,
        upstreamWs,
        state: 'connecting',
        reconnectAttempts: 0,
        createdAt: new Date(),
        lastActivity: new Date(),
        url,
        headers,
        pingInterval: null,
      };

      // Set up upstream handlers
      upstreamWs.on('open', () => {
        this.logger.info('Upstream WebSocket connected', { clientId, streamId });
        metadata.state = 'connected';
        metadata.upstreamWs = upstreamWs;
        this.startPing(metadata);
        this.sendFrame(clientWs, {
          id: streamId,
          type: 'message',
          data: { connected: true },
        });
      });

      upstreamWs.on('message', (data: WebSocket.Data) => {
        this.handleUpstreamMessage(internalKey, data);
      });

      upstreamWs.on('error', (error: Error) => {
        this.logger.error('Upstream WebSocket error', {
          clientId,
          streamId,
          error: error.message,
        });
        this.handleUpstreamError(internalKey, error);
      });

      upstreamWs.on('close', (code: number, reason: Buffer) => {
        const reasonText = reason?.toString?.('utf8') || '';
        this.logger.info('Upstream WebSocket closed', {
          clientId,
          streamId,
          code,
          reason: reasonText,
        });
        this.handleUpstreamClose(internalKey, code, reasonText);
      });

      // Store stream
      this.streams.set(internalKey, metadata);
      const clientStreams = this.clientToStreams.get(clientWs);
      if (clientStreams) {
        clientStreams.add(internalKey);
      }
    } catch (error) {
      this.logger.error('Failed to create upstream connection', {
        clientId,
        streamId,
        error: error instanceof Error ? error.message : String(error),
      });
      this.sendErrorFrame(
        clientWs,
        streamId,
        `Connection failed: ${error instanceof Error ? error.message : String(error)}`
      );
    }
  }

  /**
   * Handles a 'subscribe' operation from client.
   *
   * @param clientWs - Client WebSocket
   * @param operation - Subscribe operation
   */
  private handleSubscribe(clientWs: WebSocket, operation: WsRelaySubscribe): void {
    const { id: streamId, payload } = operation;
    const internalKey = this.makeInternalKey(clientWs, streamId);
    const metadata = this.streams.get(internalKey);

    if (!metadata || !metadata.upstreamWs) {
      this.sendErrorFrame(clientWs, streamId, 'Stream not connected');
      return;
    }

    if (metadata.upstreamWs.readyState !== WebSocket.OPEN) {
      this.sendErrorFrame(clientWs, streamId, 'Upstream connection not open');
      return;
    }

    // Forward payload to upstream (as-is, no parsing)
    try {
      // Important: if payload is already a string, do NOT JSON.stringify it (that adds quotes).
      const message =
        payload === undefined
          ? ''
          : typeof payload === 'string'
            ? payload
            : JSON.stringify(payload);
      metadata.upstreamWs.send(message);
      metadata.lastActivity = new Date();
    } catch (error) {
      this.logger.error('Failed to send subscribe payload', {
        streamId,
        error: error instanceof Error ? error.message : String(error),
      });
      this.sendErrorFrame(clientWs, streamId, 'Failed to send message');
    }
  }

  /**
   * Handles a 'send' operation from client.
   *
   * @param clientWs - Client WebSocket
   * @param operation - Send operation
   */
  private handleSend(clientWs: WebSocket, operation: WsRelaySend): void {
    const { id: streamId, payload } = operation;
    const internalKey = this.makeInternalKey(clientWs, streamId);
    const metadata = this.streams.get(internalKey);

    if (!metadata || !metadata.upstreamWs) {
      this.sendErrorFrame(clientWs, streamId, 'Stream not connected');
      return;
    }

    if (metadata.upstreamWs.readyState !== WebSocket.OPEN) {
      this.sendErrorFrame(clientWs, streamId, 'Upstream connection not open');
      return;
    }

    // Forward payload to upstream (as-is)
    try {
      // Important: if payload is already a string, do NOT JSON.stringify it (that adds quotes).
      const message = typeof payload === 'string' ? payload : JSON.stringify(payload);
      metadata.upstreamWs.send(message);
      metadata.lastActivity = new Date();
    } catch (error) {
      this.logger.error('Failed to send message', {
        streamId,
        error: error instanceof Error ? error.message : String(error),
      });
      this.sendErrorFrame(clientWs, streamId, 'Failed to send message');
    }
  }

  /**
   * Handles a 'close' operation from client.
   *
   * @param clientWs - Client WebSocket
   * @param operation - Close operation
   */
  private handleClose(clientWs: WebSocket, operation: WsRelayClose): void {
    const { id: streamId } = operation;
    const internalKey = this.makeInternalKey(clientWs, streamId);
    const metadata = this.streams.get(internalKey);

    if (!metadata) {
      this.sendErrorFrame(clientWs, streamId, 'Stream not found');
      return;
    }

    // Close upstream connection
    if (metadata.upstreamWs) {
      metadata.upstreamWs.close();
    }

    // Remove from maps
    this.streams.delete(internalKey);
    const clientStreams = this.clientToStreams.get(clientWs);
    if (clientStreams) {
      clientStreams.delete(internalKey);
    }

    // Send confirmation
    this.sendFrame(clientWs, {
      id: streamId,
      type: 'close',
    });
  }

  /**
   * Handles message from upstream WebSocket.
   *
   * @param streamId - Stream ID
   * @param data - Message data from Kalshi
   */
  private handleUpstreamMessage(internalKey: string, data: WebSocket.Data): void {
    const metadata = this.streams.get(internalKey);
    if (!metadata) {
      return;
    }

    metadata.lastActivity = new Date();

    // Transport-only: forward as text without JSON parsing.
    const raw = this.upstreamDataToString(data);

    // Forward to client
    this.sendFrame(metadata.clientWs, {
      id: metadata.streamId,
      type: 'message',
      data: raw,
    });
  }

  /**
   * Handles upstream WebSocket error.
   *
   * @param streamId - Stream ID
   * @param error - Error from upstream
   */
  private handleUpstreamError(internalKey: string, error: Error): void {
    const metadata = this.streams.get(internalKey);
    if (!metadata) {
      return;
    }

    metadata.state = 'disconnected';
    this.stopPing(metadata);

    // Send error frame to client
    this.sendFrame(metadata.clientWs, {
      id: metadata.streamId,
      type: 'error',
      error: error.message,
    });

    // Attempt reconnection if configured
    if (metadata.reconnectAttempts < this.reconnectConfig.maxAttempts) {
      this.attemptReconnect(internalKey);
    }
  }

  /**
   * Handles upstream WebSocket close.
   *
   * @param streamId - Stream ID
   */
  private handleUpstreamClose(internalKey: string, code: number, reason: string): void {
    const metadata = this.streams.get(internalKey);
    if (!metadata) {
      return;
    }

    metadata.state = 'disconnected';
    metadata.upstreamWs = null;
    this.stopPing(metadata);

    // Send close frame to client
    this.sendFrame(metadata.clientWs, {
      id: metadata.streamId,
      type: 'close',
    });

    // Attempt reconnection if configured
    const shouldRetry = this.isRetryableCloseCode(code);
    if (!shouldRetry) {
      // Don't flap on auth/policy/protocol closes; cleanly remove stream mapping.
      this.logger.info('Not retrying upstream close (non-retryable)', {
        clientId: metadata.clientId,
        streamId: metadata.streamId,
        code,
        reason,
      });
      this.removeStream(internalKey);
      return;
    }

    if (metadata.reconnectAttempts < this.reconnectConfig.maxAttempts) {
      this.attemptReconnect(internalKey);
    }
  }

  /**
   * Attempts to reconnect upstream WebSocket.
   *
   * @param streamId - Stream ID
   */
  private async attemptReconnect(internalKey: string): Promise<void> {
    const metadata = this.streams.get(internalKey);
    if (!metadata || !metadata.url || !metadata.headers) {
      return;
    }

    metadata.reconnectAttempts++;
    metadata.state = 'reconnecting';

    // Calculate backoff delay
    const delay = Math.min(
      this.reconnectConfig.delayMs *
        Math.pow(this.reconnectConfig.backoffMultiplier, metadata.reconnectAttempts - 1),
      30000 // Max 30 seconds
    );

    this.logger.info('Attempting reconnection', {
      clientId: metadata.clientId,
      streamId: metadata.streamId,
      attempt: metadata.reconnectAttempts,
      delay,
    });

    // Wait for delay
    await new Promise((resolve) => setTimeout(resolve, delay));

    // Attempt reconnection
    try {
      this.validateUpstreamUrl(metadata.url);
      const upstreamWs = new WebSocket(metadata.url, {
        headers: metadata.headers,
      });

      upstreamWs.on('open', () => {
        this.logger.info('Reconnection successful', {
          clientId: metadata.clientId,
          streamId: metadata.streamId,
        });
        metadata.state = 'connected';
        metadata.upstreamWs = upstreamWs;
        metadata.reconnectAttempts = 0;
        this.startPing(metadata);
        this.sendFrame(metadata.clientWs, {
          id: metadata.streamId,
          type: 'message',
          data: { reconnected: true },
        });
      });

      upstreamWs.on('message', (data: WebSocket.Data) => {
        this.handleUpstreamMessage(internalKey, data);
      });

      upstreamWs.on('error', (error: Error) => {
        this.handleUpstreamError(internalKey, error);
      });

      upstreamWs.on('close', (code: number, reason: Buffer) => {
        const reasonText = reason?.toString?.('utf8') || '';
        this.handleUpstreamClose(internalKey, code, reasonText);
      });
    } catch (error) {
      this.logger.error('Reconnection failed', {
        clientId: metadata.clientId,
        streamId: metadata.streamId,
        error: error instanceof Error ? error.message : String(error),
      });
      if (metadata.reconnectAttempts < this.reconnectConfig.maxAttempts) {
        // Retry again
        this.attemptReconnect(internalKey);
      } else {
        this.sendErrorFrame(
          metadata.clientWs,
          metadata.streamId,
          'Max reconnection attempts reached'
        );
      }
    }
  }

  /**
   * Sends a frame to a client.
   *
   * @param clientWs - Client WebSocket
   * @param frame - Frame to send
   */
  private sendFrame(clientWs: WebSocket, frame: WsRelayFrame): void {
    if (clientWs.readyState === WebSocket.OPEN) {
      try {
        clientWs.send(JSON.stringify(frame));
      } catch (error) {
        this.logger.error('Failed to send frame to client', {
          error: error instanceof Error ? error.message : String(error),
        });
      }
    }
  }

  /**
   * Sends an error frame to a client.
   *
   * @param clientWs - Client WebSocket
   * @param streamId - Stream ID
   * @param error - Error message
   */
  private sendErrorFrame(clientWs: WebSocket, streamId: string, error: string): void {
    this.sendFrame(clientWs, {
      id: streamId,
      type: 'error',
      error,
    });
  }

  /**
   * Cleans up all connections for a client.
   *
   * @param clientWs - Client WebSocket
   */
  cleanupClientConnections(clientWs: WebSocket): void {
    const internalKeys = this.clientToStreams.get(clientWs);
    if (!internalKeys) {
      return;
    }

    for (const internalKey of internalKeys) {
      const metadata = this.streams.get(internalKey);
      if (metadata?.upstreamWs) {
        metadata.upstreamWs.close();
      }
      if (metadata) {
        this.stopPing(metadata);
      }
      this.streams.delete(internalKey);
    }

    this.clientToStreams.delete(clientWs);
  }

  /**
   * Cleans up all WebSocket connections.
   *
   * Called during graceful shutdown to close all upstream connections.
   */
  cleanupAllConnections(): void {
    for (const [internalKey, metadata] of this.streams.entries()) {
      if (metadata.upstreamWs) {
        metadata.upstreamWs.close();
      }
      this.stopPing(metadata);
      this.streams.delete(internalKey);
    }
    this.clientToStreams.clear();
  }

  /**
   * Gets the current number of active WebSocket connections.
   *
   * @returns Number of active streams
   */
  getConnectionCount(): number {
    return this.streams.size;
  }
}
