/**
 * WebSocket Relay Integration Tests
 *
 * Validates bidirectional forwarding behavior against a mock upstream WS server.
 */

import { createServer, type Server } from 'http';
import WebSocket, { WebSocketServer } from 'ws';
import { createMockKalshiServer } from '../fixtures/mockKalshiServer';
import { WebSocketRelay } from '../../src/wsRelay';
import type { ServerConfig } from '../../src/config';
import type { Logger } from '../../src/logger';

function waitForMessage(ws: WebSocket): Promise<string> {
  return new Promise((resolve, reject) => {
    const onMessage = (data: WebSocket.Data) => {
      cleanup();
      resolve(typeof data === 'string' ? data : data.toString());
    };
    const onError = (err: Error) => {
      cleanup();
      reject(err);
    };
    const onClose = () => {
      cleanup();
      reject(new Error('WebSocket closed before message received'));
    };
    const cleanup = () => {
      ws.off('message', onMessage);
      ws.off('error', onError);
      ws.off('close', onClose);
    };
    ws.on('message', onMessage);
    ws.on('error', onError);
    ws.on('close', onClose);
  });
}

function parseRelayFrame(raw: string): {
  id: string;
  type: string;
  data?: unknown;
  error?: string;
} {
  return JSON.parse(raw) as { id: string; type: string; data?: unknown; error?: string };
}

async function listen(server: Server, port: number): Promise<void> {
  await new Promise<void>((resolve, reject) => {
    server.listen(port, '127.0.0.1', () => resolve());
    server.on('error', reject);
  });
}

async function getFreePort(): Promise<number> {
  const s = createServer();
  await new Promise<void>((resolve) => s.listen(0, '127.0.0.1', resolve));
  const addr = s.address();
  if (!addr || typeof addr === 'string') throw new Error('Failed to allocate a free port');
  const port = addr.port;
  await new Promise<void>((resolve) => s.close(() => resolve()));
  return port;
}

const noopLogger: Logger = {
  error: () => undefined,
  warn: () => undefined,
  info: () => undefined,
  debug: () => undefined,
};

describe('WebSocket Relay Integration', () => {
  let mock: ReturnType<typeof createMockKalshiServer>;
  let relayHttpServer: Server;
  let relayWsServer: WebSocketServer;
  let relay: WebSocketRelay;
  let relayPort: number;
  let kalshiPort: number;

  beforeEach(async () => {
    kalshiPort = await getFreePort();
    mock = createMockKalshiServer(kalshiPort);
    await listen(mock.server, kalshiPort);

    relayPort = await getFreePort();
    relayHttpServer = createServer();

    const config: ServerConfig = {
      port: relayPort,
      nodeEnv: 'development',
      corsOrigin: true,
      httpTimeoutMs: 30_000,
      wsReconnectDelayMs: 100,
      wsMaxReconnectAttempts: 0,
      requestSizeLimit: 10 * 1024 * 1024,
      kalshiBaseUrl: `http://127.0.0.1:${kalshiPort}`,
    };

    relay = new WebSocketRelay(config, noopLogger);
    relayWsServer = new WebSocketServer({ server: relayHttpServer, path: '/relay/ws' });
    relayWsServer.on('connection', (ws) => relay.handleClientConnection(ws));

    await listen(relayHttpServer, relayPort);
  });

  afterEach(async () => {
    if (relayWsServer) {
      await new Promise<void>((resolve) => relayWsServer.close(() => resolve()));
    }
    if (relayHttpServer) {
      await new Promise<void>((resolve) => relayHttpServer.close(() => resolve()));
    }
    if (mock) {
      await mock.close();
    }
  });

  it('forwards string payloads without JSON stringification', async () => {
    const client = new WebSocket(`ws://127.0.0.1:${relayPort}/relay/ws`);
    await new Promise<void>((resolve) => client.on('open', () => resolve()));

    const upstreamUrl = `ws://127.0.0.1:${kalshiPort}/trade-api/ws/v2`;

    client.send(
      JSON.stringify({
        op: 'connect',
        id: 's1',
        url: upstreamUrl,
        headers: {},
      })
    );

    // Wait for relay's connected ack
    const ackRaw = await waitForMessage(client);
    const ack = parseRelayFrame(ackRaw);
    expect(ack.id).toBe('s1');
    expect(ack.type).toBe('message');
    expect(ack.data).toEqual({ connected: true });

    // Send a string payload; upstream echoes it verbatim.
    client.send(
      JSON.stringify({
        op: 'subscribe',
        id: 's1',
        payload: 'hello',
      })
    );

    const echoedRaw = await waitForMessage(client);
    const echoed = parseRelayFrame(echoedRaw);
    expect(echoed.id).toBe('s1');
    expect(echoed.type).toBe('message');
    expect(echoed.data).toBe('hello'); // NOT "\"hello\""

    client.close();
  });

  it('does not JSON-parse upstream frames (keeps JSON text as string)', async () => {
    const client = new WebSocket(`ws://127.0.0.1:${relayPort}/relay/ws`);
    await new Promise<void>((resolve) => client.on('open', () => resolve()));

    const upstreamUrl = `ws://127.0.0.1:${kalshiPort}/trade-api/ws/v2`;

    client.send(
      JSON.stringify({
        op: 'connect',
        id: 's1',
        url: upstreamUrl,
        headers: {},
      })
    );

    // Drain ack
    await waitForMessage(client);

    const jsonText = '{"foo":1}';
    client.send(
      JSON.stringify({
        op: 'send',
        id: 's1',
        payload: jsonText,
      })
    );

    const echoedRaw = await waitForMessage(client);
    const echoed = parseRelayFrame(echoedRaw);
    expect(echoed.id).toBe('s1');
    expect(echoed.type).toBe('message');
    expect(typeof echoed.data).toBe('string');
    expect(echoed.data).toBe(jsonText);

    client.close();
  });

  it('allows different clients to reuse the same stream id without collisions', async () => {
    const upstreamUrl = `ws://127.0.0.1:${kalshiPort}/trade-api/ws/v2`;

    const a = new WebSocket(`ws://127.0.0.1:${relayPort}/relay/ws`);
    const b = new WebSocket(`ws://127.0.0.1:${relayPort}/relay/ws`);
    await Promise.all([
      new Promise<void>((resolve) => a.on('open', () => resolve())),
      new Promise<void>((resolve) => b.on('open', () => resolve())),
    ]);

    a.send(JSON.stringify({ op: 'connect', id: 's1', url: upstreamUrl, headers: {} }));
    b.send(JSON.stringify({ op: 'connect', id: 's1', url: upstreamUrl, headers: {} }));

    const [ackA, ackB] = await Promise.all([waitForMessage(a), waitForMessage(b)]);
    expect(parseRelayFrame(ackA).data).toEqual({ connected: true });
    expect(parseRelayFrame(ackB).data).toEqual({ connected: true });

    a.send(JSON.stringify({ op: 'send', id: 's1', payload: 'from-a' }));
    b.send(JSON.stringify({ op: 'send', id: 's1', payload: 'from-b' }));

    const [msgA, msgB] = await Promise.all([waitForMessage(a), waitForMessage(b)]);
    expect(parseRelayFrame(msgA).data).toBe('from-a');
    expect(parseRelayFrame(msgB).data).toBe('from-b');

    a.close();
    b.close();
  });

  it('restricts upstream WS host in production', async () => {
    // New relay instance with production restriction (allowed host = localhost)
    const config: ServerConfig = {
      port: relayPort,
      nodeEnv: 'production',
      corsOrigin: true,
      httpTimeoutMs: 30_000,
      wsReconnectDelayMs: 100,
      wsMaxReconnectAttempts: 0,
      requestSizeLimit: 10 * 1024 * 1024,
      kalshiBaseUrl: `http://127.0.0.1:${kalshiPort}`,
    };
    const prodRelay = new WebSocketRelay(config, noopLogger);
    relayWsServer.removeAllListeners('connection');
    relayWsServer.on('connection', (ws) => prodRelay.handleClientConnection(ws));

    const client = new WebSocket(`ws://127.0.0.1:${relayPort}/relay/ws`);
    await new Promise<void>((resolve) => client.on('open', () => resolve()));

    // Host differs (127.0.0.1 vs localhost) -> should be rejected in production
    client.send(
      JSON.stringify({
        op: 'connect',
        id: 's1',
        url: `ws://localhost:${kalshiPort}/trade-api/ws/v2`,
        headers: {},
      })
    );

    const msgRaw = await waitForMessage(client);
    const frame = parseRelayFrame(msgRaw);
    expect(frame.id).toBe('s1');
    expect(frame.type).toBe('error');
    expect(String(frame.error)).toContain('Upstream WebSocket host must be');

    client.close();
  });
});
