import crypto from 'node:crypto';
import fs from 'node:fs';
import path from 'node:path';

const keyFile = fs.readFileSync(path.resolve(import.meta.dirname, '../../localtestapikeys.streamrift'), 'utf-8');
const lines = keyFile.split('\n');
const API_KEY_ID = lines[1].trim();
const PEM = lines.slice(3).join('\n').trim();

function sign(method, p, ts) {
  const s = crypto.createSign('RSA-SHA256');
  s.update(ts + method + p); s.end();
  return s.sign({ key: PEM, padding: crypto.constants.RSA_PKCS1_PSS_PADDING, saltLength: 32 }, 'base64');
}

async function f(method, apiPath, q = {}) {
  const url = new URL(apiPath, 'https://api.elections.kalshi.com');
  for (const [k, v] of Object.entries(q)) {
    if (v !== undefined && v !== null) url.searchParams.set(k, String(v));
  }
  const ts = Math.floor(Date.now() / 1000).toString();
  const sig = sign(method, url.pathname + url.search, ts);
  const r = await fetch(url.toString(), {
    method,
    headers: {
      'Content-Type': 'application/json',
      'KALSHI-ACCESS-KEY': API_KEY_ID,
      'KALSHI-ACCESS-SIGNATURE': sig,
      'KALSHI-ACCESS-TIMESTAMP': ts,
    },
  });
  if (!r.ok) throw new Error(`${r.status}: ${(await r.text()).slice(0, 200)}`);
  return r.json();
}

function save(name, data) {
  fs.writeFileSync(path.join(import.meta.dirname, name), JSON.stringify(data, null, 2));
}

function sleep(ms) { return new Promise(r => setTimeout(r, ms)); }

async function main() {
  // Get a live CBB market
  console.log('=== CBB Men\'s open markets ===');
  const d = await f('GET', '/trade-api/v2/markets', { series_ticker: 'KXNCAAMBGAME', status: 'open', limit: 5 });
  const markets = d.markets || [];
  console.log(`Count: ${markets.length}`);

  if (markets.length === 0) {
    console.log('No open CBB markets found');
    return;
  }

  // Show first 5 markets
  for (const m of markets) {
    console.log(`\n  ${m.ticker}`);
    console.log(`    title: "${m.title}"`);
    console.log(`    subtitle: "${m.subtitle || ''}"`);
    console.log(`    event_ticker: ${m.event_ticker}`);
    console.log(`    status: ${m.status}`);
    console.log(`    yes_bid: ${m.yes_bid}, yes_ask: ${m.yes_ask}`);
    console.log(`    no_bid: ${m.no_bid}, no_ask: ${m.no_ask}`);
    console.log(`    last_price: ${m.last_price}`);
    console.log(`    volume: ${m.volume}, open_interest: ${m.open_interest}`);
    console.log(`    close_time: ${m.close_time}`);
    console.log(`    open_time: ${m.open_time}`);
    console.log(`    expiration_time: ${m.expiration_time}`);
  }

  save('markets-cbb-mens-open-sample.json', { markets: markets.slice(0, 5) });

  await sleep(300);

  // Orderbook for first market
  const ticker = markets[0].ticker;
  console.log(`\n=== Orderbook: ${ticker} ===`);
  const ob = await f('GET', `/trade-api/v2/markets/${ticker}/orderbook`);
  save('orderbook-cbb-live.json', ob);
  const obd = ob.orderbook || ob;
  console.log(`  Top-level keys: ${Object.keys(ob).join(', ')}`);
  console.log(`  Orderbook keys: ${Object.keys(obd).join(', ')}`);
  console.log(`  yes: ${JSON.stringify(obd.yes?.slice(0, 5))}`);
  console.log(`  no: ${JSON.stringify(obd.no?.slice(0, 5))}`);

  await sleep(300);

  // Market detail
  console.log(`\n=== Market detail: ${ticker} ===`);
  const detail = await f('GET', `/trade-api/v2/markets/${ticker}`);
  save('market-detail-cbb-live.json', detail);
  const md = detail.market || detail;
  console.log(`  All keys: ${Object.keys(md).sort().join(', ')}`);
  console.log(`  title: "${md.title}"`);
  console.log(`  yes_sub_title: "${md.yes_sub_title || ''}"`);
  console.log(`  no_sub_title: "${md.no_sub_title || ''}"`);

  await sleep(300);

  // Event
  const eticker = markets[0].event_ticker;
  console.log(`\n=== Event: ${eticker} ===`);
  const ed = await f('GET', `/trade-api/v2/events/${eticker}`);
  save('event-cbb-live.json', ed);
  const ev = ed.event || ed;
  console.log(`  Wrapper keys: ${Object.keys(ed).join(', ')}`);
  console.log(`  Event keys: ${Object.keys(ev).sort().join(', ')}`);
  console.log(`  title: "${ev.title}"`);
  console.log(`  sub_title: "${ev.sub_title || ''}"`);
  console.log(`  series_ticker: ${ev.series_ticker}`);
  console.log(`  category: ${ev.category}`);
  if (ed.markets) {
    console.log(`  Nested markets: ${ed.markets.length}`);
    ed.markets.forEach(m => console.log(`    ${m.ticker} — "${m.title}" (${m.status})`));
  }

  // Count how many CBB men's games are open across all pages
  console.log('\n=== Total open CBB men\'s games ===');
  let total = 0;
  let cursor;
  let pages = 0;
  do {
    const params = { series_ticker: 'KXNCAAMBGAME', status: 'open', limit: 200 };
    if (cursor) params.cursor = cursor;
    const page = await f('GET', '/trade-api/v2/markets', params);
    const m = page.markets || [];
    total += m.length;
    cursor = page.cursor;
    pages++;
    console.log(`  Page ${pages}: ${m.length} markets (cursor: ${!!cursor})`);
    await sleep(300);
  } while (cursor && pages < 10);
  console.log(`  TOTAL: ${total} open men's CBB markets`);

  // Also count women's
  console.log('\n=== Total open CBB women\'s games ===');
  total = 0; cursor = undefined; pages = 0;
  do {
    const params = { series_ticker: 'KXNCAAWBGAME', status: 'open', limit: 200 };
    if (cursor) params.cursor = cursor;
    const page = await f('GET', '/trade-api/v2/markets', params);
    const m = page.markets || [];
    total += m.length;
    cursor = page.cursor;
    pages++;
    console.log(`  Page ${pages}: ${m.length} markets (cursor: ${!!cursor})`);
    await sleep(300);
  } while (cursor && pages < 10);
  console.log(`  TOTAL: ${total} open women's CBB markets`);

  // NBA total
  console.log('\n=== Total open NBA games ===');
  total = 0; cursor = undefined; pages = 0;
  do {
    const params = { series_ticker: 'KXNBAGAME', status: 'open', limit: 200 };
    if (cursor) params.cursor = cursor;
    const page = await f('GET', '/trade-api/v2/markets', params);
    const m = page.markets || [];
    total += m.length;
    cursor = page.cursor;
    pages++;
    console.log(`  Page ${pages}: ${m.length} markets (cursor: ${!!cursor})`);
    await sleep(300);
  } while (cursor && pages < 10);
  console.log(`  TOTAL: ${total} open NBA markets`);

  console.log('\n=== Done ===');
}

main().catch(e => { console.error('Fatal:', e); process.exit(1); });
