/**
 * Error Types
 *
 * Standardized error types for the application.
 */
/**
 * Error codes used throughout the application.
 */
export var ErrorCode;
(function (ErrorCode) {
    ErrorCode["NETWORK_ERROR"] = "NETWORK_ERROR";
    ErrorCode["TIMEOUT_ERROR"] = "TIMEOUT_ERROR";
    ErrorCode["VALIDATION_ERROR"] = "VALIDATION_ERROR";
    ErrorCode["AUTH_ERROR"] = "AUTH_ERROR";
    ErrorCode["UPSTREAM_ERROR"] = "UPSTREAM_ERROR";
    ErrorCode["CONNECTION_ERROR"] = "CONNECTION_ERROR";
    ErrorCode["INVALID_OPERATION"] = "INVALID_OPERATION";
})(ErrorCode || (ErrorCode = {}));
/**
 * Base error class for relay-related errors.
 */
export class RelayError extends Error {
    code;
    statusCode;
    requestId;
    constructor(message, code, statusCode = 500, requestId) {
        super(message);
        this.name = 'RelayError';
        this.code = code;
        this.statusCode = statusCode;
        this.requestId = requestId;
        Error.captureStackTrace(this, this.constructor);
    }
}
/**
 * Base error class for Kalshi API-related errors.
 */
export class KalshiApiError extends Error {
    code;
    statusCode;
    response;
    constructor(message, code, statusCode, response) {
        super(message);
        this.name = 'KalshiApiError';
        this.code = code;
        this.statusCode = statusCode;
        this.response = response;
        Error.captureStackTrace(this, this.constructor);
    }
}
/**
 * Network connectivity error.
 */
export class NetworkError extends RelayError {
    constructor(message = 'Network error', requestId) {
        super(message, ErrorCode.NETWORK_ERROR, 500, requestId);
        this.name = 'NetworkError';
    }
}
/**
 * Request timeout error.
 */
export class TimeoutError extends RelayError {
    constructor(message = 'Request timeout', requestId) {
        super(message, ErrorCode.TIMEOUT_ERROR, 504, requestId);
        this.name = 'TimeoutError';
    }
}
/**
 * Request validation error.
 */
export class ValidationError extends RelayError {
    constructor(message = 'Validation error', requestId) {
        super(message, ErrorCode.VALIDATION_ERROR, 400, requestId);
        this.name = 'ValidationError';
    }
}
/**
 * Authentication/signing error.
 */
export class AuthenticationError extends KalshiApiError {
    constructor(message = 'Authentication error', response) {
        super(message, ErrorCode.AUTH_ERROR, 401, response);
        this.name = 'AuthenticationError';
    }
}
/**
 * Upstream (Kalshi API) error.
 */
export class UpstreamError extends RelayError {
    constructor(message, statusCode, requestId) {
        super(message, ErrorCode.UPSTREAM_ERROR, statusCode, requestId);
        this.name = 'UpstreamError';
    }
}
/**
 * Connection error (WebSocket).
 */
export class ConnectionError extends RelayError {
    constructor(message = 'Connection error', streamId) {
        super(message, ErrorCode.CONNECTION_ERROR, 500, streamId);
        this.name = 'ConnectionError';
    }
}
/**
 * Invalid operation error.
 */
export class InvalidOperationError extends RelayError {
    constructor(message = 'Invalid operation', streamId) {
        super(message, ErrorCode.INVALID_OPERATION, 400, streamId);
        this.name = 'InvalidOperationError';
    }
}
//# sourceMappingURL=errors.js.map