/**
 * Error Types
 *
 * Standardized error types for the application.
 */

/**
 * Error codes used throughout the application.
 */
export enum ErrorCode {
  NETWORK_ERROR = 'NETWORK_ERROR',
  TIMEOUT_ERROR = 'TIMEOUT_ERROR',
  VALIDATION_ERROR = 'VALIDATION_ERROR',
  AUTH_ERROR = 'AUTH_ERROR',
  UPSTREAM_ERROR = 'UPSTREAM_ERROR',
  CONNECTION_ERROR = 'CONNECTION_ERROR',
  INVALID_OPERATION = 'INVALID_OPERATION',
}

/**
 * Base error class for relay-related errors.
 */
export class RelayError extends Error {
  code: string;
  statusCode: number;
  requestId?: string;

  constructor(message: string, code: ErrorCode, statusCode: number = 500, requestId?: string) {
    super(message);
    this.name = 'RelayError';
    this.code = code;
    this.statusCode = statusCode;
    this.requestId = requestId;
    Error.captureStackTrace(this, this.constructor);
  }
}

/**
 * Base error class for Kalshi API-related errors.
 */
export class KalshiApiError extends Error {
  code: string;
  statusCode: number;
  response?: unknown;

  constructor(message: string, code: ErrorCode, statusCode: number, response?: unknown) {
    super(message);
    this.name = 'KalshiApiError';
    this.code = code;
    this.statusCode = statusCode;
    this.response = response;
    Error.captureStackTrace(this, this.constructor);
  }
}

/**
 * Network connectivity error.
 */
export class NetworkError extends RelayError {
  constructor(message: string = 'Network error', requestId?: string) {
    super(message, ErrorCode.NETWORK_ERROR, 500, requestId);
    this.name = 'NetworkError';
  }
}

/**
 * Request timeout error.
 */
export class TimeoutError extends RelayError {
  constructor(message: string = 'Request timeout', requestId?: string) {
    super(message, ErrorCode.TIMEOUT_ERROR, 504, requestId);
    this.name = 'TimeoutError';
  }
}

/**
 * Request validation error.
 */
export class ValidationError extends RelayError {
  constructor(message: string = 'Validation error', requestId?: string) {
    super(message, ErrorCode.VALIDATION_ERROR, 400, requestId);
    this.name = 'ValidationError';
  }
}

/**
 * Authentication/signing error.
 */
export class AuthenticationError extends KalshiApiError {
  constructor(message: string = 'Authentication error', response?: unknown) {
    super(message, ErrorCode.AUTH_ERROR, 401, response);
    this.name = 'AuthenticationError';
  }
}

/**
 * Upstream (Kalshi API) error.
 */
export class UpstreamError extends RelayError {
  constructor(message: string, statusCode: number, requestId?: string) {
    super(message, ErrorCode.UPSTREAM_ERROR, statusCode, requestId);
    this.name = 'UpstreamError';
  }
}

/**
 * Connection error (WebSocket).
 */
export class ConnectionError extends RelayError {
  constructor(message: string = 'Connection error', streamId?: string) {
    super(message, ErrorCode.CONNECTION_ERROR, 500, streamId);
    this.name = 'ConnectionError';
  }
}

/**
 * Invalid operation error.
 */
export class InvalidOperationError extends RelayError {
  constructor(message: string = 'Invalid operation', streamId?: string) {
    super(message, ErrorCode.INVALID_OPERATION, 400, streamId);
    this.name = 'InvalidOperationError';
  }
}
