/**
 * Auth Commands
 *
 * Manage Kalshi API credentials
 */

import { Command } from 'commander';
import * as fs from 'fs';
import * as readline from 'readline';
import {
  setCredentials,
  getCredentials,
  clearCredentials,
  isAuthenticated,
  getConfigPath,
  loadConfig,
} from '../lib/config';
import { createKalshiClient } from '../lib/kalshi';
import { output, success, error } from '../lib/output';

export const authCommand = new Command('auth').description(
  'Manage Kalshi API authentication'
);

/**
 * Prompt for input (with optional hidden input for sensitive data)
 */
async function prompt(question: string, hidden = false): Promise<string> {
  const rl = readline.createInterface({
    input: process.stdin,
    output: process.stdout,
  });

  return new Promise((resolve) => {
    if (hidden && process.stdin.isTTY) {
      // For hidden input, we need to handle it differently
      process.stdout.write(question);
      const stdin = process.stdin;
      stdin.setRawMode(true);
      stdin.resume();
      stdin.setEncoding('utf8');

      let input = '';
      const onData = (char: string) => {
        if (char === '\n' || char === '\r') {
          stdin.setRawMode(false);
          stdin.removeListener('data', onData);
          process.stdout.write('\n');
          rl.close();
          resolve(input);
        } else if (char === '\u0003') {
          // Ctrl+C
          process.exit();
        } else if (char === '\u007F') {
          // Backspace
          if (input.length > 0) {
            input = input.slice(0, -1);
          }
        } else {
          input += char;
        }
      };
      stdin.on('data', onData);
    } else {
      rl.question(question, (answer) => {
        rl.close();
        resolve(answer);
      });
    }
  });
}

/**
 * Read multiline input until END marker or double blank line
 */
async function promptMultiline(question: string): Promise<string> {
  console.log(question);
  console.log('');
  console.log('Paste your PEM key below, then:');
  console.log('  - Type END on a new line and press Enter, OR');
  console.log('  - Press Enter twice after the key');
  console.log('');

  const rl = readline.createInterface({
    input: process.stdin,
    output: process.stdout,
  });

  return new Promise((resolve) => {
    const lines: string[] = [];
    let emptyLineCount = 0;

    rl.on('line', (line) => {
      // Check for END marker
      if (line.trim().toUpperCase() === 'END') {
        rl.close();
        resolve(lines.join('\n'));
        return;
      }

      if (line === '') {
        emptyLineCount++;
        // After the closing line of PEM, allow single blank line to finish
        if (emptyLineCount >= 1 && lines.length > 0 && lines[lines.length - 1].includes('-----END')) {
          rl.close();
          resolve(lines.join('\n'));
          return;
        }
        // Two blank lines in a row also finishes
        if (emptyLineCount >= 2 && lines.length > 0) {
          rl.close();
          resolve(lines.join('\n'));
          return;
        }
      } else {
        emptyLineCount = 0;
        lines.push(line);
      }
    });
  });
}

// ============================================================================
// auth login
// ============================================================================
authCommand
  .command('login')
  .description('Store Kalshi API credentials')
  .option('--access-key <key>', 'Kalshi API access key')
  .option('--private-key <pem>', 'Kalshi API private key (PEM format)')
  .option('--private-key-file <path>', 'Path to private key file')
  .option('--env <environment>', 'Environment: prod or demo')
  .action(async (options) => {
    try {
      console.log('');
      console.log('=== Kalshi API Authentication ===');
      console.log('');

      let accessKey = options.accessKey;
      let privateKey = options.privateKey;
      let environment = options.env as 'prod' | 'demo' | undefined;

      // Prompt for environment if not provided
      if (!environment) {
        const envChoice = await prompt('Environment (prod/demo) [prod]: ');
        environment = envChoice.toLowerCase() === 'demo' ? 'demo' : 'prod';
      }

      console.log(`Using ${environment} environment`);
      console.log('');

      // Prompt for access key if not provided
      if (!accessKey) {
        accessKey = await prompt('Access Key ID: ');
      }

      // Read private key from file or prompt
      if (options.privateKeyFile) {
        console.log(`Reading private key from: ${options.privateKeyFile}`);
        privateKey = fs.readFileSync(options.privateKeyFile, 'utf-8');
      } else if (!privateKey) {
        privateKey = await promptMultiline('Private Key (PEM):');
      }

      if (!accessKey || !privateKey) {
        error('Access key and private key are required');
        process.exit(1);
      }

      // Validate PEM format
      if (!privateKey.includes('-----BEGIN') || !privateKey.includes('-----END')) {
        error('Invalid private key format. Expected PEM format with BEGIN/END markers.');
        process.exit(1);
      }

      // Test credentials by making a simple API call
      console.log('');
      console.log('Testing credentials...');
      setCredentials(accessKey, privateKey, environment);

      try {
        const client = createKalshiClient();
        const balance = await client.getBalance();
        console.log('');
        success(`Authenticated successfully!`);
        console.log(`  Environment: ${environment}`);
        console.log(`  Balance: $${(balance.balance / 100).toFixed(2)}`);
        console.log(`  Config: ${getConfigPath()}`);
      } catch (err) {
        // Clear invalid credentials
        clearCredentials();
        console.log('');
        error(`Authentication failed: ${err instanceof Error ? err.message : err}`);
        process.exit(1);
      }
    } catch (err) {
      error(`Login failed: ${err instanceof Error ? err.message : err}`);
      process.exit(1);
    }
  });

// ============================================================================
// auth logout
// ============================================================================
authCommand
  .command('logout')
  .description('Remove stored credentials')
  .action(() => {
    if (!isAuthenticated()) {
      console.log('Not currently authenticated');
      return;
    }

    clearCredentials();
    success('Credentials removed');
  });

// ============================================================================
// auth status
// ============================================================================
authCommand
  .command('status')
  .description('Show authentication status')
  .action(() => {
    const config = loadConfig();
    const creds = getCredentials();

    if (!creds) {
      output({
        authenticated: false,
        configPath: getConfigPath(),
      });
      return;
    }

    output({
      authenticated: true,
      environment: creds.environment,
      accessKey: creds.accessKey.slice(0, 8) + '...',
      authenticatedAt: config.authenticatedAt,
      configPath: getConfigPath(),
    });
  });

// ============================================================================
// auth whoami
// ============================================================================
authCommand
  .command('whoami')
  .description('Show current user and test connection')
  .action(async () => {
    if (!isAuthenticated()) {
      error('Not authenticated. Run `galactus auth login` first.');
      process.exit(1);
    }

    try {
      const client = createKalshiClient();
      const balance = await client.getBalance();
      const creds = getCredentials()!;

      output({
        environment: creds.environment,
        accessKey: creds.accessKey.slice(0, 8) + '...',
        balance: balance.balance / 100, // Convert cents to dollars
        status: 'connected',
      });
    } catch (err) {
      error(`Connection failed: ${err instanceof Error ? err.message : err}`);
      process.exit(1);
    }
  });
