/**
 * Polymarket Commands
 *
 * Explore Polymarket sports markets via the Gamma API
 */

import { Command } from 'commander';
import { output, outputTable, error } from '../lib/output';

const GAMMA_BASE = 'https://gamma-api.polymarket.com';

interface GammaMarket {
  id?: string;
  slug?: string;
  question?: string;
  conditionId?: string;
  clobTokenIds?: string | string[];
  outcomes?: string;
  outcomePrices?: string;
  volume?: string;
  liquidity?: string;
  startDate?: string;
  endDate?: string;
  category?: string;
  enableOrderBook?: boolean;
  active?: boolean;
  closed?: boolean;
  archived?: boolean;
  events?: Array<{
    id?: string;
    slug?: string;
    title?: string;
    category?: string;
    negRisk?: boolean;
  }>;
}

interface GammaEvent {
  slug?: string;
  title?: string;
  startTime?: string;
  score?: string;
  live?: boolean;
  ended?: boolean;
  markets?: Array<{
    slug?: string;
    sportsMarketType?: string;
    conditionId?: string;
    clobTokenIds?: string | string[];
    outcomePrices?: string | string[];
    liquidity?: string | number;
  }>;
}

async function fetchGamma(endpoint: string): Promise<unknown> {
  const url = `${GAMMA_BASE}${endpoint}`;

  const resp = await fetch(url, {
    headers: {
      Accept: 'application/json',
      'User-Agent': 'Mozilla/5.0',
    },
  });

  if (!resp.ok) {
    throw new Error(`Gamma API error: ${resp.status} ${resp.statusText}`);
  }

  return resp.json();
}

export const polymarketCommand = new Command('polymarket')
  .alias('poly')
  .description('Explore Polymarket sports markets via Gamma API');

// ============================================================================
// polymarket search <query>
// ============================================================================
polymarketCommand
  .command('search <query...>')
  .description('Search for markets by text')
  .option('-l, --limit <n>', 'Max results', '50')
  .action(async (queryParts: string[], options) => {
    const query = queryParts.join(' ');
    try {
      const data = (await fetchGamma(
        `/markets?_q=${encodeURIComponent(query)}&active=true&closed=false&limit=${options.limit}`
      )) as GammaMarket[];

      const rows = data.map((m) => ({
        slug: m.slug || '',
        question: m.question?.slice(0, 60) || '',
        category: m.events?.[0]?.category || m.category || '',
        active: m.active ? 'yes' : 'no',
      }));

      output(rows);
    } catch (err) {
      error(`Failed: ${err instanceof Error ? err.message : err}`);
      process.exit(1);
    }
  });

// ============================================================================
// polymarket sports
// ============================================================================
polymarketCommand
  .command('sports')
  .description('List sports-related markets')
  .action(async () => {
    const queries = ['NBA', 'NFL', 'NHL', 'MLB', 'soccer', 'football', 'basketball', 'hockey'];
    const allMarkets: GammaMarket[] = [];
    const seenSlugs = new Set<string>();

    try {
      for (const q of queries) {
        const data = (await fetchGamma(
          `/markets?_q=${encodeURIComponent(q)}&active=true&closed=false&limit=100`
        )) as GammaMarket[];
        for (const m of data) {
          if (m.slug && !seenSlugs.has(m.slug)) {
            seenSlugs.add(m.slug);
            allMarkets.push(m);
          }
        }
      }

      // Group by category
      const byCategory = new Map<string, GammaMarket[]>();
      for (const m of allMarkets) {
        const cat = m.events?.[0]?.category || m.category || 'Unknown';
        if (!byCategory.has(cat)) byCategory.set(cat, []);
        byCategory.get(cat)!.push(m);
      }

      for (const [category, markets] of byCategory.entries()) {
        console.log(`\n=== ${category} (${markets.length} markets) ===\n`);
        const rows = markets.slice(0, 10).map((m) => ({
          slug: m.slug || '',
          question: m.question?.slice(0, 70) || '',
        }));
        outputTable(rows);
        if (markets.length > 10) {
          console.log(`  ... and ${markets.length - 10} more`);
        }
      }
    } catch (err) {
      error(`Failed: ${err instanceof Error ? err.message : err}`);
      process.exit(1);
    }
  });

// ============================================================================
// polymarket slug <slug>
// ============================================================================
polymarketCommand
  .command('slug <slug>')
  .description('Look up a specific market by slug')
  .action(async (slug: string) => {
    try {
      const data = (await fetchGamma(
        `/markets?slug=${encodeURIComponent(slug)}`
      )) as GammaMarket[];

      if (data.length === 0) {
        error(`No market found for slug: ${slug}`);
        process.exit(1);
      }

      const market = data[0];

      // Parse token IDs
      let tokenIds: string[] = [];
      if (Array.isArray(market.clobTokenIds)) {
        tokenIds = market.clobTokenIds;
      } else if (typeof market.clobTokenIds === 'string') {
        try {
          tokenIds = JSON.parse(market.clobTokenIds);
        } catch {
          // Invalid JSON, keep empty array
        }
      }

      output({
        slug: market.slug,
        question: market.question,
        conditionId: market.conditionId,
        tokenIds: tokenIds.join(', '),
        outcomes: market.outcomes,
        category: market.events?.[0]?.category || market.category,
        active: market.active,
        closed: market.closed,
        volume: market.volume,
        liquidity: market.liquidity,
      });
    } catch (err) {
      error(`Failed: ${err instanceof Error ? err.message : err}`);
      process.exit(1);
    }
  });

// ============================================================================
// polymarket series
// ============================================================================
polymarketCommand
  .command('series')
  .description('List sports series')
  .action(async () => {
    const sportsSeries = [
      'nba-2026',
      'nfl-2025',
      'nhl-2025',
      'mlb-2025',
      'nba-2025',
      'nfl-2026',
      'nhl-2026',
    ];

    try {
      for (const seriesSlug of sportsSeries) {
        const data = (await fetchGamma(
          `/events?series_slug=${encodeURIComponent(seriesSlug)}&limit=5&active=true`
        )) as GammaEvent[];

        if (data.length > 0) {
          console.log(`\n=== Series: ${seriesSlug} (${data.length}+ events) ===`);
          const rows = data.slice(0, 5).map((e) => ({
            slug: e.slug || 'no-slug',
            title: e.title?.slice(0, 60) || '',
            start: e.startTime || 'N/A',
          }));
          outputTable(rows);
        }
      }
    } catch (err) {
      error(`Failed: ${err instanceof Error ? err.message : err}`);
      process.exit(1);
    }
  });

// ============================================================================
// polymarket events <series-slug>
// ============================================================================
polymarketCommand
  .command('events <seriesSlug>')
  .description('List events in a series (e.g., nba-2026)')
  .option('-l, --limit <n>', 'Max results', '30')
  .action(async (seriesSlug: string, options) => {
    try {
      const data = (await fetchGamma(
        `/events?series_slug=${encodeURIComponent(seriesSlug)}&limit=${options.limit}&active=true`
      )) as GammaEvent[];

      const rows = data.map((e) => ({
        slug: e.slug || '',
        title: e.title?.slice(0, 50) || '',
        start: e.startTime || 'N/A',
        live: e.live ? 'yes' : 'no',
        ended: e.ended ? 'yes' : 'no',
        markets: e.markets?.length || 0,
      }));

      output(rows);
    } catch (err) {
      error(`Failed: ${err instanceof Error ? err.message : err}`);
      process.exit(1);
    }
  });

// ============================================================================
// Shared handler for inspecting event markets by type
// ============================================================================
const MARKET_TYPE_FILTERS: Record<string, string[]> = {
  spread: ['spreads', 'spread', 'tennis_set_handicap'],
  total: ['totals', 'total', 'tennis_match_totals'],
};

async function inspectEventMarkets(gameSlug: string, marketType: 'spread' | 'total') {
  try {
    const data = (await fetchGamma(
      `/events?slug=${encodeURIComponent(gameSlug)}`
    )) as GammaEvent[];

    if (!Array.isArray(data) || data.length === 0) {
      error(`No event found for slug: ${gameSlug}`);
      process.exit(1);
    }

    const evt = data[0];
    const markets = Array.isArray(evt?.markets) ? evt.markets : [];
    const allowedTypes = MARKET_TYPE_FILTERS[marketType] ?? [];
    const filtered = markets.filter((m) => {
      const t = String(m?.sportsMarketType ?? '').toLowerCase();
      return allowedTypes.includes(t);
    });

    const rows = filtered.map((m) => ({
      slug: m?.slug || '',
      type: m?.sportsMarketType || '',
      conditionId: m?.conditionId || '',
      tokenIds:
        Array.isArray(m?.clobTokenIds)
          ? m.clobTokenIds.join(', ')
          : typeof m?.clobTokenIds === 'string'
            ? m.clobTokenIds
            : '',
      outcomePrices:
        Array.isArray(m?.outcomePrices)
          ? m.outcomePrices.join(', ')
          : typeof m?.outcomePrices === 'string'
            ? m.outcomePrices
            : '',
      liquidity: m?.liquidity ?? '',
    }));

    output({
      eventSlug: evt?.slug ?? gameSlug,
      title: evt?.title ?? '',
      [`${marketType}Count`]: rows.length,
    });
    output(rows);
  } catch (err) {
    error(`Failed: ${err instanceof Error ? err.message : err}`);
    process.exit(1);
  }
}

// ============================================================================
// polymarket spread-event <game-slug>
// ============================================================================
polymarketCommand
  .command('spread-event <gameSlug>')
  .description('Inspect spread markets for a specific game slug (e.g., nba-ind-mil-2026-02-06)')
  .action((gameSlug: string) => inspectEventMarkets(gameSlug, 'spread'));

// ============================================================================
// polymarket total-event <game-slug>
// ============================================================================
polymarketCommand
  .command('total-event <gameSlug>')
  .description('Inspect total markets for a specific game slug (e.g., nba-ind-mil-2026-02-06)')
  .action((gameSlug: string) => inspectEventMarkets(gameSlug, 'total'));

// ============================================================================
// polymarket analyze
// ============================================================================
polymarketCommand
  .command('analyze')
  .description('Analyze slug patterns from sports markets')
  .action(async () => {
    const queries = ['NBA', 'NFL', 'NHL', 'MLB', 'UFC', 'tennis', 'soccer'];
    const allSlugs: Array<{ slug: string; question: string }> = [];
    const seenSlugs = new Set<string>();

    try {
      for (const q of queries) {
        const data = (await fetchGamma(
          `/markets?_q=${encodeURIComponent(q)}&limit=50`
        )) as GammaMarket[];
        for (const m of data) {
          if (m.slug && !seenSlugs.has(m.slug)) {
            seenSlugs.add(m.slug);
            allSlugs.push({
              slug: m.slug,
              question: m.question || '',
            });
          }
        }
      }

      console.log(`Collected ${allSlugs.length} slugs\n`);

      // Group by detected pattern
      const patterns: Record<string, string[]> = {
        'nba-*': [],
        'nfl-*': [],
        'nhl-*': [],
        'mlb-*': [],
        'ufc-*': [],
        'soccer-*': [],
        'tennis-*': [],
        other: [],
      };

      const patternMatchers: Array<{ prefix: string; key: string }> = [
        { prefix: 'nba-', key: 'nba-*' },
        { prefix: 'nfl-', key: 'nfl-*' },
        { prefix: 'nhl-', key: 'nhl-*' },
        { prefix: 'mlb-', key: 'mlb-*' },
        { prefix: 'ufc-', key: 'ufc-*' },
      ];

      for (const { slug, question } of allSlugs) {
        const lower = slug.toLowerCase();
        let matched = false;

        for (const { prefix, key } of patternMatchers) {
          if (lower.startsWith(prefix)) {
            patterns[key].push(`${slug} -> ${question.slice(0, 50)}`);
            matched = true;
            break;
          }
        }

        if (!matched) {
          if (lower.includes('soccer') || lower.includes('football')) {
            patterns['soccer-*'].push(`${slug} -> ${question.slice(0, 50)}`);
          } else if (lower.includes('tennis')) {
            patterns['tennis-*'].push(`${slug} -> ${question.slice(0, 50)}`);
          } else {
            patterns['other'].push(`${slug} -> ${question.slice(0, 50)}`);
          }
        }
      }

      for (const [pattern, slugs] of Object.entries(patterns)) {
        if (slugs.length === 0) continue;
        console.log(`\n=== ${pattern} (${slugs.length} markets) ===\n`);
        for (const s of slugs.slice(0, 15)) {
          console.log(`  ${s}`);
        }
        if (slugs.length > 15) {
          console.log(`  ... and ${slugs.length - 15} more`);
        }
      }
    } catch (err) {
      error(`Failed: ${err instanceof Error ? err.message : err}`);
      process.exit(1);
    }
  });
