/**
 * Kalshi API Client for CLI
 *
 * Direct HTTP calls to Kalshi API (no relay needed - no CORS in Node.js)
 * Uses RSA-PSS signing for authentication
 */

import * as crypto from 'crypto';
import { getCredentials } from './config';

type Environment = 'prod' | 'demo';

const BASE_URLS: Record<Environment, string> = {
  prod: 'https://api.elections.kalshi.com/trade-api/v2',
  demo: 'https://demo-api.kalshi.co/trade-api/v2',
};

export interface KalshiSeries {
  ticker: string;
  title: string;
  category?: string;
  subcategory?: string;
  tags?: string[];
  frequency?: string;
}

export interface KalshiMarket {
  market_ticker: string;
  event_ticker: string;
  series_ticker: string;
  title: string;
  subtitle?: string;
  status: string;
  yes_bid?: number | null;
  yes_ask?: number | null;
  no_bid?: number | null;
  no_ask?: number | null;
  last_price?: number | null;
  volume?: number;
  open_interest?: number;
  floor_strike?: number | null;
  cap_strike?: number | null;
  strike_type?: string;
  functional_strike?: string;
  expiration_time?: string;
  close_time?: string;
}

export interface KalshiEvent {
  event_ticker: string;
  series_ticker: string;
  title: string;
  subtitle?: string;
  category?: string;
}

/**
 * Convert PEM private key to crypto.KeyObject
 */
function parsePrivateKey(pem: string): crypto.KeyObject {
  // Normalize PEM format
  let normalizedPem = pem.trim();

  // Handle PKCS#1 format (RSA PRIVATE KEY) - convert to PKCS#8
  if (normalizedPem.includes('-----BEGIN RSA PRIVATE KEY-----')) {
    // Node's crypto can handle PKCS#1 directly
    return crypto.createPrivateKey({
      key: normalizedPem,
      format: 'pem',
    });
  }

  // Handle PKCS#8 format (PRIVATE KEY)
  if (normalizedPem.includes('-----BEGIN PRIVATE KEY-----')) {
    return crypto.createPrivateKey({
      key: normalizedPem,
      format: 'pem',
    });
  }

  throw new Error('Unsupported private key format. Expected PEM format.');
}

/**
 * Sign a message using RSA-PSS with SHA-256
 * saltLength: 32 matches the dashboard's WebCrypto implementation
 */
function signMessage(privateKey: crypto.KeyObject, message: string): string {
  const signature = crypto.sign('sha256', Buffer.from(message), {
    key: privateKey,
    padding: crypto.constants.RSA_PKCS1_PSS_PADDING,
    saltLength: 32, // Must match dashboard's WebCrypto saltLength
  });
  return signature.toString('base64');
}

/**
 * Normalize path for Kalshi signing
 * - Strip query params
 * - Ensure leading slash
 * - Remove trailing slash (except root "/")
 */
function normalizePath(path: string): string {
  let normalized = path.split('?')[0].trim();
  if (!normalized.startsWith('/')) {
    normalized = `/${normalized}`;
  }
  if (normalized.length > 1 && normalized.endsWith('/')) {
    normalized = normalized.slice(0, -1);
  }
  return normalized;
}

/**
 * Build authentication headers for Kalshi API
 * Signing payload format: ${timestamp}${METHOD}${path}
 */
function buildAuthHeaders(
  accessKey: string,
  privateKey: crypto.KeyObject,
  method: string,
  path: string
): Record<string, string> {
  const timestamp = Date.now().toString();
  const upperMethod = method.toUpperCase();
  const normalizedPath = normalizePath(path);

  // Kalshi signature payload format
  const message = `${timestamp}${upperMethod}${normalizedPath}`;
  const signature = signMessage(privateKey, message);

  return {
    'KALSHI-ACCESS-KEY': accessKey,
    'KALSHI-ACCESS-SIGNATURE': signature,
    'KALSHI-ACCESS-TIMESTAMP': timestamp,
    'Content-Type': 'application/json',
  };
}

/**
 * Create authenticated Kalshi API client
 */
export function createKalshiClient() {
  const creds = getCredentials();
  if (!creds) {
    throw new Error('Not authenticated. Run `galactus auth login` first.');
  }

  const { accessKey, privateKey: privateKeyPem, environment } = creds;
  const baseUrl = BASE_URLS[environment];
  const privateKey = parsePrivateKey(privateKeyPem);

  async function request<T>(
    method: string,
    path: string,
    body?: unknown
  ): Promise<T> {
    const url = `${baseUrl}${path}`;
    // Sign the full path including /trade-api/v2
    const fullPath = `/trade-api/v2${path}`;
    const headers = buildAuthHeaders(accessKey, privateKey, method, fullPath);

    const response = await fetch(url, {
      method,
      headers,
      body: body ? JSON.stringify(body) : undefined,
    });

    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`Kalshi API error (${response.status}): ${errorText}`);
    }

    return response.json() as Promise<T>;
  }

  return {
    /**
     * Get all series, optionally filtered by category
     */
    async getSeries(params?: { category?: string }): Promise<KalshiSeries[]> {
      const queryParams = new URLSearchParams();
      if (params?.category) {
        queryParams.set('category', params.category);
      }
      const path = `/series${queryParams.toString() ? `?${queryParams}` : ''}`;
      const data = await request<{ series: KalshiSeries[] }>('GET', path);
      return data.series || [];
    },

    /**
     * Get markets with various filters
     */
    async getMarkets(params?: {
      series_ticker?: string;
      event_ticker?: string;
      status?: string;
      limit?: number;
      cursor?: string;
    }): Promise<KalshiMarket[]> {
      const queryParams = new URLSearchParams();
      if (params?.series_ticker) queryParams.set('series_ticker', params.series_ticker);
      if (params?.event_ticker) queryParams.set('event_ticker', params.event_ticker);
      if (params?.status) queryParams.set('status', params.status);
      if (params?.limit) queryParams.set('limit', params.limit.toString());
      if (params?.cursor) queryParams.set('cursor', params.cursor);

      const path = `/markets${queryParams.toString() ? `?${queryParams}` : ''}`;
      const data = await request<{ markets: KalshiMarket[]; cursor?: string }>('GET', path);
      return data.markets || [];
    },

    /**
     * Get events with various filters
     */
    async getEvents(params?: {
      series_ticker?: string;
      status?: string;
      limit?: number;
    }): Promise<KalshiEvent[]> {
      const queryParams = new URLSearchParams();
      if (params?.series_ticker) queryParams.set('series_ticker', params.series_ticker);
      if (params?.status) queryParams.set('status', params.status);
      if (params?.limit) queryParams.set('limit', params.limit.toString());

      const path = `/events${queryParams.toString() ? `?${queryParams}` : ''}`;
      const data = await request<{ events: KalshiEvent[]; cursor?: string }>('GET', path);
      return data.events || [];
    },

    /**
     * Get a single market by ticker
     */
    async getMarket(ticker: string): Promise<KalshiMarket> {
      const data = await request<{ market: KalshiMarket }>('GET', `/markets/${ticker}`);
      return data.market;
    },

    /**
     * Get account balance
     */
    async getBalance(): Promise<{ balance: number }> {
      return request<{ balance: number }>('GET', '/portfolio/balance');
    },

    /**
     * Raw request for custom queries
     */
    request,
  };
}

export type KalshiClient = ReturnType<typeof createKalshiClient>;
