/**
 * CLI Output Formatting
 *
 * Supports both human-readable (pretty) and machine-readable (JSON) output
 */

let outputFormat: 'pretty' | 'json' = 'pretty';

export function setOutputFormat(format: 'pretty' | 'json'): void {
  outputFormat = format;
}

export function getOutputFormat(): 'pretty' | 'json' {
  return outputFormat;
}

/**
 * Output data in the appropriate format
 */
export function output(data: unknown): void {
  if (outputFormat === 'json') {
    console.log(JSON.stringify(data, null, 2));
  } else {
    if (Array.isArray(data)) {
      outputTable(data);
    } else if (typeof data === 'object' && data !== null) {
      outputObject(data as Record<string, unknown>);
    } else {
      console.log(data);
    }
  }
}

/**
 * Output an object as key-value pairs
 */
export function outputObject(obj: Record<string, unknown>): void {
  const maxKeyLength = Math.max(...Object.keys(obj).map((k) => k.length));

  for (const [key, value] of Object.entries(obj)) {
    const paddedKey = key.padEnd(maxKeyLength);
    const displayValue =
      typeof value === 'object' ? JSON.stringify(value) : String(value ?? '');
    console.log(`${paddedKey}  ${displayValue}`);
  }
}

/**
 * Get terminal width
 */
function getTerminalWidth(): number {
  return process.stdout.columns || 120;
}

/**
 * Truncate string to max length
 */
function truncate(str: string, maxLen: number): string {
  if (str.length <= maxLen) return str;
  return str.slice(0, maxLen - 3) + '...';
}

/**
 * Output an array as a table
 */
export function outputTable(
  rows: Record<string, unknown>[],
  options?: {
    columns?: string[];
    maxWidth?: number;
  }
): void {
  if (rows.length === 0) {
    console.log('(no results)');
    return;
  }

  const termWidth = options?.maxWidth || getTerminalWidth();

  // Determine columns
  const columns = options?.columns || Object.keys(rows[0]);

  // Calculate column widths
  const colWidths: Record<string, number> = {};
  for (const col of columns) {
    const headerLen = col.length;
    const maxValueLen = Math.max(
      ...rows.map((row) => String(row[col] ?? '').length)
    );
    colWidths[col] = Math.max(headerLen, Math.min(maxValueLen, 50));
  }

  // Adjust widths to fit terminal
  const totalWidth =
    Object.values(colWidths).reduce((a, b) => a + b, 0) +
    (columns.length - 1) * 2;
  if (totalWidth > termWidth) {
    // Find the widest non-ID column and shrink it
    const shrinkableCol = columns.find(
      (c) => !c.toLowerCase().includes('ticker') && colWidths[c] > 20
    );
    if (shrinkableCol) {
      colWidths[shrinkableCol] = Math.max(
        20,
        colWidths[shrinkableCol] - (totalWidth - termWidth)
      );
    }
  }

  // Print header
  const header = columns.map((col) => col.toUpperCase().padEnd(colWidths[col])).join('  ');
  console.log(header);
  console.log(columns.map((col) => '─'.repeat(colWidths[col])).join('  '));

  // Print rows
  for (const row of rows) {
    const line = columns
      .map((col) => {
        const value = String(row[col] ?? '');
        return truncate(value, colWidths[col]).padEnd(colWidths[col]);
      })
      .join('  ');
    console.log(line);
  }

  console.log(`\n(${rows.length} results)`);
}

/**
 * Output success message
 */
export function success(message: string): void {
  if (outputFormat === 'json') {
    console.log(JSON.stringify({ success: true, message }));
  } else {
    console.log(`✓ ${message}`);
  }
}

/**
 * Output error message
 */
export function error(message: string): void {
  if (outputFormat === 'json') {
    console.error(JSON.stringify({ success: false, error: message }));
  } else {
    console.error(`✗ ${message}`);
  }
}
