#!/usr/bin/env bash
#
# deploy-staging.sh — Build and deploy Galactus One to dev.g1.newbx.com
#
# Usage:
#   ./scripts/deploy-staging.sh
#
# Reads config from scripts/.staging-env (see .staging-env.example).
# Builds shared → dashboard → relay, then deploys via tar+ssh to the
# staging server and restarts the relay via PM2.
#
# Requires SSH config for g1.newbx.com (see ~/.ssh/config).

set -euo pipefail

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
REPO_ROOT="$(cd "$SCRIPT_DIR/.." && pwd)"
ENV_FILE="$SCRIPT_DIR/.staging-env"

# ── Load config ─────────────────────────────────────────────────────────────

if [[ ! -f "$ENV_FILE" ]]; then
  echo "ERROR: $ENV_FILE not found."
  echo "Run: cp scripts/.staging-env.example scripts/.staging-env"
  echo "Then fill in the values."
  exit 1
fi

set -a
# shellcheck source=/dev/null
source "$ENV_FILE"
set +a

# Validate required vars
for var in STAGING_SSH_HOST STAGING_DOMAIN STAGING_RELAY_PORT STAGING_DASHBOARD_PASSWORD_HASH; do
  if [[ -z "${!var:-}" ]]; then
    echo "ERROR: $var is not set in $ENV_FILE"
    exit 1
  fi
done

SSH_HOST="$STAGING_SSH_HOST"
REMOTE_DASHBOARD_DIR="public_html/dev.g1.newbx.com"
REMOTE_RELAY_DIR="galactus-relay-dev"

echo "╔══════════════════════════════════════════════════════╗"
echo "║  Galactus One — Staging Deploy                      ║"
echo "╠══════════════════════════════════════════════════════╣"
echo "║  Host:    $SSH_HOST"
echo "║  Domain:  $STAGING_DOMAIN"
echo "║  Relay:   port $STAGING_RELAY_PORT"
echo "║  Branch:  $(git -C "$REPO_ROOT" branch --show-current)"
echo "║  Commit:  $(git -C "$REPO_ROOT" log -1 --format='%h %s')"
echo "╚══════════════════════════════════════════════════════╝"
echo ""
read -rp "Deploy to staging? [y/N] " confirm
if [[ "$confirm" != [yY] ]]; then
  echo "Aborted."
  exit 0
fi

# ── Build ────────────────────────────────────────────────────────────────────

cd "$REPO_ROOT"

echo ""
echo "▸ Building shared types..."
npm run build --workspace=@galactus/shared

echo ""
echo "▸ Building dashboard (with staging env)..."
VITE_DASHBOARD_PASSWORD_HASH="$STAGING_DASHBOARD_PASSWORD_HASH" \
VITE_RELAY_HTTP_URL=/relay/http \
VITE_RELAY_WS_URL="wss://$STAGING_DOMAIN/relay/ws" \
  npm run dashboard:build

echo ""
echo "▸ Building relay..."
npm run relay:build

# ── Deploy dashboard ────────────────────────────────────────────────────────

echo ""
echo "▸ Deploying dashboard..."

ssh "$SSH_HOST" "mkdir -p ~/$REMOTE_DASHBOARD_DIR"
ssh "$SSH_HOST" "find ~/$REMOTE_DASHBOARD_DIR -mindepth 1 -not -name '.htaccess' -delete 2>/dev/null || true"

tar -czf - -C "$REPO_ROOT/apps/dashboard/dist" . \
  | ssh "$SSH_HOST" "tar -xzf - -C ~/$REMOTE_DASHBOARD_DIR/" 2>/dev/null

echo "▸ Writing version.json..."
DEPLOY_VERSION=$(git -C "$REPO_ROOT" rev-parse --short HEAD)
DEPLOY_TIME=$(date -u +"%Y-%m-%dT%H:%M:%SZ")
ssh "$SSH_HOST" "cat > ~/$REMOTE_DASHBOARD_DIR/version.json" << VERJSON
{"version":"$DEPLOY_VERSION","deployedAt":"$DEPLOY_TIME"}
VERJSON

echo "▸ Writing .htaccess..."
ssh "$SSH_HOST" "cat > ~/$REMOTE_DASHBOARD_DIR/.htaccess" << HTACCESS
Options -Indexes
DirectoryIndex index.html

<IfModule mod_headers.c>
  <FilesMatch "\.(js|css|woff2?|ttf|eot|svg|png|jpg|gif|ico|webp)$">
    Header set Cache-Control "public, max-age=31536000, immutable"
  </FilesMatch>
  <FilesMatch "\.(html|json)$">
    Header set Cache-Control "no-cache, no-store, must-revalidate"
    Header set Pragma "no-cache"
    Header set Expires "0"
  </FilesMatch>
</IfModule>

<IfModule mod_rewrite.c>
  RewriteEngine On
  RewriteBase /

  # Proxy /relay/ to local relay server (must come before SPA fallback)
  RewriteRule ^relay/(.*)\$ http://127.0.0.1:$STAGING_RELAY_PORT/relay/\$1 [P,L]

  # SPA fallback — serve index.html for non-file, non-dir requests
  RewriteCond %{REQUEST_FILENAME} !-f
  RewriteCond %{REQUEST_FILENAME} !-d
  RewriteRule . /index.html [L]
</IfModule>
HTACCESS

# ── Deploy relay ────────────────────────────────────────────────────────────

echo ""
echo "▸ Deploying relay..."

ssh "$SSH_HOST" "mkdir -p ~/$REMOTE_RELAY_DIR/packages/shared"

# Upload relay dist + config
tar -czf - -C "$REPO_ROOT/apps/relay" \
  package.json ecosystem.config.cjs dist/ \
  | ssh "$SSH_HOST" "tar -xzf - -C ~/$REMOTE_RELAY_DIR/" 2>/dev/null

# Upload shared package (dist + package.json)
tar -czf - -C "$REPO_ROOT/packages/shared" \
  package.json dist/ \
  | ssh "$SSH_HOST" "tar -xzf - -C ~/$REMOTE_RELAY_DIR/packages/shared/" 2>/dev/null

# Rewrite @galactus/shared from workspace wildcard to local file path
ssh "$SSH_HOST" "cd ~/$REMOTE_RELAY_DIR && sed -i 's|\"@galactus/shared\": \"\\*\"|\"@galactus/shared\": \"file:./packages/shared\"|' package.json"

# Inject relay .env
echo "▸ Injecting relay .env..."
ssh "$SSH_HOST" "cat > ~/$REMOTE_RELAY_DIR/.env" << ENV
PORT=$STAGING_RELAY_PORT
NODE_ENV=production
CORS_ORIGIN=https://$STAGING_DOMAIN
ENV

# ── Install deps & restart PM2 ──────────────────────────────────────────────

echo ""
echo "▸ Installing relay dependencies..."
ssh "$SSH_HOST" "cd ~/$REMOTE_RELAY_DIR && npm install --omit=dev 2>&1 | tail -3"

echo ""
echo "▸ Restarting relay via PM2..."
ssh "$SSH_HOST" "export PATH=\$HOME/.npm-global/bin:\$PATH && cd ~/$REMOTE_RELAY_DIR && pm2 delete galactus-relay 2>/dev/null || true && pm2 start ecosystem.config.cjs && pm2 save"

# ── Health checks ────────────────────────────────────────────────────────────

echo ""
echo "▸ Running health checks..."
sleep 2

RELAY_HEALTH=$(ssh "$SSH_HOST" "curl -sf http://localhost:$STAGING_RELAY_PORT/health 2>/dev/null || echo 'FAIL'")
if [[ "$RELAY_HEALTH" == "FAIL" ]]; then
  echo "  ✗ Relay health check FAILED (localhost:$STAGING_RELAY_PORT)"
  echo "    ssh $SSH_HOST 'export PATH=\$HOME/.npm-global/bin:\$PATH && pm2 logs galactus-relay --lines 20'"
else
  echo "  ✓ Relay healthy (localhost:$STAGING_RELAY_PORT)"
fi

DASHBOARD_STATUS=$(curl -sf -o /dev/null -w "%{http_code}" "https://$STAGING_DOMAIN/" 2>/dev/null || echo "000")
if [[ "$DASHBOARD_STATUS" == "200" ]]; then
  echo "  ✓ Dashboard live (https://$STAGING_DOMAIN/)"
else
  echo "  ✗ Dashboard HTTP $DASHBOARD_STATUS (https://$STAGING_DOMAIN/)"
fi

echo ""
echo "══════════════════════════════════════════════════════"
echo "  Deploy complete."
echo "  Dashboard: https://$STAGING_DOMAIN/"
echo "  Relay:     port $STAGING_RELAY_PORT (PM2: galactus-relay)"
echo "══════════════════════════════════════════════════════"
