#!/usr/bin/env bash
#
# post-receive-dev.sh — Auto-deploy dev branch to dev.g1.newbx.com
#
# Called by the post-receive router. Receives: $1=oldrev $2=newrev $3=branch
# Config: ~/g1-deploy.env (existing config, unchanged)
# Log: ~/g1-deploy.log
#
# This is the "promoted" environment — code lands here after StreamRift
# reviews and polishes from streamrift.dev.g1 or domino.dev.g1.
#
# Auto-increments patch version tag on successful deploy.
#
# Installs to: ~/deploy-hooks/post-receive-dev.sh

set -euo pipefail

COMMIT_SHORT="${2:0:7}"
DEPLOY_BRANCH="${3:-dev}"

ENV_FILE="$HOME/g1-deploy.env"
if [[ ! -f "$ENV_FILE" ]]; then
  echo "  ERROR: $ENV_FILE not found. Skipping deploy."
  exit 0
fi

echo "  Launching auto-deploy for $DEPLOY_BRANCH @ $COMMIT_SHORT"
echo "  Logs: ~/g1-deploy.log"

nohup bash -c "
set -euo pipefail
set -a; source \"$ENV_FILE\"; set +a
export PATH=\"\$HOME/.npm-global/bin:\$PATH\"
unset GIT_DIR GIT_WORK_TREE

LOG=\"\$HOME/g1-deploy.log\"
exec > \"\$LOG\" 2>&1

echo '══════════════════════════════════════════════════════'
echo '  Dev Deploy: $DEPLOY_BRANCH @ $COMMIT_SHORT'
echo \"  \$(date -u +'%Y-%m-%d %H:%M:%S UTC')\"
echo '══════════════════════════════════════════════════════'

BUILD_START=\$SECONDS

# ── Update working copy ──────────────────────────────────────────────────
if [[ ! -d \"\$BUILD_DIR\" ]]; then
  echo '▸ Cloning...'
  git clone \"\$GIT_REPO\" \"\$BUILD_DIR\" --branch \"$DEPLOY_BRANCH\"
else
  echo '▸ Updating...'
  cd \"\$BUILD_DIR\"
  git fetch origin
  git checkout \"$DEPLOY_BRANCH\"
  git reset --hard \"origin/$DEPLOY_BRANCH\"
fi
cd \"\$BUILD_DIR\"

# ── Build ────────────────────────────────────────────────────────────────
echo '▸ Installing dependencies...'
npm install

echo '▸ Building shared...'
npm run build --workspace=@galactus/shared

echo '▸ Building dashboard...'
VITE_DASHBOARD_PASSWORD_HASH=\"\$STAGING_DASHBOARD_PASSWORD_HASH\" \\
VITE_RELAY_HTTP_URL=/relay/http \\
VITE_RELAY_WS_URL=\"wss://\$STAGING_DOMAIN/relay/ws\" \\
VITE_ODDS_API_KEY=\"\$STAGING_ODDS_API_KEY\" \\
  npm run dashboard:build

echo '▸ Building relay...'
npm run relay:build

# ── Deploy dashboard ─────────────────────────────────────────────────────
echo '▸ Deploying dashboard...'
mkdir -p \"\$DASHBOARD_DIR\"
find \"\$DASHBOARD_DIR\" -mindepth 1 -not -name '.htaccess' -not -name 'status.html' -not -name 'status.json' -delete 2>/dev/null || true
cp -r \"\$BUILD_DIR/apps/dashboard/dist/.\" \"\$DASHBOARD_DIR/\"

DEPLOY_VERSION=\$(git rev-parse --short HEAD)
DEPLOY_TIME=\$(date -u +'%Y-%m-%dT%H:%M:%SZ')
DEPLOY_MSG=\$(git log -1 --format='%s')
FILES_CHANGED=\$(git diff --name-only HEAD~1 HEAD 2>/dev/null | wc -l | tr -d ' ')

printf '{\"version\":\"%s\",\"deployedAt\":\"%s\",\"branch\":\"%s\"}' \\
  \"\$DEPLOY_VERSION\" \"\$DEPLOY_TIME\" \"$DEPLOY_BRANCH\" > \"\$DASHBOARD_DIR/version.json\"

cat > \"\$DASHBOARD_DIR/.htaccess\" << 'HTEOF'
Options -Indexes
DirectoryIndex index.html

<IfModule mod_headers.c>
  <FilesMatch \"\.(js|css|woff2?|ttf|eot|svg|png|jpg|gif|ico|webp)\$\">
    Header set Cache-Control \"public, max-age=31536000, immutable\"
  </FilesMatch>
  <FilesMatch \"\.(html|json)\$\">
    Header set Cache-Control \"no-cache, no-store, must-revalidate\"
    Header set Pragma \"no-cache\"
    Header set Expires \"0\"
  </FilesMatch>
</IfModule>

<IfModule mod_rewrite.c>
  RewriteEngine On
  RewriteBase /
  RewriteRule ^relay/(.*)\$ http://127.0.0.1:\$STAGING_RELAY_PORT/relay/\$1 [P,L]
  RewriteCond %{REQUEST_FILENAME} !-f
  RewriteCond %{REQUEST_FILENAME} !-d
  RewriteRule . /index.html [L]
</IfModule>
HTEOF

# ── Deploy relay ─────────────────────────────────────────────────────────
echo '▸ Deploying relay...'
mkdir -p \"\$RELAY_DIR/packages/shared\"
cp \"\$BUILD_DIR/apps/relay/package.json\" \"\$RELAY_DIR/\"
cp \"\$BUILD_DIR/apps/relay/ecosystem.config.cjs\" \"\$RELAY_DIR/\"
cp -r \"\$BUILD_DIR/apps/relay/dist/.\" \"\$RELAY_DIR/dist/\"
cp \"\$BUILD_DIR/packages/shared/package.json\" \"\$RELAY_DIR/packages/shared/\"
cp -r \"\$BUILD_DIR/packages/shared/dist/.\" \"\$RELAY_DIR/packages/shared/dist/\"

cd \"\$RELAY_DIR\"
sed -i 's|\"@galactus/shared\": \"\\*\"|\"@galactus/shared\": \"file:./packages/shared\"|' package.json

cat > \"\$RELAY_DIR/.env\" << ENVEOF
PORT=\$STAGING_RELAY_PORT
NODE_ENV=production
CORS_ORIGIN=https://\$STAGING_DOMAIN
ENVEOF

echo '▸ Installing relay deps...'
npm install --omit=dev

echo '▸ Restarting relay...'
pm2 delete galactus-relay 2>/dev/null || true
pm2 start ecosystem.config.cjs
pm2 save

# ── Health check ─────────────────────────────────────────────────────────
sleep 2
RELAY_STATUS='healthy'
RELAY_HEALTH=\$(curl -sf \"http://localhost:\$STAGING_RELAY_PORT/health\" 2>/dev/null || echo 'FAIL')
if [[ \"\$RELAY_HEALTH\" == 'FAIL' ]]; then
  echo '  ✗ Relay health check FAILED'
  RELAY_STATUS='failed'
else
  echo '  ✓ Relay healthy'
fi

BUILD_DURATION=\$((\$SECONDS - \$BUILD_START))

# ── Auto-increment version tag ───────────────────────────────────────────
echo '▸ Auto-tagging version...'
cd \"\$BUILD_DIR\"
LATEST_TAG=\$(git tag -l 'v*' --sort=-v:refname | head -1)
if [[ -n \"\$LATEST_TAG\" ]]; then
  # Parse semver: vMAJOR.MINOR.PATCH
  IFS='.' read -r MAJOR MINOR PATCH <<< \"\${LATEST_TAG#v}\"
  NEW_PATCH=\$((\$PATCH + 1))
  NEW_TAG=\"v\${MAJOR}.\${MINOR}.\${NEW_PATCH}\"
else
  NEW_TAG='v0.5.1'
fi
git tag -a \"\$NEW_TAG\" -m \"Auto-deploy to dev.g1: \$DEPLOY_MSG\" 2>/dev/null || true
git push origin \"\$NEW_TAG\" 2>/dev/null || true
echo \"  Tagged: \$NEW_TAG\"

# ── Generate status files ────────────────────────────────────────────────
cat > \"\$DASHBOARD_DIR/status.json\" << STATUSJSON
{
  \"branch\": \"$DEPLOY_BRANCH\",
  \"commit\": \"\$DEPLOY_VERSION\",
  \"commitMessage\": \"\$(echo \$DEPLOY_MSG | sed 's/\"/\\\\\"/g')\",
  \"deployedAt\": \"\$DEPLOY_TIME\",
  \"buildDurationSec\": \$BUILD_DURATION,
  \"status\": \"success\",
  \"filesChanged\": \$FILES_CHANGED,
  \"relayHealthy\": \$([ \"\$RELAY_STATUS\" = 'healthy' ] && echo true || echo false),
  \"developer\": \"streamrift\",
  \"versionTag\": \"\$NEW_TAG\"
}
STATUSJSON

echo ''
echo '══════════════════════════════════════════════════════'
echo \"  Deploy complete: $DEPLOY_BRANCH @ \$DEPLOY_VERSION (\$NEW_TAG)\"
echo \"  https://\$STAGING_DOMAIN/\"
echo '══════════════════════════════════════════════════════'
" </dev/null >> "$HOME/g1-deploy.log" 2>&1 &

echo "  Build started in background (PID $!)."
