#!/bin/bash
# Galactus One — Domino Bootstrap Setup
# Run this once after unzipping. Sets up SSH keys + git remote.

set -e

SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
SSH_DIR="$HOME/.ssh"
KEY_NAME="g1_domino"
BOOTSTRAP_DIR="$SCRIPT_DIR/.ssh-bootstrap"

echo ""
echo "  Galactus One — Setup"
echo "  ─────────────────────"
echo ""

# 1. Install SSH key
if [ ! -f "$BOOTSTRAP_DIR/g1_domino" ]; then
  echo "  ERROR: Bootstrap key not found at $BOOTSTRAP_DIR/g1_domino"
  echo "  Re-download the zip from domino.dev.g1.newbx.com"
  exit 1
fi

mkdir -p "$SSH_DIR"
chmod 700 "$SSH_DIR"

if [ -f "$SSH_DIR/$KEY_NAME" ]; then
  echo "  SSH key already exists at $SSH_DIR/$KEY_NAME"
  read -p "  Overwrite? [y/N] " -n 1 -r
  echo
  if [[ ! $REPLY =~ ^[Yy]$ ]]; then
    echo "  Keeping existing key."
  else
    cp "$BOOTSTRAP_DIR/g1_domino" "$SSH_DIR/$KEY_NAME"
    cp "$BOOTSTRAP_DIR/g1_domino.pub" "$SSH_DIR/$KEY_NAME.pub"
    chmod 600 "$SSH_DIR/$KEY_NAME"
    echo "  Key installed to $SSH_DIR/$KEY_NAME"
  fi
else
  cp "$BOOTSTRAP_DIR/g1_domino" "$SSH_DIR/$KEY_NAME"
  cp "$BOOTSTRAP_DIR/g1_domino.pub" "$SSH_DIR/$KEY_NAME.pub"
  chmod 600 "$SSH_DIR/$KEY_NAME"
  echo "  Key installed to $SSH_DIR/$KEY_NAME"
fi

# 2. Add SSH config
SSH_CONFIG="$SSH_DIR/config"
if grep -q "Host g1" "$SSH_CONFIG" 2>/dev/null; then
  echo "  SSH config entry 'g1' already exists — skipping"
else
  cat >> "$SSH_CONFIG" << 'SSHEOF'

# Galactus One — git remote
Host g1
  HostName g1.newbx.com
  User g1bx
  IdentityFile ~/.ssh/g1_domino
  IdentitiesOnly yes
SSHEOF
  chmod 600 "$SSH_CONFIG"
  echo "  SSH config added (Host g1 → g1.newbx.com)"
fi

# 3. Test SSH connection
echo ""
echo "  Testing SSH connection..."
if ssh -o StrictHostKeyChecking=accept-new -o ConnectTimeout=10 g1 "echo 'connected'" 2>/dev/null; then
  echo "  SSH connection works!"
else
  echo "  WARNING: SSH connection failed. This might be a network issue."
  echo "  You can test manually: ssh g1 echo hello"
fi

# 4. Initialize git + set remote
echo ""
if [ -d "$SCRIPT_DIR/.git" ]; then
  echo "  Git repo already initialized"
else
  cd "$SCRIPT_DIR"
  git init -q
  echo "  Git initialized"
fi

cd "$SCRIPT_DIR"
if git remote get-url origin >/dev/null 2>&1; then
  echo "  Git remote 'origin' already set"
else
  git remote add origin g1:git/g1.1.git
  echo "  Git remote set: origin → g1:git/g1.1.git"
fi

# 5. Fetch and checkout domino-0
echo ""
echo "  Fetching from remote..."
if git fetch origin 2>/dev/null; then
  git checkout -B domino-0 origin/domino-0 2>/dev/null || git checkout -b domino-0
  echo "  On branch: domino-0"
else
  echo "  WARNING: Could not fetch. You can do this later: git fetch origin && git checkout domino-0"
fi

# 6. Install dependencies
echo ""
echo "  Installing npm dependencies..."
npm install --silent 2>/dev/null && echo "  Dependencies installed!" || echo "  WARNING: npm install had issues. Try running it manually."

echo ""
echo "  ─────────────────────"
echo "  Setup complete!"
echo ""
echo "  Next steps:"
echo "    npm run dashboard:dev     Start the dev server"
echo "    open localhost:5173        Password: streamrift"
echo "    claude                     Start your AI assistant"
echo "    /get-started               First-time setup interview"
echo ""
